/*! 
 * @file navigation_data.h
 * @brief Data definitions for (De)serialization used mainly by 
 *        GlobalNavigation Component.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 * 主に大経路コンポーネントが使用する，シリアライズ／デシリアライズ可能
 * なデータ型の定義。
 * 
 * <ul>
 *  <li>
 *    RGISのidlから生成されたデータのうち，基本データのみのクラスは
 *    そのまま使用する(non-intrusiveなシリアライズ関数を定義)
 *  </li>
 *  <li>
 *    そうでない場合はenum -> int, CORBA::String --> std::stringなどと変換
 *    したクラスを定義
 *  </li>
 * </ul>
 *
 * @todo readFromXML, writeToXmlは基底クラスに用意してtemplate method化
 *       したほうが良かったかもしれない。特にstring&を引数とする
 *       overload を各クラスで定義するのは冗長。継承を行った場合の
 *       boostのシリアライザの挙動についてテスト不足のためこのようになっ
 *       ている。
 *    
 */



#if !defined(NAVIGATION_DATA_HEADER_)
#define      NAVIGATION_DATA_HEADER_

#include <sstream>
#include "xml_serdes.h"
#include "quantized_vector.h"
#include "RgisOperator_simpleStub.h"

namespace RGIS {
struct PntC;
struct BoxC;
struct RotationC;

namespace Geo {
struct GridInfoC;

} /* namespace Geo */
} /* namespace RGIS */




namespace boost {
namespace serialization {
//
//  文字列／列挙型など，CORBA独自の内部機構に依存しない値のみを
//  を持つ構造体は直接使用する。下記は非侵入型シリアライズ関数
//

/*! @brief @a RGIS::PntC用シリアライズ関数.
 *  @tparam Archive Boostのアーカイブクラス名
 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
 *  @param[in] p シリアライズ／デシリアライズされる@a RGIS::PntC
 *  @param[in] (未使用) バージョン
 */
template <class Archive> 
void
serialize(Archive& ar, RGIS::PntC& p, const unsigned int /*version*/)
{
	ar & make_nvp("x", p.x);
	ar & make_nvp("y", p.y);
	ar & make_nvp("z", p.z);
}
/*! @brief @a RGIS::BoxC用シリアライズ関数.
 *  @tparam Archive Boostのアーカイブクラス名
 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
 *  @param[in] b シリアライズ／デシリアライズされる@a RGIS::BoxC
 *  @param[in] (未使用) バージョン
 */
template <class Archive> 
void
serialize(Archive& ar, RGIS::BoxC& b, const unsigned int /*version*/)
{
	ar & make_nvp("minX", b.minX);
	ar & make_nvp("minY", b.minY);
	ar & make_nvp("minZ", b.minZ);
	ar & make_nvp("maxX", b.maxX);
	ar & make_nvp("maxY", b.maxY);
	ar & make_nvp("maxZ", b.maxZ);
}
/*! @brief @a RGIS::RotationC用シリアライズ関数.
 *  @tparam Archive Boostのアーカイブクラス名
 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
 *  @param[in] r シリアライズ／デシリアライズされる@a RGIS::RotationC
 *  @param[in] (未使用) バージョン
 */
template <class Archive> 
void
serialize(Archive& ar, RGIS::RotationC& r, const unsigned int /*version*/)
{
	ar & make_nvp("roll", r.roll);
	ar & make_nvp("pitch", r.pitch);
	ar & make_nvp("yaw", r.yaw);
}

} /* namespace serialization */
} /* namespace boost */


namespace NavData {

extern const char CATEGORY[];	//!< カテゴリ文字列 
extern const char COORD_JGD9[]; //!< 平面直角座標系9を表す文字列

//  RGIS::CrsInfo
//  RGIS::Geo::GridInfoはCORBAの列挙型(class)を含むため，
//  類似の構造体を用いる

/*! @struct CrsInfo
 *  @brief @a RGIS::CrsInfoに対応する構造体。
 */
struct CrsInfo {
	std::string name;			//!< 座標系の名称 
	std::string params;			//!< 座標系のパラメータ（proj4形式）例：+init=4326

	//! コンストラクタ。座標系は9系として初期化。paramsは空
	CrsInfo() : name(COORD_JGD9), params() { }

	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */)
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & make_nvp("name", name);
		ar & make_nvp("params", params);
	}
};

/*! @struct GridInfo
 *  @brief RGIS::Geo::GridInfoCに対応する構造体
 */
struct GridInfo {
	int numCellsW;				//!< 横方向のセル数
	int numCellsH;				//!< 縦方向のセル数
	int numBands;				//!< バンド数
	int valType;				//!< セルの値の型
	double noDataVal;			//!< NoData値
	bool useNoData;				//!< NoData値を使用するかどうか
	double valScale;			//!< 値のスケール
	double cellWidth;			//!< セル1つの横幅 
	double cellHeight;			//!< セル1つの高さ
	RGIS::PntC position;		//!< グリッドの左上の位置座標
	RGIS::RotationC rotation;	//!< グリッドの回転
	CrsInfo crsInfo;			//!< 空間参照

	//! デフォルトコンストラクタ
	GridInfo() :
		numCellsW(), numCellsH(), numBands(),
		valType(), noDataVal(), useNoData(),
		valScale(), cellWidth(), cellHeight(),
		position(), rotation(), crsInfo() { }

	//! RGIS::Geo::GridInfoCからの型変換コンストラクタ(explicit)
	explicit GridInfo(const RGIS::Geo::GridInfoC& g) :
		numCellsW(g.numCellsW), numCellsH(g.numCellsH), numBands(g.numBands),
		valType(g.valType), noDataVal(g.noDataVal), useNoData(g.useNoData),
		valScale(g.valScale), cellWidth(g.cellWidth), cellHeight(g.cellHeight),
		position(g.position), rotation(g.rotation), crsInfo() { }

	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */)
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & BOOST_SERIALIZATION_NVP(numCellsW);
		ar & BOOST_SERIALIZATION_NVP(numCellsH);
		ar & BOOST_SERIALIZATION_NVP(numBands);
		ar & BOOST_SERIALIZATION_NVP(valType);
		ar & BOOST_SERIALIZATION_NVP(noDataVal);
		ar & BOOST_SERIALIZATION_NVP(useNoData);
		ar & BOOST_SERIALIZATION_NVP(valScale);
		ar & BOOST_SERIALIZATION_NVP(cellWidth);
		ar & BOOST_SERIALIZATION_NVP(cellHeight);
		ar & BOOST_SERIALIZATION_NVP(position);
		ar & BOOST_SERIALIZATION_NVP(rotation);
		ar & BOOST_SERIALIZATION_NVP(crsInfo);
	}
};

/* ---------------------------------------------------------------- *
 *  送受信の単位となるデータ。
 * ---------------------------------------------------------------- */
/*! @struct PotentialGrid
 *  @brief ポテンシャルグリッド送受信のための構造体
 * 
 */
struct PotentialGrid {
	GridInfo info;				//!< グリッド設定情報 
	QuantizedVector data;		//!< セルデータが格納される @a QuantizedVector

	//! デフォルトコンストラクタ。セルは空になる
	PotentialGrid() : info(), data() { }

	/* ---- decls for serialization ---- */

	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */) 
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & make_nvp("gridInfo", info);
		ar & make_nvp("gridData", data);
	}
	static const char TYPE_ID[]; //!< このクラスを表す型ID文字列
	
	/*! 自身をXMLとして書き出す.
	 *  @param[out] os 書き出し先ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}
	/*! 自身をXMLとして書き出す.
	 *  \overload int writeToXml(std::string& s) const
	 *  @param[out] s 書き出し先文字列
	 */
	int
	writeToXml(std::string& s) const
	{
		std::ostringstream os(s);
		return writeToXml(os);
	}

	/*! 自身をXMLから読み出す(値を設定).
	 *  @param[out] is 読み出し元ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}
	/*! 自身をXMLから読み出す(値を設定).
	 *  \overload int readFromXml(const std::string& s)
	 *  @param[in] s 読み出し元文字列
	 */
	int
	readFromXml(const std::string& s)
	{
		std::istringstream is(s);
		return readFromXml(is);
	}
};


/*! @struct PathNodes
 *  @brief 経路ノード列を表す構造体
 */
struct PathNodes {
	std::vector<RGIS::PntC> path; //!< way pointのノード 
	std::string coordinate;		  //!< 座標系
	std::string tag;			  //!< この経路列を表すメタ情報文字列
	double time;				  //!< パス生成時刻

	//! デフォルトコンストラクタ. 座標系は9系に初期化
	PathNodes() : path(), coordinate(COORD_JGD9), tag(), time(0.0) { }

	//! RGISのLineStringCからの型変換コンストラクタ(explicit)
	explicit PathNodes(std::vector<RGIS::PntC>& rgis_path)
        : path(rgis_path.begin(), rgis_path.end()),
		  coordinate(COORD_JGD9), tag(), time(0.0) { }

	/* ---- decls for serialization ---- */
	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */) 
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & BOOST_SERIALIZATION_NVP(path);
		ar & BOOST_SERIALIZATION_NVP(coordinate);
		ar & BOOST_SERIALIZATION_NVP(tag);
		ar & BOOST_SERIALIZATION_NVP(time);
	}
	static const char TYPE_ID[]; //!< このクラスを表す型ID文字列
	
	/*! 自身をXMLとして書き出す.
	 *  @param[out] os 書き出し先ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}
	/*! 自身をXMLとして書き出す.
	 *  \overload int writeToXml(std::string& s) const
	 *  @param[out] s 書き出し先文字列
	 */
	int
	writeToXml(std::string& s) const
	{
		std::ostringstream os(s);
		return writeToXml(os);
	}

	/*! 自身をXMLから読み出す(値を設定).
	 *  @param[out] is 読み出し元ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}
	/*! 自身をXMLから読み出す(値を設定).
	 *  \overload int readFromXml(const std::string& s)
	 *  @param[in] s 読み出し元文字列
	 */
	int
	readFromXml(const std::string& s)
	{
		std::istringstream is(s);
		return readFromXml(is);
	}
};

/*! @struct CommandString
 *  @brief コマンド文字列を表す構造体
 */
struct CommandString {
	std::string type;			//!<コマンドの種類

	/* ---- decls for serialization ---- */
	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */) 
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & BOOST_SERIALIZATION_NVP(type);
	}
	static const char TYPE_ID[]; //!< このクラスを表す型ID文字列
	
	/*! 自身をXMLとして書き出す.
	 *  @param[out] os 書き出し先ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}
	/*! 自身をXMLとして書き出す.
	 *  \overload int writeToXml(std::string& s) const
	 *  @param[out] s 書き出し先文字列
	 */
	int
	writeToXml(std::string& s) const
	{
		std::ostringstream os(s);
		return writeToXml(os);
	}

	/*! 自身をXMLから読み出す(値を設定).
	 *  @param[out] is 読み出し元ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}
	/*! 自身をXMLから読み出す(値を設定).
	 *  \overload int readFromXml(const std::string& s)
	 *  @param[in] s 読み出し元文字列
	 */
	int
	readFromXml(const std::string& s)
	{
		std::istringstream is(s);
		return readFromXml(is);
	}
};

/*! @struct TaggedPoint
 *  @brief 情報付き位置を表す構造体。目的地などに使用
 */
struct TaggedPoint {
	TaggedPoint() : coordinate(COORD_JGD9) { }

	std::string coordinate;		//!< 座標系
	RGIS::PntC position;		//!< 位置
	std::string tag;			//!< この点のメタ情報
	double time;				//!< 時刻

	/*! boostから呼ばれるシリアライズメソッド
	 * @tparam Archive boostのArchiveクラス
	 * @param ar boostのArchiveクラスのインスタンス
	 * @param (未使用) version
	 */
	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */) 
	{
		using namespace boost::serialization;
		using namespace boost::archive;
		ar & BOOST_SERIALIZATION_NVP(coordinate);
		ar & BOOST_SERIALIZATION_NVP(position);
		ar & BOOST_SERIALIZATION_NVP(tag);
		ar & BOOST_SERIALIZATION_NVP(time);
	}

	static const char TYPE_ID[]; //!< このクラスを表す型ID文字列
	
	/*! 自身をXMLとして書き出す.
	 *  @param[out] os 書き出し先ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}
	/*! 自身をXMLとして書き出す.
	 *  \overload int writeToXml(std::string& s) const
	 *  @param[out] s 書き出し先文字列
	 */
	int
	writeToXml(std::string& s) const
	{
		std::ostringstream os(s);
		return writeToXml(os);
	}

	/*! 自身をXMLから読み出す(値を設定).
	 *  @param[out] is 読み出し元ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}
	/*! 自身をXMLから読み出す(値を設定).
	 *  \overload int readFromXml(const std::string& s)
	 *  @param[in] s 読み出し元文字列
	 */
	int
	readFromXml(const std::string& s)
	{
		std::istringstream is(s);
		return readFromXml(is);
	}
};


} /* namespace NavData */




#endif    /*!NAVIGATION_DATA_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
