/*!
 * @file baseRTC.h
 * @brief XML SERialization/DESerialization header file
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#if !defined(BASE_RTC_HEADER_)
#define      BASE_RTC_HEADER_

#include <string>
#include <iosfwd>
#include <vector>

namespace baseRTC {

/*! 
 * @struct Header
 * @brief Serialization 共通情報。boostによるシリアライズ情報を隠蔽。
 *
 * XMLへオブジェクトをシリアライズする場合，必ず下記のヘッダー構造体を
 * 最初にシリアライズしなければならない。
 * 
 * 林原先生の最初のXML仕様案を参考に，category, sender_name, 時刻，及びヘッダ
 * 自身のバージョンを格納している。これらの情報の詳細な意味付け，モジュー
 * ル毎の使用規則は議論の必要あり。
 *
 * <h2>データの規則</h2>
 *
 * シリアライズされたデータの先頭には，必ず下記のヘッダ情報が含まれる。
 * 書込み側でこれを保証すること。(writeHeaderを使用)。
 * 
 * ヘッダ情報の後には任意のオブジェクトが含まれて良いが，ヘッダ情報と格
 * 納されるオブジェクト情報については，読み書き側で厳密に合意がとれてい
 * る必要がある。すなわち，読み側と書き側で共通のデータ構造を定義し，ヘッ
 * ダ情報を元にしてデータ型の判断，処理の分岐方法について取り決める必要
 * がある。
 *
 * (2009/12/04追記)
 * 現在考えられている使用方法は以下の通り
 * <dl>
 *   <dt>category: 大分類</dt>
 *   <dd>まだ曖昧だがモジュールベンダネーム？C++ namespace?</dd>
 *   <dt>type</dt> <dd>ヘッダの後に続く型に対応する文字列を格納</dd>
 *   <dt>sender_name</dt> <dd>送信元コンポーネントの名称</dd>
 *   <dt>id </dt>  
 *   <dd> 送信元コンポーネントのID (同一のセンサが複数の場合などを想定)</dd>
 *   <dt>tag</dt>
 *   <dd> その他データの情報についての文字列 </dd>
 * </dl>
 *
 * <ul>
 *   <li>category/typeまでで，デシリアライズすべき型が一意に決定する</li>
 *   <li>データ生成したコンポーネントの種類がnameで決まる</li>
 *   <li>データ生成したコンポーネントをidで決定</li>
 * </ul>
 * 
 *
 * <h2>モジュール同士の通信方法</h2>
 *
 * <h3>データの表現</h3>
 *
 *  通信データを表現するクラスを作成し，シリアライズ／デシリアライズ
 *    を行うメソッドを用意する。この場合，
 *  <ol>
 *    <li> このヘッダに定義するreadXXXFromXml/writeXXXXmlを利用して
 *        データメンバー毎に処理を行う</li>
 *    <li> このクラスが背後で利用するboostのserializationの枠組みを利用 </li>
 *  </ol>
 *    の二つの方法がある。
 *    
 * <h3>送受信の例</h3>
 *
 * <h4>送信の例</h4>
 *  <pre>
 *      ostringstream ost; //何らかのストリームを用意
 *
 *      Header hdr("???", "??", "?????"); //定められた書式
 *      hdr.writeToXml(ost);
 *
 *      実際のデータをここで送信
 *  </pre>
 * <h4>受信の例</h4>
 *  <pre>
 *      istringstream ist; 読み出しストリームを用意。適宜初期化
 *
 *      Header hdr;
 *
 *      if (hdr.readFromXml(ist) &lt; 0) {
 *              エラー処理
 *      }
 *     
 *      if (hdr.category=="X" && hdr.sender_name=="YYY" && hdr.type=="Hoge") {
 *              HogeData hd; 
 *              if (hd.readFromXml(ist) &lt; 0) {
 *                      エラー: 送受信規約に違反
 *              }
 *              hdを使った処理を行う
 *      } else if (hdr.category=="X" && hdr.sender_name=="Y2" && hdr.type=="Fuga") {
 *              FugaData fd; 
 *              if (fd.readFromXml(ist) &lt; 0) {
 *                      エラー: 送受信規約に違反
 *              }
 *              fdを使った処理を行う
 *      }
 * </pre>
 *
 */
struct Header {
	Header();		//!< コンストラクタ。timeのみ自動初期化

	/*! @brief コンストラクタ。
	 * 
	 * category, type, sender_name, id, tagを初期化。timeは生成時刻となる
	 * 
	 */
	Header(const std::string& category, const std::string& type,
	       const std::string& sender_name, 
		   const std::string& id=std::string(),
		   const std::string& tag=std::string());

	std::string category;	//!< データ or モジュールのカテゴリ
	std::string type;		//!< データタイプ
	std::string sender_name;	//!< 送信コンポーネント名
	std::string id;		   //!< モジュールID(同一モジュールが複数の時)
	std::string tag;	   //!< タグ文字列(データに関するその他任意の情報)

	double time;		//!< 送信時刻 [s] from unix epoch
	unsigned int hdr_version; //!< ヘッダのバージョン

	/*!
	 * @brief 入力streamからXml Serialization に関するHeader情報を読み，
	 *        自身を更新する。
	 *
	 * @param[in,out] is シリアライズされた結果のXMLを含むストリーム
	 * @retval 0 以上であれば成功。失敗すると負の値を返す
	 */
	int readFromXml(std::istream& is);

	/*!
	 * @brief 入力文字列からXml Serialization に関するHeader情報を読み，
	 *        自身を更新する。
	 *
	 * @param[in] xml シリアライズされた結果のXML文字列
	 * @retval 0以上 成功。
	 * @retval negative 失敗。失敗の場合中身は空になる
	 */
	int readFromXml(const std::string& xml);

	/*!
	 * @brief 出力streamへXml Serialization に関するHeader情報を書く
	 *
	 * @param[in,out] os 書き込み先stream
	 * @retval 0以上 成功。 
	 * @retval negative 失敗
	 */
	int writeToXml(std::ostream& os) const;

	//! ヘッダのメイジャーバージョン
	static const int HEADER_MAJOR_VERSION = 0;
	//! ヘッダのマイナーバージョン
	static const int HEADER_MINOR_VERSION = 0;
	//! ヘッダのバージョン。
	static const int HEADER_VERSION = HEADER_MAJOR_VERSION*1000 + HEADER_MINOR_VERSION;

	static const char HEADER_TAG[]; //!< ヘッダを格納する際のタグ文字列
};


} /* namespace baseRTC */



#endif    /*!BASE_RTC_HEADER_*/

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */

