/*!
 * @file widget_observer.h
 * @brief Observes the move/resize/close event of target window
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#if !defined(WIDGET_OBSERVER_HEADER_)
#define      WIDGET_OBSERVER_HEADER_

#include <QObject>
#include <QSize>
#include <QPoint>
#include "common_defs.h"
QT_FORWARD_DECLARE_CLASS(QWidget);

/*! @class WidgetObserver
 *  @brief Widgetのイベントを監視し，場所と位置情報を覚えておく
 *
 *  Widgetの場所と位置情報を覚えておくためのクラス。 QtのeventFilter
 *  の枠組みを利用する。ただし，基本的にeventの処理は元のeventハンドラを
 *  呼び出し，自身ではeventの中を除くだけである。
 *
 *  また，eventとsignalの変換も行う。resize, move, closeについてsignalとして
 *  通知することが出来る。
 */
class WidgetObserver : public QObject 
{
	Q_OBJECT;
public:
	WidgetObserver(QObject* parent = 0)
		: QObject(parent), observable_(0) { } //!> コンストラクタ

	//!< デストラクタ。eventFilterが登録されていれば外す。
	virtual ~WidgetObserver() { disconnect(); }

	void disconnect();		  //!< eventFilterが登録されていれば外す。

	void setObservable(QWidget* observable); //!< 被監視widgetを設定

	//! 最後に登録された場所を返す
	QPoint lastPosition() const { return lastPos_; }
	//! 最後に登録された位置を返す
	QSize  lastSize() const     { return lastSize_; }

signals:
	/*! signal: サイズ変更時に発火
	 *  @param target eventを受け取ったwidget
	 *  @param oldsize 変更前のサイズ
	 *  @param newsize 新しいサイズ
	 */
	void resized(QWidget* target, const QSize& oldsize, const QSize& newsize);

	/*! signal: 位置変更時に発火
	 *  @param target eventを受け取ったwidget
	 *  @param oldpos 変更前の位置
	 *  @param newpos 新しい位置
	 */
	void moved(QWidget* target, const QPoint& oldpos, const QPoint& newpos);

	/*! signal: window閉じる際に発火
	 *  @param target eventを受け取ったwidget
	 *  @param accepted slotがこれをtrueにして返すとeventが受理されたことになる
	 */
	void closing(QWidget* target, bool& accepted);

private:

	//! イベントフィルタ。@see QObject::installEventFilter() 
	bool eventFilter(QObject* target, QEvent* event);

	QWidget* observable_;		//!< 被監視widget
	QPoint lastPos_;			//!< widgetの位置 
	QSize lastSize_;			//!< widgetの大きさ
};


#endif    /*!WIDGET_OBSERVER_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
