/*!
 * @file mapview_rtc.cpp
 * @brief Class for RTC functionalities. 
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#include <cstdio>
#include <iostream>

#include "debug_utils.h"
#include "mapview_rtc.h"
#include "config_data.h"
#include "gui_main.h"

using namespace std;
using namespace boost;

namespace /* unnamed */ {
//! 現在時刻をdoubleとして返す。unix通算秒を実数にしたもの
double now()
{
	struct timeval tv;
	struct timezone dummy;
	gettimeofday(&tv, &dummy);
	return double(tv.tv_sec) + tv.tv_usec * 1.0e-6;
}
/*! 現在時刻をval(OpenRTMのTimedXXを想定)に設定
 * @tparam TimedValue tm.sec tm.nsecメンバを持つ型
 * @param val TimedValue型オブジェクトへの参照
 */
template <class TimedValue>
void
setTimeStamp(TimedValue& val)
{
	struct timeval tv;
	struct timezone dummy;
	gettimeofday(&tv, &dummy);

	val.tm.sec  = tv.tv_sec;
	val.tm.nsec = tv.tv_usec * 1000;
}
//! 設定ファイル名の初期値
const char DEFAULT_CONF[] = "./MapViewer.xml";
} /* unnamed namespace */



//! このコンポーネントの情報
static const char* MapViewRTC_spec[] =
{
	"implementation_id", "MapViewRTC",
	"type_name",         "MapViewRTC",
	"description",       "MapViewer",
	"version",           "1.0.0",
	"vendor",            "fuRo at Chiba Institute of Technology",
	"category",          "Navigation",
	"activity_type",     "SPORADIC",
	"kind",              "DataFlowComponent",
	"max_instance",      "1",
	"language",          "C++",
	"lang_type",         "compile",
	// Configuration variables
	"conf.default.filename", DEFAULT_CONF,
	""
};

MapViewRTC::MapViewRTC(RTC::Manager* manager)
	// <rtc-template block="initializer">
	: RTC::DataFlowComponentBase(manager),
	  m_outOut("out", m_out),
	  gui_(new MapViewerGUI)
	  // </rtc-template>
{
	// Registration: InPort/OutPort/Service
	// <rtc-template block="registration">
	// Set InPort buffers
	// Set OutPort buffer
	registerOutPort("out", m_outOut);
        

	// Set service provider to Ports

	// Set service consumers to Ports

	// Set CORBA Service Ports

	// </rtc-template>

}

MapViewRTC::~MapViewRTC()
{
	gui_->wait();
}

/*!
 * @details
 *
 * 初期化時に呼び出されるルーチン。ここでは以下の処理を行う
 *  <ul>
 *    <li> コンフィグパラメータのバインド</li>
 *    <li> コンフィグ情報をファイルから読む</li>
 *    <li> 入力ポート数の設定</li>
 *    <li> @a MapViewerGUIの初期化</li>
 *  </ul>
 *
 */
RTC::ReturnCode_t MapViewRTC::onInitialize()
{
	FUNC_TRACE;
	// <rtc-template block="bind_config">
	// Bind variables and configuration variable
	bindParameter("filename", conf_filename_, DEFAULT_CONF);
	// </rtc-template>

	ConfigData cd;
	if (!cd.readFromFile(conf_filename_)) {
		cerr << "going on anyway ... " << endl;
	}
	cout << cd << "\n";

	typedef boost::shared_ptr<TimedString> StringHolder_t;
	typedef boost::shared_ptr<InPort<TimedString> > PortHolder_t;
	for (int i=0, imax=cd.numberOfInPorts(); i < imax; ++i) {
		char portname[16];
		snprintf(portname, sizeof portname, "in%d", i);

		m_ins.push_back(StringHolder_t(new TimedString()));
		PortHolder_t p(new InPort<TimedString>(portname, *m_ins[i]));
		m_inIns.push_back(p);

		registerInPort(portname, *(m_inIns[i].get()));
	}
	gui_->readConfigFrom(cd);

	gui_->activate();

	return RTC::RTC_OK;
}


/*
  RTC::ReturnCode_t MapViewRTC::onFinalize()
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onStartup(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onShutdown(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onActivated(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onDeactivated(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/

/*!
 *  @details
 *  アクティブ時，周期的にコールされるルーチン
 *  <ol>
 *    <li> 全ての入力ポートをサーチし，入力があれば，@a gui_に処理させる</li>
 *    <li> @a gui_から出力があればそれはqueueに入るので，内容を出力ポートに書き出す</li>
 *
 *  </ol>
 */

RTC::ReturnCode_t MapViewRTC::onExecute(RTC::UniqueId ec_id)
{
	(void) ec_id;

	if (!gui_->is_running()) {
		cout << "GUI is not running\n";
		return RTC::RTC_OK;
	}
	enum { DISPLEN = 4096 };

	for (size_t i=0, imax=m_inIns.size(); i < imax; ++i) {
		if (m_inIns[i]->isNew()) {
			m_inIns[i]->read();
			string recv(m_ins[i]->data);
			size_t len = recv.length();
			const string& disp = (len>DISPLEN) ? recv.substr(0, DISPLEN) : recv;

			cout << "On Input Port No. " << i << "\n"
				 << "Received:  "   << disp << "\n";
			if (len > DISPLEN)
				cout << " .... " << len - DISPLEN << " bytes truncated\n";
			cout << "TimeStamp: "   << m_ins[i]->tm.sec << "[s] "
				 << m_ins[i]->tm.nsec << "[ns]" << endl;
                        
			gui_->enqueueXml(recv);
		}
	}

	string result;
	while (gui_->tryDequeueResult(result)) {
		setTimeStamp(m_out);
		m_out.data = result.c_str();
		m_outOut.write();
		size_t len = result.length();
		const string& disp = (len>DISPLEN) ? result.substr(0, DISPLEN) : result;


		cout << "Sent:  "       << disp << "\n";
		if (len > DISPLEN)
			cout << " .... " << len - DISPLEN << " bytes truncated\n";
		cout << "TimeStamp: "   << m_out.tm.sec << "[s] "
			 << m_out.tm.nsec << "[ns]" << endl;
	}

	return RTC::RTC_OK;
}

/*
  RTC::ReturnCode_t MapViewRTC::onAborting(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onError(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onReset(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onStateUpdate(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/
/*
  RTC::ReturnCode_t MapViewRTC::onRateChanged(RTC::UniqueId ec_id)
  {
  return RTC::RTC_OK;
  }
*/


extern "C"
{
	/*! このコンポーネントの初期化メソッド.
	 *  RTC::Managerにこのコンポーネントの情報，生成／破壊関数を登録する。
	 *  @param[in] manager managerへのポインタ
	 */
	void MapViewRTCInit(RTC::Manager* manager)
	{
		RTC::Properties profile(MapViewRTC_spec);
		manager->registerFactory(profile,
								 RTC::Create<MapViewRTC>,
								 RTC::Delete<MapViewRTC>);
	}
  
};
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
