/*!
 * @file map_view.h
 * @brief Widget for displaying Map Layers.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */


#if !defined(MAPVIEW_HEADER_)
#define      MAPVIEW_HEADER_

#include <QFrame>
#include "common_defs.h"
QT_FORWARD_DECLARE_CLASS(QEvent);
QT_FORWARD_DECLARE_CLASS(QGraphicsView);
QT_FORWARD_DECLARE_CLASS(QGraphicsScene);
QT_FORWARD_DECLARE_CLASS(QGraphicsItem);
QT_FORWARD_DECLARE_CLASS(QSlider);
QT_FORWARD_DECLARE_CLASS(QToolButton);
QT_FORWARD_DECLARE_CLASS(QLabel);
QT_FORWARD_DECLARE_CLASS(QAction);
QT_FORWARD_DECLARE_CLASS(QGraphicsSceneContextMenuEvent);
QT_FORWARD_DECLARE_CLASS(QGraphicsSceneMouseEvent);
class MapGrid;


/*! 
 * @class MapView
 * @brief 地図レイヤ表示widget。
 *  
 *  QGraphicsView/QGraphicsSceneを持つ表示画面と，各種操作用
 *  (zoom/scroll等) のwidgetから成る。MapLayerはQGraphicsItemから派生す
 *  るため，QGraphicsItemとして表示される。
 *  
 *  地図レイヤの他に，特殊なQGraphicsItemとしてMapGridを内部管理する。
 *  MapGridは全ての地図レイヤの上に描画される座標格子である。
 *  
 *  MapGridはZ値としてGRID_ZVALUE値を持つ。ほかのMapLayerがこれより小さ
 *  なZ値を持つ(というルールを守る)ことで，Gridの最上位への描画を実現し
 *  ている。
 */
class MapView : public QFrame
{
	Q_OBJECT;
public:
	//! zoom 縮尺のデフォルト値: 単位は1mあたりのpixel数
	static const qreal DEFAULT_PIXEL_PER_METER = 10.0;
	//! 座標格子グリッドのZ値。地図レイヤのZ値はこれ未満とせよ
	static const qreal GRID_ZVALUE = 100.0;

	//! レイヤの取りうる最大のzvalue
	static const qreal LAYER_ZVALUE_MAX = 0;

	MapView(QWidget *parent = 0); //!> コンストラクタ

	void addItem(QGraphicsItem* item); //!> QGraphicsItemを追加 
	void removeItem(QGraphicsItem* item); //!>QGraphicsItemを削除

	bool isAutoScrolling() const { return is_autoscrolling_; } //!
public slots:
	void scrollMap(int dx, int dy); //!> Slot:地図をスクロール

	/*! 地図座標のposをcenterにする。
	 *
	 *  地図座標をGraphicsView座標に変換してQGraphicsView::centerOn()を
	 *  呼ぶため，点がポートの外部に近い場合は厳密に中央にはならない。
	 * 
	 *  @see QGraphicsView::centerOn(const QPointF&);
	 */
	void centerOnGeo(const QPointF& pos);

	void updateScene();				//! QGraphicsSceneの再描画
signals:
	//! signal:目的地変更時に発火
	void destinationChanged(const QPointF& pos);

	//! signal:現在地変更時に発火
	//! @param pos 設定する自己の現在地
	//! @param theta 方位角[rad]
	void locationChanged(const QPointF& pos, qreal theta);

	//! signal: autoscroll 変更時に発火
	void autoscrollToggled(bool autoscroll_is_on);
protected:
	/*! QGraphicsSceneのContextMenuEvent ハンドラ。
	 *  eventFilter経由で呼ばれる。
	 */
	void sceneContextMenuEvent(QGraphicsSceneContextMenuEvent* e);

	/*! QGraphicsSceneのMouseMoveハンドラ。地図上の現在位置表示に使用
	 *  eventFilter経由で呼ばれる。
	 */
	void sceneMouseMoveEvent(QGraphicsSceneMouseEvent* e);

	/*! QGraphicsSceneのLeaveイベントハンドラ。Sceneからマウスが外れた。
	 *  地図上の現在位置表示のために使用
	 */
	void sceneLeaveEvent(QEvent* e);

	/*! 本widgetが管理するMapGridの表示範囲を更新。itemの追加，削除
	 * 時に呼ばれる
	 */
	void updateGridExtent();

	/*! イベントフィルタ。@see QObject::installEventFilter() 
	 *
	 *  ここでは，QGraphicsSceneのイベントをこのウィジェットでハン
	 *  ドルさせるために使用
	 */
	bool eventFilter(QObject *target, QEvent *event);

	//! 画像を保存する。
	bool save(const QImage& image);
private slots:
	void takeSnapShot();       //!< 表示地図全体のsnapshot作成
	void takeWindowSnapShot(); //!< 表示windowのsnapshot作成
	void resetView();          //!< viewをリセット(zoom，位置初期化) 
	void setupMatrix();        //!< scene->view変換行列の計算
	void zoomIn();             //!< 一段zoomIn
	void zoomOut();            //!< 一段zoomOut
        
	void toggleGrid(int state); //!< Grid表示/非表示変更
	void toggleAutoScroll(int state); //!< 自動スクロール変更

	void setDestination();  //!< 目的地を設定
	void setLocation();  //!< 現在地を設定

	//!< グリッドの間隔が変化したときにコールされ，ラベルを変更
	void onGridSpacingChanged(qreal xGridSpacing, qreal xFineGridSpacing,
							  qreal yGridSpacing, qreal yFineGridSpacing); 

private:
	QGraphicsView* view_;   //!< 地図表示用QGraphicsView
	QGraphicsScene* scene_; //!< 地図表示用QGraphicsScene
	QSlider* zoomSlider_;   //!< zoom変更用スライダ
	QLabel* scaleLabel_;    //!< 縮尺表示ラベル
	QLabel* posLabel_;      //!< カーソル位置表示用ラベル
	MapGrid* mapGrid_;      //!< 座標格子

	QAction* setDestinationAction_; //!< Action: 目的地設定
	QAction* setLocationAction_;	//!< Action: 現在地設定
	QAction* snapshotAction_;       //!< Action: 地図snapshot作成
	QAction* snapshotWindowAction_; //!< Action: Window Snapshot作成

	bool is_autoscrolling_;		//!< flag: auto scrollのOn/Off 
};


#endif /*!MAPVIEW_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
