/*!
 * @file map_view.cpp
 * @brief Widget for displaying Map Layers.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <QtGui>
#include <memory>
#include <qmath.h>

#include "map_view.h"
#include "map_grid.h"

using namespace std;
namespace /* unnamed */ {
QPointF
imageToGeo(const QPointF& p)
{
	return QPointF(p.x(), -p.y());
}
QPoint
imageToGeo(const QPoint& p)
{
	return QPoint(p.x(), -p.y());
}
QPoint
imageToGeo(int x, int y)
{
	return QPoint(x, -y);
}
} /* unnamed namespace */

/* ================================================================ *
 *  Class MapView
 * ================================================================ */
MapView::MapView(QWidget* parent)
	: QFrame(parent), is_autoscrolling_(false)
{
	setFrameStyle(Sunken | StyledPanel);
	view_ = new QGraphicsView;
	view_->setRenderHint(QPainter::Antialiasing, false);
	view_->setDragMode(QGraphicsView::ScrollHandDrag);
	view_->setOptimizationFlags(QGraphicsView::DontSavePainterState);
	view_->setOptimizationFlags(QGraphicsView::DontClipPainter);
	view_->setViewportUpdateMode(QGraphicsView::SmartViewportUpdate);

	scene_ = new QGraphicsScene;
	view_->setScene(scene_);

	mapGrid_ = new MapGrid;
	mapGrid_->setZValue(GRID_ZVALUE);
	scene_->addItem(mapGrid_);

	int size = style()->pixelMetric(QStyle::PM_ToolBarIconSize);
	QSize iconSize(size, size);
	QToolButton *zoomInIcon = new QToolButton;
	zoomInIcon->setAutoRepeat(true);
	zoomInIcon->setAutoRepeatInterval(33);
	zoomInIcon->setAutoRepeatDelay(0);
	zoomInIcon->setIcon(QPixmap(":images/zoomin.png"));
	zoomInIcon->setIconSize(iconSize);
	QToolButton *zoomOutIcon = new QToolButton;
	zoomOutIcon->setAutoRepeat(true);
	zoomOutIcon->setAutoRepeatInterval(33);
	zoomOutIcon->setAutoRepeatDelay(0);
	zoomOutIcon->setIcon(QPixmap(":images/zoomout.png"));
	zoomOutIcon->setIconSize(iconSize);
	zoomSlider_ = new QSlider;
	zoomSlider_->setMinimum(-250);
	zoomSlider_->setMaximum(250);
	zoomSlider_->setValue(0);
	zoomSlider_->setTickPosition(QSlider::TicksRight);

	QVBoxLayout *zoomSliderLayout = new QVBoxLayout;
	zoomSliderLayout->addWidget(zoomInIcon);
	zoomSliderLayout->addWidget(zoomSlider_);
	zoomSliderLayout->addWidget(zoomOutIcon);

	QHBoxLayout *checkBoxLayout = new QHBoxLayout;
	QCheckBox *showGridCheckBox = new QCheckBox(tr("Show Grid"));
	showGridCheckBox->setCheckState(mapGrid_->isVisible() ? 
									Qt::Checked : Qt::Unchecked);

	QCheckBox *autoScrollCheckBox = new QCheckBox(tr("Auto Scroll"));
	autoScrollCheckBox->setCheckState(isAutoScrolling() ? 
									  Qt::Checked : Qt::Unchecked);
	scaleLabel_ = new QLabel;
	posLabel_ = new QLabel("Grid: xxxxx, FineGrid: yyyy");

	checkBoxLayout->addWidget(showGridCheckBox);
	checkBoxLayout->addWidget(autoScrollCheckBox);
	checkBoxLayout->addStretch();
	checkBoxLayout->addWidget(posLabel_);
	checkBoxLayout->addWidget(scaleLabel_);

	scene_->installEventFilter(this);

	snapshotAction_ = new QAction(QIcon(":images/camera.png"),
								  tr("Snapshot Whole Map"), this);
	QToolButton *snapshotIcon = new QToolButton;
	snapshotIcon->setDefaultAction(snapshotAction_);

	snapshotWindowAction_ = new QAction(QIcon(":images/camera_win.png"),
										tr("Snapshot Window"), this);
	QToolButton *snapshotWinIcon = new QToolButton;
	snapshotWinIcon->setDefaultAction(snapshotWindowAction_);
        
	QHBoxLayout *snapButtonsLayout = new QHBoxLayout;
	snapButtonsLayout->addWidget(snapshotIcon);
	snapButtonsLayout->addWidget(snapshotWinIcon);
        
	QGridLayout *topLayout = new QGridLayout;
	topLayout->addLayout(zoomSliderLayout,  0, 0);
	topLayout->addWidget(view_,             0, 1);
	topLayout->addLayout(snapButtonsLayout, 1, 0);
	topLayout->addLayout(checkBoxLayout,    1, 1);

	setLayout(topLayout);

	connect(zoomSlider_, SIGNAL(valueChanged(int)), this, 
			SLOT(setupMatrix()));
	connect(zoomInIcon, SIGNAL(clicked()), this, SLOT(zoomIn()));
	connect(zoomOutIcon, SIGNAL(clicked()), this, SLOT(zoomOut()));

	connect(showGridCheckBox, SIGNAL(stateChanged(int)), 
			this, SLOT(toggleGrid(int)));

	connect(autoScrollCheckBox, SIGNAL(stateChanged(int)), 
			this, SLOT(toggleAutoScroll(int)));

	connect(snapshotAction_, SIGNAL(triggered()),
			this, SLOT(takeSnapShot()));
	connect(snapshotWindowAction_, SIGNAL(triggered()),
			this, SLOT(takeWindowSnapShot()));

	connect(mapGrid_, SIGNAL(gridSpacingChanged(qreal,qreal,qreal,qreal)),
			this, SLOT(onGridSpacingChanged(qreal,qreal,qreal,qreal)));

	setDestinationAction_ = new QAction(QIcon(":/images/pos_set.png"),
										tr("Set &Destination"), this);
        
	setDestinationAction_->setStatusTip(tr("Set Destination"));
	connect(setDestinationAction_, SIGNAL(triggered()), 
			this, SLOT(setDestination()));

	setLocationAction_ = new QAction(QIcon(":/images/pos_set.png"),
									 tr("Set Currnet &Location"), this);
        
	setLocationAction_->setStatusTip(tr("Set Current Location"));
	connect(setLocationAction_, SIGNAL(triggered()), 
			this, SLOT(setLocation()));

	setupMatrix();
}
void
MapView::addItem(QGraphicsItem* item)
{
	scene_->addItem(item);
	updateGridExtent();
}
void
MapView::removeItem(QGraphicsItem* item)
{
	scene_->removeItem(item);
	updateGridExtent();
}
void
MapView::sceneContextMenuEvent(QGraphicsSceneContextMenuEvent* e)
{
	QPointF pos(imageToGeo(e->scenePos()));

	QMenu contextMenu(this);

	QString coord(tr(" to (%1, %2)")
				  .arg(pos.x(), 10, 'f', 3)
				  .arg(pos.y(), 10, 'f', 3));

	// store old value
	const QString destActionText(setDestinationAction_->text());
	const QString locationActionText(setLocationAction_->text());

	setDestinationAction_->setText(QString(destActionText).append(coord));
	setLocationAction_->setText(QString(locationActionText).append(coord));

	setDestinationAction_->setData(pos);
	setLocationAction_->setData(pos);

	contextMenu.addAction(setDestinationAction_);
	contextMenu.addSeparator();
	contextMenu.addAction(setLocationAction_);	
	contextMenu.addSeparator();
	contextMenu.addAction(snapshotAction_);
	contextMenu.addAction(snapshotWindowAction_);

	contextMenu.exec(QCursor::pos());
	setDestinationAction_->setText(destActionText);
	setLocationAction_->setText(locationActionText);
}
void
MapView::sceneMouseMoveEvent(QGraphicsSceneMouseEvent* e)
{
//        qDebug() << "e->scenePos()" << imageToGeo(e->scenePos());
	QPointF pos(imageToGeo(e->scenePos()));

	posLabel_->setText(tr("(%1, %2)")
					   .arg(pos.x(), 10, 'f', 3)
					   .arg(pos.y(), 10, 'f', 3));
}
void
MapView::sceneLeaveEvent(QEvent* event)
{
	Q_UNUSED(event);
	posLabel_->setText("(None, None)");
}
void
MapView::updateGridExtent()
{
	QRectF gridExtent;
	bool first = true;
	foreach (QGraphicsItem* item, scene_->items()) {
		if (item == mapGrid_) {
			continue;
		} else if (first) {
			first = false;
			gridExtent = item->boundingRect();
		} else {        // !first
			gridExtent = gridExtent.united(item->boundingRect());
		}
	}
	mapGrid_->setGridExtent(gridExtent);
}

void
MapView::resetView()
{
	zoomSlider_->setValue(0);
	setupMatrix();
}
void
MapView::setupMatrix()
{
	qreal scale = DEFAULT_PIXEL_PER_METER * 
		qPow(qreal(2), (zoomSlider_->value()) / qreal(50));
	qDebug() << "setupMatrix: scale = " << scale << "\n";

	QMatrix m;
	m.scale(scale, scale);
	view_->setMatrix(m);
}
void
MapView::zoomIn()
{
	zoomSlider_->setValue(zoomSlider_->value() + 1);
	qDebug() << "zoomin: " 
             << zoomSlider_->value() << "\n";
}
void
MapView::zoomOut()
{
	zoomSlider_->setValue(zoomSlider_->value() - 1);
	qDebug() << "zoomout: " 
             << zoomSlider_->value() << "\n";
}
void
MapView::onGridSpacingChanged(qreal xGridSpacing, qreal xFineGridSpacing,
                              qreal yGridSpacing, qreal yFineGridSpacing)
{
	if (mapGrid_->isVisible()) {
		QString label("Grid: ");
		if (xGridSpacing < 0)
			label.append("none");
		else
			label.append(tr("%1").arg(xGridSpacing));

		label.append(" x ");

		if (yGridSpacing < 0)
			label.append("none");
		else
			label.append(tr("%1").arg(yGridSpacing));

		label.append("[m] Fine: ");
		if (xFineGridSpacing < 0)
			label.append("none");
		else
			label.append(tr("%1").arg(xFineGridSpacing));

		label.append(" x ");

		if (yFineGridSpacing < 0)
			label.append("none");
		else
			label.append(tr("%1").arg(yFineGridSpacing));
		label.append("[m]");                
                              
		scaleLabel_->setText(label);
	} else {
		scaleLabel_->setText(tr(""));
	}

}

void
MapView::takeSnapShot()
{
	qDebug() << "takeSnapShot()";
	// sceneのスクリーンショット。表示されている部分ではなく
	// sceneにrenderされた全体がcaptureされる
	// とりあえず機能の確認。

	// Todo:
	// ここで保存までしてしまうのはよろしくないので変更
	// `snapshot できたよ' signal にするのが良い？
        
	static const int MIN_SIZE = 480;

	QRectF sceneRect = view_->sceneRect();
	const qreal sceneWidth  = sceneRect.width();
	const qreal sceneHeight = sceneRect.height();

	qDebug() << "sceneRect = " << sceneRect;

	qreal scale = 1.0;
	int tmp = qMin(sceneWidth, sceneHeight);
	if (tmp < MIN_SIZE)
		scale = qreal(MIN_SIZE) / qMax(tmp, 1);

	QImage image(int(sceneWidth*scale), int(sceneHeight*scale),
				 QImage::Format_ARGB32);
	QPainter painter(&image);
	painter.setRenderHint(QPainter::Antialiasing, true);
	painter.setRenderHint(QPainter::SmoothPixmapTransform, true);

	scene_->render(&painter);
	painter.end();
	save(image);
}
void
MapView::takeWindowSnapShot()
{
	qDebug() << "takeWindowShot()";
	// Windowのスクリーンショット。表示されている部分をcapture
	// cf. takeSnapShot()
	QRectF vrect(view_->viewport()->rect());

	// BUG? : qreal->intをcastで済ましているが，
	//        round/ceil/floorのいずれが適切か調べる必要あり
        
	QImage image(int(vrect.width()), int(vrect.height()),
				 QImage::Format_ARGB32);
	QPainter painter(&image);
	painter.setRenderHint(QPainter::Antialiasing, true);
	painter.setRenderHint(QPainter::SmoothPixmapTransform, true);

	view_->render(&painter);
	painter.end();
	save(image);
}
void
MapView::setDestination()
{
	QAction* action = qobject_cast<QAction*>(sender());
	if (action) {
		QPointF pos_geo = action->data().toPointF();
		qDebug() << "SetDestination: current pos in geo coord" 
				 << pos_geo;
		emit destinationChanged(pos_geo);
	}
}
void
MapView::setLocation()
{
	QAction* action = qobject_cast<QAction*>(sender());
	if (action) {
		QPointF pos_geo = action->data().toPointF();
		qDebug() << "setLocation:current pos in geo coord" 
				 << pos_geo;

		bool ok = false;
		double deg = QInputDialog::getDouble(this, 
											 tr("QInputDialog::getDouble()"),
											 tr("Theta[deg]:"), 
											 0, 0, 360, 1, &ok);
		if (ok)
			emit locationChanged(pos_geo, 3.14159265358979 * deg / 180.0);
	}	
}
void
MapView::toggleGrid(int state)
{
	mapGrid_->setVisible(state == Qt::Checked);
}
void
MapView::scrollMap(int dx, int dy)
{
	qDebug() << "scrollMap(" << dx << ", " << dy << ")\n";
	QPoint geo(imageToGeo(dx, dy));
	view_->scroll(geo.x(), geo.y());
}
void
MapView::centerOnGeo(const QPointF& pos)
{
	view_->centerOn(imageToGeo(pos));
}
void
MapView::updateScene()
{
	view_->update();
	scene_->update();
}
void 
MapView::toggleAutoScroll(int state)
{
	is_autoscrolling_ = (state == Qt::Checked);
	qDebug() << "toggleAutoScroll : " << (is_autoscrolling_  ? " ON" : " OFF");
	emit autoscrollToggled(is_autoscrolling_);
}

bool
MapView::eventFilter(QObject *target, QEvent *event)
{
	if (target != scene_)
		return QFrame::eventFilter(target, event);

	const QEvent::Type type = event->type();
                
	if (type == QEvent::GraphicsSceneMouseMove) {
		sceneMouseMoveEvent((QGraphicsSceneMouseEvent*)event);
		return false;   // must call its own handler
	} else if (type == QEvent::Leave) {
		sceneLeaveEvent(event);
		return false;   // must call its own handler
	} else if (type == QEvent::GraphicsSceneContextMenu) {
		sceneContextMenuEvent((QGraphicsSceneContextMenuEvent*)event);
		return true;
	}

	return false;
}
bool
MapView::save(const QImage& image)
{
	QString str_now(QDateTime::currentDateTime().toString(
						"'./snapshot_'yyyymmddhhmmss_zzz'.png'"));
	QString fileName = QFileDialog::
		getSaveFileName(this, tr("Save Snapshot"), 
						str_now,
						tr("Images (*.png *.xpm *.jpg)"));
	if (fileName.isEmpty()) {
		qDebug() << "not saved";
		return false;
	}
        
	return image.save(fileName);
}
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
