/*!
 * @file map_manager.h
 * @brief Manager class to handle adding, deleting and XML-parsing of MapLayers
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#ifndef MAP_MANAGER_HEADER_
#define MAP_MANAGER_HEADER_

#include <string>
#include <QObject>
#include <QDir>
#include <boost/shared_ptr.hpp>
#include "shared_queue.h"
#include "common_defs.h"

QT_FORWARD_DECLARE_CLASS(QSize);
QT_FORWARD_DECLARE_CLASS(QPoint);
QT_FORWARD_DECLARE_CLASS(QString);
QT_FORWARD_DECLARE_CLASS(QPointF);
QT_FORWARD_DECLARE_CLASS(QImage);
QT_FORWARD_DECLARE_CLASS(QTimer);

class MapView;
class LayerView;
class MapLayer;
class WidgetObserver;

/*!
 * @class MapManager
 *
 * @brief MapViewとLayerViewの間に入り両者の仲介と，RTC/IFからの
 *  データ入出力を管理する。
 *  
 */
class MapManager : public QObject 
{
	Q_OBJECT;

	typedef SharedQueue<std::string> Queue_t;
public:
	//! コンストラクタ。管理下に置くMapView, LayerViewへのポインタを指定
	MapManager(MapView* mapView, LayerView* layerView);
	virtual ~MapManager();  //!<仮想デストラクタ

	//! MapLayerの追加。
	//! @param[in] layer 追加するlayer
	int addMapLayer(MapLayer* layer);

	//! 名称からレイヤを返す。
	//! @param[in] name 探すレイヤの名称
	MapLayer* layerOfName(const QString& name);

	//! 名称を指定したMapLayerの削除。見つからなければなにもしない
	//! @param[in] name 削除するレイヤの名前
	//! @retval 実際に削除されたかどうか
	bool removeLayerOfName(const QString& name);

	//! 入出力queueを設定
	//! @see MapManager::XmlIF::setIOQueue()
	void setIOQueue(Queue_t* xmlq, Queue_t* resultq);

	//! 偽色表示のための値をセット
	//! @see MapManager::XmlIF::setPotentialRange()
	void setPotentialRange(double pot_min, double pot_max);

	//! Windowの位置，大きさ情報をファイルから読み出し，反映する
	//! @retval 0 成功
	//! @retval negative 失敗
	int loadGuiSettings();

	//! Windowの位置，大きさ情報をファイルに保存する
	//! @retval 0 成功
	//! @retval negative 失敗
	int saveGuiSettings();

	void saveAllLayers() const;	//!< 表示中のレイヤをファイルに保存
	void loadAllLayers();		//!< 保存したレイヤを読み込む

	/*! 起動時にデータを読み込むディレクトリ名を設定
	 *  @param[in] newdir ディレクトリ名
	 */
	void setStartupDirectory(const QString& newdir);

	/*! ダイアログを出してレイヤの名称を訪ねる。
	 *  @param [out] name 新名称をここに格納
	 *  @return 処理を続行してよいか。
	 */
	bool askNewLayerName(QString& name);

public slots:
	//! MapViewからの「目的地変更」のためのslot：
	void on_destinationChanged(const QPointF& pos);

	//! MapViewからの「現在地変更」のためのslot：
	void on_locationChanged(const QPointF& pos, qreal theta);

private:
	/*! Layerを上書きしてよいか確認する。
	 *  @param[in] layer_name レイヤの名前
	 *  @return この名称のlayerを上書きして良いかどうか
	 */ 
	bool mayOverwrite(const QString& layer_name);

	MapView* mapView_;      //!< 管理するmapViewへのポインタ
	LayerView* layerView_;  //!< 管理するLayerViewへのポインタ
	QTimer* autoscroll_timer_;	//!< mapViewのautoScrollに使用するタイマ

	WidgetObserver* mapViewObserver_; //!< mapView_のサイズ，位置変更を監視
	WidgetObserver* layerViewObserver_;//!< layerview_のサイズ，位置変更を監視

	class XmlIF;			//!< 実装クラスの宣言
	boost::shared_ptr<XmlIF> xmlif_; //!< xml解析を受け持つ内部クラス

	QDir startup_dir_;			//!< 初期設定ファイルが格納されたディレクトリ
	static const char INIT_FILE_NAME[]; //!< 初期設定ファイルの名称
	static const char LAYER_ARCHIVE_FILENAME[]; //!< レイヤ保存ファイルの名称
	static const int QUEUE_POLLING_INTERVAL = 30; //!< queueをpoll間隔[msec]
	static const int AUTOSCROLL_INTERVAL = 250;    //!< autoscroll間隔[msec]

private slots:

	/*! layerViewのsignalを受け取るためのprivate Slot。mapViewのアップデート
	 *  を行う。
	 *  @param [in] layers 変化後のlayer情報
	 */
	void on_layersChanged(const QVector<MapLayer*>& layers);

	/*! layerViewのsignalを受け取るためのprivate Slot。
	 *  layerの削除を行う
	 *  @param[in] 削除されるべきレイヤ。
	 */
	void on_removeLayerRequested(MapLayer* layer);

	//! 入力Queueをのぞき，中身があればdequeueして処理
	//!  @see MapManager::XmlIF::try_dequeue()
	void try_dequeue();

	//! layerViewの新規レイヤ作成シグナルを受信。
	//! ダイアログを作成し，ユーザにファイルを
	//! 指定させ，読み込み処理を行う
	void openNewLayerFromFile();

	/*! layerView, mapViewのclose eventがWindowObserver経由で
	 *  ここに伝わる。どちらのウィンドウから来た場合も
	 *  終了確認ダイアログを出す。
	 */
	void on_closing(QWidget* target, bool& accepted);

	/*! mapView_のsignal: autoscrollToggledを受け取る。
	 *  timerを有効にして自動スクロール処理をコールするようになる
	 */
	void on_autoscrollToggled(bool autoscroll_is_on);

	void do_autoscroll();		//!< mapViewの自動スクロールを行う
	void closeAll();			//!< windowを閉じて終了
	
};

#endif /* MAP_MANAGER_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
