/*!
 * @file map_manager.cpp
 * @brief Manager class to handle adding, deleting and XML-parsing of MapLayers
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <QtGui>

#include <sys/time.h>
#include <time.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <streambuf>
#include <cctype>
#include <cmath>

#include "baseRTC.h"
#include "layer_view.h"
#include "map_view.h"
#include "map_layer.h"
#include "map_manager.h"
#include "gui_settings.h"
#include "debug_utils.h"
#include "navigation_data.h"
#include "Localization.h"
#include "GridMap.h"
#include "Path.h"
#include "widget_observer.h"
#include "layer_archive.h"

using namespace std;

/*! @class MapManager::XmlIF
 *  @brief MapManagerのXml読み書きを行う部分を分離した実装クラス
 */
class MapManager::XmlIF {
public:
	/*! コンストラクタ。
	 * @param[in] 自身を生成したMapManagerへの参照
	 */
	XmlIF(MapManager* mgr);

	/*! 入出力に用いるqueueを設定する。
	 * @param[in] xmlq このクラスがxmlを読み込むqueue。RTCはこれに書き込む
	 * @param[in] resultq このクラスが出力を書込むqueue RTCはここから読む
	 */
	void setIOQueue(Queue_t* xmlq, Queue_t* resultq);

	//! GUIから指定された目的地をXMLとして書き出す
	//! @param [in] pos 書き出す目的地情報
	void emitDestination(const QPointF& pos) const;

	//! GUIから指定された現在地XMLとして書き出す
	//! @param [in] pos 書き出す目的地情報
	//! @param [in] theta 方位[rad]
	void emitLocation(const QPointF& pos, qreal theta) const;

	//! 入力queueをのぞき，有ればdequeして処理する。
	//! timerから呼ばれる。
	void try_dequeue();

	/*! Pseudo Color表示のための値をセット。
	 * @param[in] pot_min これ以下のポテンシャルは最低値として同じ色にする
	 * @param[in] pot_max これ以下のポテンシャルは最高値として同じ色にする
	 */
	void setPotentialRange(double pot_min, double pot_max);

	//! XML処理のエントリポイント。ヘッダをパースして下請け関数をdispatch
	//! @param[in] xml 処理するXMLが格納された文字列
	//! @param[in] ask_overwrite 同名のレイヤが有ったとき確認するかどうか
	bool processXml(const string& xml, bool ask_overwrite=false);

	/*! XMLのヘッダを読み込む
	 *  @param is 入力ストリーム
	 *  @param ihdr ここにヘッダを読み込む
	 *  @return 成功したかどうか
	 */
	bool readXmlHeader(istream& is, baseRTC::Header& ihdr);

	//! 現在地を無効化
	void invalidateCurpos();

	/*! 現在地が有れば返す
	 *
	 *  @param[out] pos 現在地があればここに返される
	 *  @retval false 現在地は設定されていない。引数@a posは不変
	 *  @retval true 現在地が引数@a posに返された
	 */
	bool getCurrentPos(QPointF& pos) const;

	static const char CURPOS_LAYER_NAME[]; //! 現在地layerを表す名前

private:

	//! processXmlの下請け関数。目的地レイヤーを生成
	//! @param[in] hdr 受信したヘッダ
	//! @param[in] is XMLを読むためのストリーム
	//! @retval nonnull  生成されたレイヤへのポインタ
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkDestinationLayer(const baseRTC::Header& hdr, istream& is);

	//! processXmlの下請け関数。現在地レイヤーを生成
	//! @param[in] hdr 受信したヘッダ
	//! @param[in] is XMLを読むためのストリーム
	//! @retval nonnull  生成されたレイヤへのポインタ
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkSelfPosLayer(const baseRTC::Header& hdr, istream& is);

	//! processXmlの下請け関数。経路レイヤーを生成
	//! @param[in] is XMLを読むためのストリーム
	//! @param[in] hdr 受信したヘッダ
	//! @retval nonnull  生成されたレイヤへのポインタ
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkPathLayer(const baseRTC::Header& hdr, istream& is);

	//! processXmlの下請け関数。ポテンシャルレイヤを作成
	//! @param[in] hdr 受信したヘッダ
	//! @param[in] is XMLを読むためのストリーム
	//! @retval nonnull  生成されたレイヤへのポインタ
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkPotentialLayer(const baseRTC::Header& hdr, istream& is);

	//! processXmlの下請け関数。確率グリッドレイヤを作成
	//! @param[in] hdr 受信したヘッダ
	//! @param[in] is XMLを読むためのストリーム
	//! @retval nonnull  生成されたレイヤへのポインタ
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkGridMapLayer(const baseRTC::Header& hdr, istream& is);

	//! processXmlの下請け関数。CommandStringのtypeを見て処理する。
	//! @param[in] hdr 受信したヘッダ
	//! @param[in] is XMLを読むためのストリーム
	//! @retval nonnull  生成されたレイヤへのポインタ(もしあったら)
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* proc_CommandString(const baseRTC::Header& hdr, istream& is);


	//! proc_CommandStringの下請け関数。text layerを生成
	//! @param[in] arg コマンドの引数文字列
	//! @retval nonnull  生成されたレイヤへのポインタ(もしあったら)
	//! @retval NULL  パースエラーなどで生成されなかった場合
	MapLayer* mkTextLayer(const string& arg);



	//! 偽色表示のためのテーブル作成
	void calc_color_tbl();

	Queue_t *xmlq_;    //!< このqueueからxmlを読む  
	Queue_t *resultq_; //!< このqueueへ結果を出す

	MapManager* mgr_; //!< 自分を生成したMapManagerへの参照

	static const char NAME[];				   //!< XML送信ヘッダのnameの値
	static const int N_COLOR_TBL_ENTRY = 1024; //!< カラーテーブルの大きさ
	QVector<QRgb> tbl_;			//!< pseudo color table 
	qreal pot_min_; //!< 偽色表示の際，これ以下の値は最低値と見なし同色とする
	qreal pot_max_; //!< 偽色表示の際，これ以上の値は最高値と見なし同色とする

	Localization curpos_;		//!< 現在位置
	bool have_cur_pos_;			//!< 現在位置が設定されたかどうか

	// temporal
	time_t last_path_time_;		// temporal
	
};
const char MapManager::XmlIF::NAME[] = "MapViewer";
const char MapManager::XmlIF::CURPOS_LAYER_NAME[] = "CurrentPosition";

MapManager::XmlIF::XmlIF(MapManager* mgr) 
	: xmlq_(0), resultq_(0), mgr_(mgr),  tbl_(),
	  pot_min_(0.0), pot_max_(1000.0),
	  curpos_(), have_cur_pos_(false),
	  last_path_time_(0)			// temporal
{
	calc_color_tbl();
}
void 
MapManager::XmlIF::calc_color_tbl()
{
	static const double PI = 3.14159265358979323846264338327950288;
	tbl_.clear();

	for (int i = 0; i < N_COLOR_TBL_ENTRY; ++i) {
		double x = double(i) * PI / N_COLOR_TBL_ENTRY;
		double s = sin(x);
		double c = cos(x);
		uint8_t r = 255 * (1.0 - c) / 2.0;
		uint8_t g = 255 * s;
		uint8_t b = 255 * (1.0 + c) / 2.0;

		tbl_.push_back(qRgb(r, g, b));
	}
}
void
MapManager::XmlIF::setIOQueue(Queue_t* new_xmlq, Queue_t* new_resultq)
{
	xmlq_ = new_xmlq;
	resultq_ = new_resultq;
}
void
MapManager::XmlIF::emitDestination(const QPointF& pos) const
{
	FUNC_TRACE;
	if (resultq_ == NULL) {
		qDebug() << "No result queue for output!";
	}
	baseRTC::Header hdr(NavData::CATEGORY,
						NavData::TaggedPoint::TYPE_ID,
						NAME);

	NavData::TaggedPoint destination;
	destination.tag = "destination";
	QDateTime now = QDateTime::currentDateTime();
	destination.time = double(now.toTime_t()) + now.time().msec()*1.0e-3;
	destination.position.x = pos.x();
	destination.position.y = pos.y();
	destination.position.z = 0.0;

	ostringstream os;
	hdr.writeToXml(os);
	destination.writeToXml(os);

	resultq_->enqueue(os.str());
}
void
MapManager::XmlIF::emitLocation(const QPointF& pos, qreal theta) const
{
	FUNC_TRACE;
	if (resultq_ == NULL) {
		qDebug() << "No result queue for output!";
	}
	qDebug() << "pos = " << pos << "\n"
			 << "theta = " << theta << "[rad] = " 
			 << theta * 180 / 3.1415926535 << "[deg]\n";

	baseRTC::Header hdr("navigation",
						Localization::TYPE_ID,
						NAME,
						"",		// id
						"initialize"); // tag
	Localization localization;
	localization.pos_x = pos.x();
	localization.pos_y = pos.y();
	localization.pos_the = theta;

	ostringstream os;
	hdr.writeToXml(os);
	localization.writeToXml(os);
	
	qDebug() << os.str().c_str() << "\n";
	resultq_->enqueue(os.str());
}
void
MapManager::XmlIF::try_dequeue()
{
	if (xmlq_ == NULL) {
		qDebug() << "No queue to read\n";
		return;
	}
	string result;
	while (xmlq_->try_dequeue(result)) {
		qDebug() << "pop!!";
#if 0
		enum { MAXDISP = 256 };
		int len = result.length();
		if (len < MAXDISP)
			qDebug() << result.c_str();
		else
			qDebug() << result.substr(0, MAXDISP).c_str() << "\n"
					 << " .... " << len - MAXDISP << "bytes truncated\n";
#endif
		processXml(result);
	}
}
MapLayer*
MapManager::XmlIF::mkDestinationLayer(const baseRTC::Header& hdr, istream& is)
{
	Q_UNUSED(hdr);
	FUNC_TRACE;

	NavData::TaggedPoint dest;
	if (dest.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}
	
	MarkerMapLayer* layer = new MarkerMapLayer(dest.tag.c_str());
	layer->setMarker(0, 
					 MarkerMapLayer::Vector2D_t(), // Null velocity
					 QPointF(dest.position.x, dest.position.y),
					 0, 0, 0.0,
					 QPen(Qt::blue, 0.3, 
						  Qt::SolidLine, Qt::RoundCap, Qt::MiterJoin));


	return layer;
}
MapLayer*
MapManager::XmlIF::mkSelfPosLayer(const baseRTC::Header& hdr, istream& is)
{
	Q_UNUSED(hdr);

	FUNC_TRACE;

	// メンバ変数を使用する。
	if (curpos_.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}
	have_cur_pos_ = true;

	qDebug() << "have_cur_pos_ = " << have_cur_pos_;

	MarkerMapLayer* layer = new MarkerMapLayer(CURPOS_LAYER_NAME);
	layer->setMarker(0, 
					 QPointF(curpos_.pos_x, curpos_.pos_y),
					 MarkerMapLayer::Vector2D_t(curpos_.vel_x, curpos_.vel_y),
					 curpos_.pos_err_x, curpos_.pos_err_y, curpos_.pos_the,
					 QPen(Qt::red, 0.3, 
						  Qt::SolidLine, Qt::RoundCap, Qt::MiterJoin));
	return layer;
}
MapLayer*
MapManager::XmlIF::mkPathLayer(const baseRTC::Header& hdr, istream& is)
{
	FUNC_TRACE;
	Path path;
	if (path.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}
	time_t now = time(NULL);
	if (now <= last_path_time_) {
		return NULL;
	}
	// going on!
	last_path_time_ = now;

	VectorMapLayer* layer = new VectorMapLayer(QString(hdr.tag.c_str()));
	layer->setPen(0, QPen(Qt::magenta, 0.3,
						  Qt::SolidLine,
						  Qt::RoundCap, 
						  Qt::MiterJoin));
	size_t n_points = path.path.size();
	QVector<QPointF> path_data;
	for (size_t i = 0; i < n_points; ++i) {
		path_data.append(QPointF(path.path[i].node.x, path.path[i].node.y));
	}
	layer->setPath(0, path_data);


	qDebug() << "extent of layer: " << layer->geoExtent();	

	return layer;
}

MapLayer*
MapManager::XmlIF::mkPotentialLayer(const baseRTC::Header& hdr, istream& is)
{
	Q_UNUSED(hdr);

	FUNC_TRACE;
	NavData::PotentialGrid grid;
	if (grid.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}

	const NavData::GridInfo& info = grid.info; // just for readability
	const int width  = info.numCellsW;
	const int height = info.numCellsH;

	if (width*height != int(grid.data.size())) {
		qDebug() << "GridSize mismatch!\n" 
			 << "width x height must be " 
			 << width << " x " << height << " = " << width*height << "\n"
			 << "But grid.data.size() = " << grid.data.size() << "\n";
		return 0;
	}

	const float range_min = grid.data.rangeMin();
	const float range_max = grid.data.rangeMax();

	qDebug() << "range_min = " << range_min << "\n"
			 << "range_max = " << range_max << "\n";
	vector<float> data;
	grid.data.copy_to(back_inserter(data));

	QImage color(QSize(width, height), QImage::Format_ARGB32_Premultiplied);
	
	vector<float>::const_iterator sp = data.begin();
	if (qAbs(range_max - range_min) > tbl_.size()) {
		const float range_wid = range_max - range_min;
		qDebug() << "tone compression\n";
		for (int y = 0; y < height; ++y) {
			QRgb* dp = (QRgb*)color.scanLine(y);
			for (int x = 0; x < width; ++x) {
				float fval = N_COLOR_TBL_ENTRY*((*sp++) - range_min)/range_wid;
				int ival = int(fval);
				if (ival < 0)
					ival = 0;
				else if (N_COLOR_TBL_ENTRY <= ival)
					ival = N_COLOR_TBL_ENTRY - 1;

				dp[x] = tbl_[ival];
			}
		}
	} else {
		qDebug() << "No tone compression\n";
		for (int y = 0; y < height; ++y) {
			QRgb* dp = (QRgb*)color.scanLine(y);
			for (int x = 0; x < width; ++x) {
				float fval = ((*sp++) - pot_min_);
				int ival = int(fval); // truncated toward zero

				if (ival < 0)
					ival = 0;
				else if (N_COLOR_TBL_ENTRY <= ival)
					ival = N_COLOR_TBL_ENTRY - 1;

				dp[x] = tbl_[ival];
			}
		}
	}
	RasterMapLayer* layer = new RasterMapLayer("Potential Grid");
	const QPointF topLeft(info.position.x, info.position.y);
	qDebug() << topLeft;

	const QPointF bottomRight(topLeft.x() + width * info.cellWidth,
							  topLeft.y() - height * info.cellHeight);
	qDebug() << bottomRight;
	layer->setPixmap(QPixmap::fromImage(color),
					 QRectF(topLeft, bottomRight).normalized());

	return layer;
}
MapLayer*
MapManager::XmlIF::mkGridMapLayer(const baseRTC::Header& hdr, istream& is)
{
	FUNC_TRACE;
	GridMap gm;
	if (gm.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}

	if (gm.coordinate != "self") {
		qDebug() << "Unsupported Coordinate!";
		return 0;
	}

	const int width  = gm.sizeX();
	const int height = gm.sizeY();
	if (width*height == 0) {
		qDebug() << "empty GridMap";
		return 0;
	}
	
	if (int(gm.map.size()) != width*height) {
		qDebug() << "Broken Grid Map: size mismatch !"
				 << "(width, height) = (" << width << ", " << height << ")"
				 << " gm.map.size() = " << gm.map.size() << " != " 
				 << "width * height = " << width*height;
		return 0;
	}
	// Assume Grid map ranges between 0 ... 1

	QImage color(QSize(width, height), QImage::Format_ARGB32_Premultiplied);

	vector<float>::const_iterator sp = gm.map.begin();
	for (int y = 0; y < height; ++y) {
		QRgb* dp = (QRgb*)color.scanLine(y);
		for (int x = 0; x < width; ++x) {
			float fval = (*sp++);
			int ival = int(fval * (N_COLOR_TBL_ENTRY - 1));
			if (ival < 0)
				ival = 0;
			else if (N_COLOR_TBL_ENTRY <= ival)
				ival = N_COLOR_TBL_ENTRY - 1;

			dp[x] = tbl_[ival];
		}
	}
	GridMapLayer* layer = new GridMapLayer(QString(tr("GridMap ")).
										   arg(hdr.tag.c_str()));
	layer->setPixmap(QPixmap::fromImage(color));

	QPointF pos(have_cur_pos_ ? QPointF(curpos_.pos_x, curpos_.pos_y) 
				: QPointF(0.0, 0.0));

	qreal theta = have_cur_pos_ ? curpos_.pos_the : 0.0;

	layer->setGridInfo(pos,
					   QRectF(QPointF(gm.area_x0, gm.area_y0),
							  QPointF(gm.area_x1, gm.area_y1)).normalized(),
					   theta);
					  
	return layer;
}
MapLayer*
MapManager::XmlIF::proc_CommandString(const baseRTC::Header& hdr, istream& is)
{
	FUNC_TRACE;
	Q_UNUSED(hdr);

	NavData::CommandString cs;
	if (cs.readFromXml(is) < 0) {
		qDebug() << "Parse error\n";
		return 0;
	}
	QStringList cmd(QString(cs.type.c_str()).split(QRegExp("\\s+")));
	
	QStringList::const_iterator ci;
	for (ci = cmd.constBegin(); ci != cmd.constEnd(); ++ci)
		qDebug() << *ci;

	if (cmd.length() == 0) {
		qDebug() << "proc_CommandString: empty!";
		return 0;
	}

	string::size_type off = cmd[0].size();
	const string& s = cs.type;
	while (isspace(s[off]))
		++off;
	if (off > s.size())
		off = s.size();

	qDebug() << "cmd = [" << cmd[0] << "]\n"
			 << "arg = [" << s.substr(off).c_str() << "]\n";
	if (cmd[0] == "text")
		return mkTextLayer(s.substr(off));
	else
		qDebug() << "Unknown cmd: " << cmd[0] << "\n";

	return 0;
}
MapLayer*
MapManager::XmlIF::mkTextLayer(const std::string& s)
{
	FUNC_TRACE;

	string name;
	qreal x = 0.0, y = 0.0;		// default
	istringstream is(s);
	is >> name >> x >> y;
	
	while (isspace(is.peek()))
		is.ignore();
	if (is.peek() == ',')
		is.ignore();			// space (or even ',') is allowd after ',';

	string text = string(istreambuf_iterator<char>(is), 
						 istreambuf_iterator<char>());

	TextMapLayer* layer = new TextMapLayer(QString(name.c_str()));
	layer->setText(0, QString(text.c_str()), QPointF(x, y));
	
	return layer;
}
bool
MapManager::XmlIF::readXmlHeader(istream& is, baseRTC::Header& ihdr)
{
	if (ihdr.readFromXml(is) < 0) {
		qDebug() << "Cannot Read Header\n";
		return false;
	}
	
	cout << "category:    " << ihdr.category << "\n";
	cout << "sender_name: " << ihdr.sender_name << "\n";
	cout << "type:        " << ihdr.type << "\n";
	cout << "time:        " << setprecision(6) << fixed << ihdr.time << "\n";
	cout << "id:          " << ihdr.id << "\n";
	cout << "tag:         " << ihdr.tag << "\n";
	cout << "version:     " << ihdr.hdr_version << "\n";
	return true;
}
struct LayerIndexInfo {
	QString name;
	int lidx;
	int vidx;

	LayerIndexInfo(): name(), lidx(-1), vidx(-1) { }
	
	LayerIndexInfo(const QString& name, int lidx, int vidx)
		: name(name), lidx(lidx), vidx(vidx) { }
};

QVector<LayerIndexInfo>
getLayerInfo(LayerView* lv)
{
	QVector<MapLayer*> layers = lv->getLayers();
	QVector<LayerIndexInfo> infos;

	foreach (MapLayer* l, layers) {
		int vidx = lv->visualIndexOfLayer(l);
		int lidx = lv->logicalIndexOfLayer(l);
		infos.push_back(LayerIndexInfo(l->name(), lidx, vidx));
		qDebug() << "name =  " << l->name() << "\n"
				 << "lidx =  " << lidx << "\n"
				 << "vidx =  " << vidx << "\n";
	}
	return infos;
}
void
restore_view_order(LayerView* lv, const QVector<LayerIndexInfo>& order)
{
	int maxVidx = order.size();
	for (int toVidx = 0; toVidx < maxVidx; ++toVidx) {
		const QString& name = order[toVidx].name;
		lv->moveSectionOfName(name, toVidx);
	}
}
bool
MapManager::XmlIF::processXml(const string& xml, bool ask_overwrite)
{
	using namespace NavData;

	istringstream is(xml);
	baseRTC::Header ihdr;

	if (!readXmlHeader(is, ihdr))
		return false;

	const struct {
		const char* const type;  
		MapLayer* (MapManager::XmlIF::*handler)(const baseRTC::Header& hdr, istream&);
	} TBL[] = {
		{ TaggedPoint::TYPE_ID,   &MapManager::XmlIF::mkDestinationLayer },
		{ Localization::TYPE_ID,  &MapManager::XmlIF::mkSelfPosLayer},
		{ Path::TYPE_ID,          &MapManager::XmlIF::mkPathLayer},
		{ PotentialGrid::TYPE_ID, &MapManager::XmlIF::mkPotentialLayer},
		{ GridMap::TYPE_ID,       &MapManager::XmlIF::mkGridMapLayer},
		{ CommandString::TYPE_ID, &MapManager::XmlIF::proc_CommandString},
	};
	static const size_t TBL_SIZE = sizeof(TBL) / sizeof (TBL[0]);

	for (size_t i = 0; i < TBL_SIZE; ++i) {
		if (ihdr.type == TBL[i].type) {
			MapLayer* layer = (this->*(TBL[i].handler))(ihdr, is);
			if (!layer) 
				return false;	// none modified
			
			layer->setXml(QString(xml.c_str()));
			if (ask_overwrite) {
				QString newname(layer->name());
				if (!mgr_->askNewLayerName(newname)) {
					delete layer;
					return false;
				}
				layer->setName(newname);
			}
			MapLayer* old = mgr_->layerOfName(layer->name());
			if (old) {
				layer->setVisible(old->isVisible());
				layer->setOpacity(old->opacity());
				layer->setZValue(old->zValue());
			}
			qDebug() << "before:";
			QVector<LayerIndexInfo> before = getLayerInfo(mgr_->layerView_);

			bool is_deleted = mgr_->removeLayerOfName(layer->name());
			mgr_->addMapLayer(layer);
			if (layer->name() == CURPOS_LAYER_NAME)
				have_cur_pos_ = true;
			mgr_->layerView_->updateLayers();
			qDebug() << "after:";
			QVector<LayerIndexInfo> after = getLayerInfo(mgr_->layerView_);

			if (is_deleted) {
				// this means we have the layer of same name.
				restore_view_order(mgr_->layerView_, before);
			}

			return true;
		}
	}
	cout << "This XML is not for me ... \n";
	return false;
}
void
MapManager::XmlIF::setPotentialRange(double pot_min, double pot_max)
{
	cout << "settting Potential Range " << pot_min << ", " << pot_max << "\n";
	pot_min_ = pot_min;
	pot_max_ = pot_max;
}
void
MapManager::XmlIF::invalidateCurpos()
{
	have_cur_pos_ = false;
}
bool
MapManager::XmlIF::getCurrentPos(QPointF& pos) const
{
	if (!have_cur_pos_) {
		return false;
	} else {
		pos = QPointF(curpos_.pos_x, curpos_.pos_y);
		return true;
	}
}
/* ---------------------------------------------------------------- *
 *   class MapManager
 * ---------------------------------------------------------------- */
const char MapManager::INIT_FILE_NAME[] = "init.xml";
const char MapManager::LAYER_ARCHIVE_FILENAME[] = "layers.xml";


MapManager::MapManager(MapView* mapView, LayerView* layerView) 
	: mapView_(mapView), layerView_(layerView),
	  autoscroll_timer_(new QTimer(this)),
	  mapViewObserver_(new WidgetObserver(this)),
	  layerViewObserver_(new WidgetObserver(this)),
	  xmlif_(new XmlIF(this))
{
	Q_ASSERT(mapView_ != NULL);
	Q_ASSERT(layerView_ != NULL);

	mapViewObserver_->setObservable(mapView_);
	connect(mapViewObserver_, SIGNAL(closing(QWidget*, bool&)),
			this, SLOT(on_closing(QWidget*, bool&)));

	layerViewObserver_->setObservable(layerView_);
	connect(layerViewObserver_, SIGNAL(closing(QWidget*, bool&)),
			this, SLOT(on_closing(QWidget*, bool&)));

	connect(layerView_, SIGNAL(layersChanged(const QVector<MapLayer*>&)),
			this, SLOT(on_layersChanged(const QVector<MapLayer*>&)));

	connect(layerView_, SIGNAL(removeLayerRequested(MapLayer*)),
			this, SLOT(on_removeLayerRequested(MapLayer*)));

	connect(layerView_, SIGNAL(newLayerRequested()),
			this, SLOT(openNewLayerFromFile()));
					   

	connect(mapView_, SIGNAL(destinationChanged(const QPointF&)),
			this, SLOT(on_destinationChanged(const QPointF&)));

	connect(mapView_, SIGNAL(locationChanged(const QPointF&, qreal)),
			this, SLOT(on_locationChanged(const QPointF&, qreal)));

	QTimer* poll_timer = new QTimer(this);
	poll_timer->start(QUEUE_POLLING_INTERVAL);
	connect(poll_timer, SIGNAL(timeout()), this, SLOT(try_dequeue()));

	autoscroll_timer_->setInterval(AUTOSCROLL_INTERVAL);
	connect(autoscroll_timer_, SIGNAL(timeout()), this, SLOT(do_autoscroll()));
	if (mapView_->isAutoScrolling())
		autoscroll_timer_->start();

	connect(mapView_, SIGNAL(autoscrollToggled(bool)), 
			this, SLOT(on_autoscrollToggled(bool)));
	

	layerView_->updateLayers();
}
MapManager::~MapManager()
{
}
int
MapManager::addMapLayer(MapLayer* layer)
{
	mapView_->addItem(layer);
	return layerView_->addMapLayer(layer);
}
MapLayer* 
MapManager::layerOfName(const QString& name)
{
	return layerView_->layerOfName(name);
}
bool
MapManager::removeLayerOfName(const QString& name)
{
	MapLayer* layer = layerView_->layerOfName(name);
	bool deleted = false;
	if (layer != NULL) {
		mapView_->removeItem(layer);
		layerView_->removeLayer(layer);
		delete layer;
		deleted = true;
	}

	if (layerView_->layerOfName(XmlIF::CURPOS_LAYER_NAME) == NULL) {
		xmlif_->invalidateCurpos();
	}
	//layerView_->updateLayers();
	return deleted;
}
void
MapManager::on_layersChanged(const QVector<MapLayer*>& layers)
{
	qreal zvalue = MapView::LAYER_ZVALUE_MAX;
	foreach (MapLayer* l, layers) {
		if (l == NULL)
			continue;
		qDebug() << (l->isVisible() ? " visible" : " invisible") << "\t"
				 << "opacity: " << 100.0*(l->opacity()) << "\t"
				 << l->name().toAscii().constData() << "\n";

		l->setZValue(zvalue);
		zvalue -= 1.0;
	}
	mapView_->updateScene();
	mapView_->update();
}
void
MapManager::on_removeLayerRequested(MapLayer* layer)
{
	FUNC_TRACE;
	layerView_->removeLayer(layer);
	mapView_->removeItem(layer);
	delete layer;
	if (layerView_->layerOfName(XmlIF::CURPOS_LAYER_NAME) == NULL) {
		xmlif_->invalidateCurpos();
	}
}
void
MapManager::on_destinationChanged(const QPointF& pos)
{
	xmlif_->emitDestination(pos);
}
void
MapManager::on_locationChanged(const QPointF& pos, qreal theta)
{
	xmlif_->emitLocation(pos, theta);
}
void
MapManager::setIOQueue(Queue_t* xmlq, Queue_t* resultq)
{
	xmlif_->setIOQueue(xmlq, resultq);
}
void
MapManager::try_dequeue()
{
	xmlif_->try_dequeue();
}
void
MapManager::setPotentialRange(double pot_min, double pot_max)
{
	xmlif_->setPotentialRange(pot_min, pot_max);
}
void
MapManager::setStartupDirectory(const QString& newdir)
{
	qDebug() << "new startup dir: " << newdir;
	startup_dir_ = QDir(newdir);
}

int
MapManager::loadGuiSettings()
{
	if (!QFileInfo(startup_dir_.path()).isDir()) {
		qDebug() << startup_dir_.path() << " is not a directory!";
		return -1;
	} else if (!startup_dir_.exists(INIT_FILE_NAME)) {
		qDebug() << startup_dir_.filePath(INIT_FILE_NAME) << " does not exists";
		return -2;
	}
	QString fname(startup_dir_.filePath(INIT_FILE_NAME));

	GuiSettings gs;
	if (gs.readFromFile(fname.toAscii().data()) < 0) {
		qDebug() << "Cannot read settings from " << fname;
		return -1;
	}

	layerView_->move(gs.layerview.position());
	layerView_->resize(gs.layerview.size());

	mapView_->move(gs.mapview.position());
	mapView_->resize(gs.mapview.size());
	return 0;

}
int
MapManager::saveGuiSettings()
{
	if (startup_dir_.exists() && 
		!QFileInfo(startup_dir_.path()).isDir()) 
	{
		// we have something the same name ....
		qDebug() << startup_dir_.path() << " is not a directory!";
		return -1;
	} 
	if (!startup_dir_.mkpath(".")) {
		qDebug() << "Cannot create path: " 
				 << startup_dir_.filePath(INIT_FILE_NAME);
		return -1;
	}

	qDebug() << "Saving Window settings ...";
	GuiSettings gs;
	gs.layerview.setPosition(layerViewObserver_->lastPosition());
	gs.layerview.setSize(layerViewObserver_->lastSize());
	gs.mapview.setPosition(mapViewObserver_->lastPosition());
	gs.mapview.setSize(mapViewObserver_->lastSize());
	const char* fname = startup_dir_.filePath(INIT_FILE_NAME).toAscii().data(); 
	if (gs.writeToFile(fname) < 0) {
		qDebug() << " failed!";
		return -1;
	} else {
		qDebug() << " done";
		return 0;
	}
}
void
MapManager::openNewLayerFromFile()
{
	qDebug() << __PRETTY_FUNCTION__;
	QStringList files = QFileDialog::getOpenFileNames(
		layerView_,
		"Select Map Layers",
		"./",
		"Layers (*.xml)");

	foreach (QString f, files) {
		ifstream ifs(f.toAscii().data());
		string s;
		// N. B. 贅沢なメモリ使用。
		s.assign(istreambuf_iterator<char>(ifs), istreambuf_iterator<char>());

		if (!xmlif_->processXml(s, true)) {
			QMessageBox::warning(layerView_, tr("MapViewer"),
								 tr("Cannot read layer from %1").arg(f));
		}
	}
}
void
MapManager::on_closing(QWidget* target, bool& accepted)
{
	Q_UNUSED(target);
	FUNC_TRACE;
	int ret = QMessageBox::warning(target, tr("MapViewer"),
								   tr("Really Close MapViewer?"),
								   QMessageBox::Close | QMessageBox::Default,
								   QMessageBox::Cancel);

	if (ret == QMessageBox::Close) {
		mapViewObserver_->disconnect();
		layerViewObserver_->disconnect();
		
		QTimer::singleShot(0, this, SLOT(closeAll()));

		accepted = true;
	} else {
		accepted = false;
	}
}
void
MapManager::closeAll()
{
	FUNC_TRACE;
	mapView_->close();
	layerView_->close();
}
void
MapManager::saveAllLayers() const
{
	FUNC_TRACE;

	const QString& fname = startup_dir_.filePath(LAYER_ARCHIVE_FILENAME);
	
	qDebug() << "Saving Layers to " << fname;
	LayerArchive la;
	foreach (MapLayer* l, layerView_->getLayers()) {
		qDebug() << "layer:" << l->name();
		la.add(l);
	}
	la.saveToFile(fname.toAscii().data());
	qDebug() << "done";
}
void
MapManager::loadAllLayers()
{
	FUNC_TRACE;
	const QString& fname = startup_dir_.filePath(LAYER_ARCHIVE_FILENAME);

	LayerArchive la;
	la.loadFromFile(fname.toAscii().data());

	foreach(const LayerArchive::Data& d, la.datas()) {
		qDebug() << "Loading ..." << d.name.c_str();
		if (!xmlif_->processXml(d.xml)) {
			QMessageBox::warning(layerView_, tr("MapViewer"),
								 tr("Cannot read layer: %1\n"
									"Broken Archive?")
								 .arg(d.name.c_str()));
		}
	}

}
void
MapManager::on_autoscrollToggled(bool autoscroll_is_on)
{
	FUNC_TRACE;
	
	if (autoscroll_is_on) {
		qDebug() << "ON!";
		autoscroll_timer_->start();
	} else {
		qDebug() << "OFF!";
		autoscroll_timer_->stop();
	}
}
void
MapManager::do_autoscroll()
{
	FUNC_TRACE;
	QPointF pos;

	if (xmlif_->getCurrentPos(pos)) {
		mapView_->centerOnGeo(pos);
	} else {
		qDebug() << "no current pos";
	}
}

bool
MapManager::mayOverwrite(const QString& layer_name)
{
	if (layerView_->layerOfName(layer_name) == NULL)
		return true;

	int ret = QMessageBox::warning(layerView_, tr("Duplicated Layer Name"),
								   tr("Really overwrite layer: %1 ?")
								   .arg(layer_name),
								   QMessageBox::Yes | QMessageBox::No);
	return (ret == QMessageBox::Yes);
}
bool
MapManager::askNewLayerName(QString& name)
{
	QString oldname(name);
	bool ok = true;
	QString newname;
	do {
		newname = QInputDialog::getText(layerView_, 
										tr("Input LayerName"), tr("LayerName:"),
										QLineEdit::Normal, oldname, &ok);
	} while (!mayOverwrite(newname));

	if (ok)
		name = newname;
	return ok;
}


/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
