/*!
 * @file map_grid.cpp
 * @brief Special QGraphicItem for displaying coordinate grid.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <QtGui>
#include <cmath>
#include "map_grid.h"

/* ================================================================ *
 *  utility functions in unnamed namespace
 * ================================================================ */
namespace /* unnamed */ {

/*!
 * @brief function for similar to pow(3) but cares about round-off error.
 * 
 * Code by Paul S. Heckbert. 
 * @see nicenum
 */
qreal
expt(qreal a, int n)
{
	qreal x;

	x = 1.;
	if (n>0) for (; n>0; n--) x *= a;
	else for (; n<0; n++) x /= a;
	return x;
}
/*!
 * @brief find a nice number for auto-calculating a grid interval.
 *  Algorithm from `Graphics Gems' (ISBN 0-12-286166-3)
 *  `Nice Numbers for Graph Labels' by Paul S. Heckbert
 */
qreal
nicenum(qreal x, int round)
{
	int expv;				/* exponent of x */
	qreal f;				/* fractional part of x */
	qreal nf;				/* nice, rounded fraction */

	expv = floor(log10(x));
	f = x/expt(10., expv);		/* between 1 and 10 */
	if (round)
		if (f<1.5) nf = 1.;
		else if (f<3.) nf = 2.;
		else if (f<7.) nf = 5.;
		else nf = 10.;
	else
		if (f<=1.) nf = 1.;
		else if (f<=2.) nf = 2.;
		else if (f<=5.) nf = 5.;
		else nf = 10.;
	return nf*expt(10., expv);
}
/*
 * 
 */
struct LabelInfo {
	bool valid;
	qreal gmin;
	qreal gmax;
	qreal spacing;
	int precision;          // # of fractional digits to show
	LabelInfo(qreal min, qreal max, int ntick);
};
LabelInfo::LabelInfo(qreal min, qreal max, int ntick)
{
	if (ntick < 1)
		ntick = 2;
	if (max < min)
		qSwap(min, max);

	valid = (max != min) && (ntick > 2);
	if (!valid) {
		gmin = gmax = spacing = 0.0;
		precision = 0;
		return;
	}
	qreal range  = nicenum(max - min, 0);
	spacing = nicenum(range/(ntick - 1), 1);
	gmin = floor(min/spacing)*spacing;
	gmax = ceil(max/spacing)*spacing;
	precision = qMax(-floor(log10(spacing)), 0.0);
}

} /* namespace unnamed */


MapGrid::MapGrid(const QRectF& extent)
	: gridExtent_(extent),
	  xGridSpacing_(NO_GRID),
	  xFineGridSpacing_(NO_GRID),
	  yGridSpacing_(NO_GRID),
	  yFineGridSpacing_(NO_GRID),
	  margin_(DEFAULT_MARGIN)
{
}
void MapGrid::setGridExtent(const QRectF& rect)
{
	gridExtent_ = rect; 
}
QRectF MapGrid::gridExtent() const
{
	return gridExtent_;
}
QRectF
MapGrid::boundingRect() const
{
#if 0
	return gridExtent_;
#else
	const qreal margin_w = gridExtent_.width() * margin_;
	const qreal margin_h = gridExtent_.height() * margin_;

	QPointF topLeft = gridExtent_.topLeft();
	QPointF bottomRight = gridExtent_.bottomRight();

	topLeft.rx() -= margin_w;
	topLeft.ry() -= margin_h;

	bottomRight.rx() += margin_w;
	bottomRight.ry() += margin_h;

//        qDebug() << "gridExtent   = " << gridExtent_;
//        qDebug() << " with margin = " << QRectF(topLeft, bottomRight);

	return QRectF(topLeft, bottomRight);
#endif
}
void
MapGrid::setMargin(qreal margin)
{
	if (margin < 0.0)
		margin_ = 0.0;
	margin_ = margin;
}
void
MapGrid::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
               QWidget *widget)
{
	Q_UNUSED(widget);

	if (!isVisible())
		return;

	const qreal& levelOfDetail = option->levelOfDetail;

	const QPen gridPen(Qt::blue, 0, Qt::SolidLine);
	const QPen fineGridPen(Qt::lightGray, 0, Qt::SolidLine);
	static const qreal gridOpacity    = 0.7;
	static const qreal fineGridOpacity = 0.2;
        
	drawGrid(painter, gridExtent_, 5.0*levelOfDetail,
			 fineGridPen, fineGridOpacity,
			 &xFineGridSpacing_, &yFineGridSpacing_);

	drawGrid(painter, gridExtent_, 0.5*levelOfDetail, 
			 gridPen, gridOpacity,
			 &xGridSpacing_, &yGridSpacing_);



	// BUG FIXME: 前の値を保存していないので，changedのタイミングで
	// はなくGridが決まるたびに毎回signalが飛ぶ。
	// 遅い以外に実害は無いが，できれば後で直す
	emit gridSpacingChanged(xGridSpacing_, xFineGridSpacing_,
							yGridSpacing_, yFineGridSpacing_);
}
void
MapGrid::setVisible(bool visible)
{
	QGraphicsItem::setVisible(visible);

	xGridSpacing_ = NO_GRID;
	yGridSpacing_ = NO_GRID;
	xFineGridSpacing_ = NO_GRID;
	yFineGridSpacing_ = NO_GRID;

	emit gridSpacingChanged(xGridSpacing_, xFineGridSpacing_,
							yGridSpacing_, yFineGridSpacing_);
}
void
MapGrid::drawGrid(QPainter* painter, const QRectF& rect, qreal level,
                  const QPen& pen, qreal opacity, 
                  qreal* xGridSpacing, qreal* yGridSpacing)
{
	// TODO: 後で整理

	painter->setPen(pen);
	painter->setOpacity(opacity);

	// FIXME: it is not efficient to calculate 2 LabelInfo's
	//        'cause `ntick' argument to the constructor is same...

	LabelInfo xinfo(rect.left(), rect.right(), nicenum(level, 1));
	if (xinfo.valid) {
		const qreal d = xinfo.spacing;
		const qreal xmin = qMax(xinfo.gmin,         rect.left());
		const qreal xmax = qMin(xinfo.gmax + 0.5*d, rect.right());
                
		for (qreal x = xmin; x < xmax; x += d) {
			painter->drawLine(x, rect.top(), 
							  x, rect.bottom());
		}
		*xGridSpacing = xinfo.spacing;
	} else {
		*xGridSpacing = NO_GRID;
	}


	LabelInfo yinfo(rect.top(), rect.bottom(), nicenum(level, 1));
	if (yinfo.valid) {
		const qreal d = yinfo.spacing;
		const qreal ymin = qMax(yinfo.gmin,         rect.top());
		const qreal ymax = qMin(yinfo.gmax + 0.5*d, rect.bottom());
                
		for (double y = ymin; y < ymax; y += d) {
			painter->drawLine(rect.left(),  y,
							  rect.right(), y);
		}
		*yGridSpacing = yinfo.spacing;
	}  else {
		*yGridSpacing = NO_GRID;
	}

}
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
