/*!
 * @file layer_view.h
 * @brief Widget for displaying informations of the layers
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#ifndef  LAYER_VIEW_HEADER_
#define  LAYER_VIEW_HEADER_

#include <QFrame>
#include "common_defs.h"

QT_FORWARD_DECLARE_CLASS(QTableView);
QT_FORWARD_DECLARE_CLASS(QModelIndex);
QT_FORWARD_DECLARE_CLASS(QAction);
QT_FORWARD_DECLARE_CLASS(QContextMenuEvent);

class MapLayer;
class MapLayerListModel;
class MapLayerDelegate;
class MapLayerTableView;

/*! 
 * @class LayerView
 * @brief 地図レイヤ情報表示Widget。地図レイヤのためのmodel, view,
 * delegateを管理する。ユーザ操作によるレイヤ変化をsignalとして通知
 */
class LayerView : public QFrame
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 * @param[in] parent 親widget
	 */
	LayerView(QWidget* parent = 0);

	/*! 地図レイヤの追加
	 * @param[in] layer 追加するlayer 
	 */
	int addMapLayer(MapLayer* layer);

	/*! 名称を指定してlayerを返す。
	 * @param[in] name レイヤの名称(name フィールドに対応)
	 * @retval non-null レイヤへのポインタ
	 * @retval NULL     見つからなかった場合
	 */
	MapLayer* layerOfName(const QString& name);
	// 2009/12/10 memo: viewとmodelの役目がうまく分かれていない気がする。
	// MapManagerとやり取りするのはLayerViewではなくmodelであるべきか？

	/*! レイヤのvisualIndexを返す
	 *  @param[in] l レイヤ
	 */
	int visualIndexOfLayer(const MapLayer* l);

	/*! レイヤのlogicalIndexを返す
	 *  @param[in] l レイヤ
	 */
	int logicalIndexOfLayer(const MapLayer* l);

	/*! レイヤをモデルから削除する。見つからなければなにもしない
	 *  @param[in] layer 削除するレイヤへのポインタ
	 */
	void removeLayer(MapLayer* layer);

	/*! レイヤのリストを取得
	 *  visualIndex順にソートされている。
	 */
	QVector<MapLayer*> getLayers() const;

	/*!
	 *
	 */
	void moveSectionOfName(const QString& name, int toVisualIndex);

public slots:
	//! レイヤ情報更新signalを強制emitするためのslot
	void updateLayers();
signals:
	/*! レイヤ情報更新時に発火するシグナル。レイヤー情報をベクタで通知
	 *  @param[in] layers 通知されるレイヤー情報
	 */
	void layersChanged(const QVector<MapLayer*>& layers);

	/*! GUIからlayer削除Actionがトリガされたときに発火。この時点では
	 * layerはdeleteされていない。
	 *  @param[in] layer 
	 */
	void removeLayerRequested(MapLayer *layer);

	/*! 「新規レイヤ」実行時に発火するシグナル。
	 * 
	 */
	void newLayerRequested();
protected:
	/*! レイヤ情報更新シグナルを発火する。(コード共通化のために存在)
	 */
	void emit_layersChanged();

	/*! 右クリックイベントのハンドラ
	 * @param[in] event ハンドルすべきイベント
	 */
	void contextMenuEvent(QContextMenuEvent* event);
private:
	/*! レイヤ情報を管理するmodel。詳細はQtのModel/View/Delegateフレー
	 *  ムワーク参照
	 */
	MapLayerListModel* model_;
	MapLayerTableView* view_; //!< レイヤー情報のview。
	MapLayerDelegate* delegate_; //!< レイヤー情報表示のためのdelegate

	QAction* upAction_;   //!< GUIのAction: 選択レイヤーを上に遷移
	QAction* downAction_;  //!< GUIのAction: 選択レイヤーを下に遷移
	QAction* saveAction_;  //!< GUIのAction: 選択レイヤーをセーブ
	QAction* deleteAction_; //!< GUIのAction: 選択レイヤーを削除
	QAction* openAction_;   //!< GUIのAction: 新規レイヤ
                              

private slots:
	//! model_からの通知用slot: indexがi0 からi1の範囲のデータ変更
	//!  @param[in] i0 範囲のはじまり
	//!  @param[in] i1 範囲の終わり
	void on_dataChanged(const QModelIndex& i0, const QModelIndex& i1);

	/*! view_からの通知用slot: 描画順の変更
	 *  @param[in] logicalIndex 移動対象となるItemのインデクス
	 *  @param[in] oldVisualIndex 移動前の場所
	 *  @param[in] newVisualIndex 移動後の場所
	 */
	void on_sectionMoved(int logicalIndex, int oldVisualIndex,
						 int newVisualIndex);

	//! @a view_ のpressイベントのハンドラ。デバグに使用
	void on_viewPressed(const QModelIndex &index);

	void saveSelected();	//!< 選択されたレイヤーを保存
	void deleteSelected();	//!< 選択されたレイヤーを削除
};




#endif /*LAYER_VIEW_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
