/*!
 *  @file gui_settings.h
 *  @brief class for saving/loading GUI configurations(pos, widht, height)
 *  @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#if !defined(GUI_SETTINGS_HEADER_)
#define      GUI_SETTINGS_HEADER_
#include <iosfwd>
#include <QSize>
#include <QPoint>

/*! @struct WindowInfo
 *  @brief ウィンドウの大きさ，位置を表現する構造体。ファイルへの保存に使用
 */
struct WindowInfo {
	int pos_x;					//!< windowの位置(左上)のx座標
	int pos_y;					//!< windowの位置(左上)のy座標
	int width;					//!< windowの幅 
	int height;					//!< windowの高さ

	//! デフォルトコンストラクタ。全て0となる
	WindowInfo() : pos_x(0), pos_y(0), width(0), height(0) { }

	//! 位置を QPointで返す。
	QPoint position() const { return QPoint(pos_x, pos_y); } 
	//! サイズを QSizeで返す。
	QSize  size() const { return QSize(width, height); }

	//! 場所をQPointで指定してセット
	void setPosition(const QPoint& pos)
	{
		pos_x = pos.x();
		pos_y = pos.y();
	}
	//! サイズをQSizeで指定してセット
	void setSize(const QSize& size)
	{
		width  = size.width();
		height = size.height();
	}
};

/*! @struct GuiSettings
 *  @brief MapView, LayerViewのWindowの位置／大きさをファイルに保存する
 *  ための構造体。MapManagerが使用する。読み書きはシリアライザを使用
 *
 */
struct GuiSettings {
	WindowInfo mapview;			//!> MapViewの位置，大きさ情報
	WindowInfo layerview;		//!> LayerViewの位置，大きさ情報

	//! デフォルトコンストラクタ
	GuiSettings() : mapview(), layerview() { }

	/*! ファイルから読み込んだ情報で自身を更新する
	 *  @param[in] fname ファイル名
	 *  @retval 0以上   成功
	 *  @retval negative 失敗
	 */
	int readFromFile(const char* fname); 

	/*! 自身の内容をファイルへ保存する。
	 *  @param[in] fname ファイル名
	 *  @retval 0以上   成功
	 *  @retval negative 失敗
	 */
	int writeToFile(const char* fname) const;

	
	/*! XMLが格納されたストリームを読み，自身を更新する。
	 *  @param[in] is 入力ストリーム
	 *  @retval 0以上   成功
	 *  @retval negative 失敗
	 */
	int readFromXml(std::istream& is);

	/*! ストリームに自身の情報をXMLとして書き出す
	 *  @param[in] os 出力ストリーム
	 *  @retval 0以上   成功
	 *  @retval negative 失敗
	 */
	int writeToXml(std::ostream& os) const;

	static const char TYPE_ID[]; //!< XMLタグに使用する型名
};




#endif   /*! GUI_SETTINGS_HEADER_ */
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
