/*!
 * @file service_if.h
 * @brief Interface to RGIS MapService and RouteService
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 */

#if !defined (SERVICE_IF_HEADER_)
#define       SERVICE_IF_HEADER_

#include "GlobalNavigationRTC.h"
#include "RgisOperator_simpleStub.h"
#include "GridMap.h"


namespace NavData {
struct PotentialGrid;

} /* namespace NavData */

struct Path;

/*! 
 * @class ServiceIF
 * @brief Interface of the RGIS MapService and RouteService.
 *
 * RGISとのインターフェースを提供する。状態を持ち，通信データ削減のために
 * データのCacheも行う。(XMLは理解しない。XMLの処理は上位 @a GNav が行う)
 */
class ServiceIF {
public:
	//! @typedef RTC::CorbaConsumer<RGIS::RTComp::MapService> MapService_t
	//! MapService_t RGISのMapServiceオブジェクトを表す型
	typedef RTC::CorbaConsumer<RGIS::RTComp::MapService> MapService_t;
	//! @typedef RTC::CorbaConsumer<RGIS::RTComp::RouteService> RouteService_t;
	//! RouteService_t RGISのRouteServiceオブジェクトを表す型	
	typedef RTC::CorbaConsumer<RGIS::RTComp::RouteService> RouteService_t;
	
	/*! constructor.
	 * @param[in] ms MapServicePortへの参照
	 * @param[in] rs RouteServicePortへの参照
	 */
	ServiceIF(MapService_t& ms, RouteService_t& rs);

	virtual ~ServiceIF() { }	//!< 仮想デストラクタ。何もしない

	void initialize();			//!< 初期状態に復帰

	/*! 目的地をセットする。ポテンシャルグリッドを出力.
	 * @param[in] dest 目的地
	 * @param[out] grid 出力される @a NavData::PotentialGrid
	 * @return 成功(true)／失敗(false)
	 */
	bool setDestination(const RGIS::PntC& dest, NavData::PotentialGrid& grid);

	/*! 現在地をセット 最短経路を出力.
	 *
	 * 現在値が過去にセットされたことがある場合，trueが返されるが，path
	 * は空の物が返される。
	 *
	 * @param[in] pos 現在地
	 * @param[out] path 出力される @a Path
	 * @return 成功(true)／失敗(false)
	 */
	bool setCurrentPosition(const CurPos& pos, Path& path);

	/*! 確率グリッドをセットし，内部に保存.
	 * @param[in] gm 保存する @a GridMap
	 * @return 成功(true)／失敗(false)
	 */
	bool storeProbabilityMap(const GridMap& gm);

	/*! PotentialGrid再計算を開始
	 * @param[out] grid 出力される @a PotentialGrid
	 * @return 成功(true)／失敗(false)
	 */
	bool recalcPotentialGrid(NavData::PotentialGrid& grid);

	/*! 目的地が設定されていれば，それを返す。
	 * 
	 * @param[out] dest 目的地があればここに格納される
	 * @return 目的地があればtrue(@a dest に格納)。なければfalse(@a dest は不変)
	 */
	bool getDestination(RGIS::PntC& dest) const;

	/*! 現在地が設定されていれば，それを返す.
	 * 
	 * @param[out] pos 現在地があればここに格納される
	 * @return 現在地があればtrue(@a pos に格納)。なければfalse(@a pos は不変)
	 */
	bool getCurrentPos(CurPos& pos) const;

	//! 障害グリッドとpotentialGridをマージする際の閾値を設定.
	//!  @param[in] th 新しい閾値。これ以上は障害物と見なす
	void setObstacleThreshold(double th);

	/*! 最短経路が設定されていればそれを返す.
	 * @param[out] path 出力される @a Path
	 * @return 経路があればtrue(@a path に格納)。なければfalse(@a path は不変)
	 */
	bool getCurrentPath(Path& path);

	/*! ポテンシャルグリッドの階調数をセット.
	 * @param[in] depth 階調数。bit数ではなく色の数であることに注意
	 */
	void setPotentialGridDepth(int depth);

private:
	/*! ポテンシャルグリッド出力.
	 *  @param[out] grid 出力される@a PotentialGrid
	 *  @return 成功／失敗
	 */
	bool emit_pgrid(NavData::PotentialGrid& grid);

	/*! 最短経路出力.
	 *  @param[out] path 出力される @a Path
	 *  @return 成功／失敗
	 */
	bool emit_path(Path& path);

	/*! 内部のポテンシャルグリッドcacheを更新
	 *  @return 成功／失敗
	 */
	bool update_pgrid();
	/*! RGISよりパスを取得.
	 *  @return 返されたパスデータ
	 */
	std::vector<RGIS::PntC> get_path();

	void merge_grid();			//!< ポテンシャルと障害グリッドをマージ
	
	MapService_t& ms_;			//!< @brief MapServicePortへの参照
	RouteService_t& rs_;		//!< @brief RouteServicePortへの参照

	RGIS::PntC dest_;			//!< 目的地
	RGIS::Geo::GridC pgrid_;		//!< ポテンシャルグリッド
	RGIS::Geo::GridC merged_grid_; //!< 障害物グリッドとマージされたpgrid
	CurPos current_pos_;		//!< 現在地
	RGIS::BoxC extent_;				//!< マップ存在範囲
	GridMap pmap_;				//!<確率グリッドマップ
	bool have_dest_;			//!<フラグ: 目的地はセットされているか
	bool have_currrent_pos_;	//!<フラグ: 現在地はセットされているか
	bool have_pgrid_;			//!<フラグ: ポテンシャルグリッドがあるか
	bool have_pmap_;			//!<フラグ: 確率グリッドはあるか

	size_t pgrid_depth_;		//!< ポテンシャルグリッドの値の数(bitではない)

	/*! 障害グリッドとポテンシャルグリッドを重ねるときにこの値以上は障害物
	 * と見なす
	 */
	double obstacle_thresh_;	
};


#endif     /*!SERVICE_IF_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */

