/*!
 * @file config_data.h
 * @brief Holder for the configuration variables.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */


#if !defined(CONFIG_DATA_HEADER_)
#define      CONFIG_DATA_HEADER_

#include "GlobalNavigationRTC.h"
#include "cur_pos.h"

/*! @class ConfigData
 *  @brief Holder for the configuration variables.
 *
 *  設定データを表すクラス。ファイルからの読み込み，値の保持，
 *  書き出しを受け持つ。
 */
class ConfigData {
public:
	ConfigData();			//!< コンストラクタ。各値をdefault値にする

	/*! ファイルからの読み込み
	 *  @param[in] filename ファイル名
	 *  @return 成功したかどうか
	 */
	bool readFromFile(const std::string& filename);
	/*! ファイルへの保存
	 *  @param[in] filename ファイル名
	 *  @return 成功したかどうか
	 */
	bool saveToFile(const std::string& filename);

	//! 入力ポートの数を返す
	int numberOfInPorts() const { return n_inports_; }
	//! 障害物判定閾値を返す
	double obstacleThreshold() const { return obs_thresh_; }

	/*! 目的地が有れば返す
	 *  @param [out] dest 返される目的地
	 *  @retval true  目的地が有った(destに返された)
	 *  @retval false 目的地が無かった(destは不変)
	 */
	bool getDestination(RGIS::PntC & dest) const;

	/*! 目的地を設定する。
	 *  @param [in] dest 設定する目的地
	 */
	void setDestination(const RGIS::PntC & dest);


	/*! 現在地が有れば返す
	 *  @param [out] curpos 返される目的地
	 *  @retval true  現在地が有った(curposに返された)
	 *  @retval false 現在地が無かった(curposは不変)
	 */
	bool getCurrentPos(CurPos & curpos) const;

	/*! 現在地を設定する。
	 *  @param [in] curpos 設定する現在地
	 */
	void setCurrentPos(const CurPos& curpos);

	/*! ポテンシャルグリッドの階調数を返す
	 *  @retval ポテンシャルグリッドの階調数
	 */
	int  getPotentialGridDepth() const { return pgrid_depth_; }

	/*! 起動時に設定ファイル内部に保存された現在位置を送信するか
	 *  @retval 送信するかどうかのboolean
	 */
	bool useSavedCurpos() const { return have_curpos_ && use_saved_curpos_; }
private:
	int n_inports_;				//!< 入力ポートの数
	double obs_thresh_;			//!< 障害物計算の閾値，これ以上は障害物とする
	RGIS::PntC dest_;			//!< 目的地
	bool have_dest_;			//!< フラグ。目的地はあるか？
	CurPos curpos_;				//!< 現在地
	bool have_curpos_;			//!< フラグ。目的地はあるか？
	int pgrid_depth_;			//!< ポテンシャルグリッドの階調数(bit数ではない)
	bool use_saved_curpos_;		//!< 起動時，設定ファイル内部の現在位置を送信するか？

	// TAG_NAMES
	static const char INPUT_NUMBER[]; //!< 入力ポート用XMLタグ名 
	static const char DESTINATION[];  //!< 目的地用XMLタグ名 
	static const char OBS_THRESHOLD[]; //!< 障害値計算閾値用XMLタグ名 
	static const char CURRENT_POS[];   //!< 現在地用XMLタグ名 
	static const char PGRID_DEPTH[];   //!< ポテンシャルグリッド階調用XMLタグ名 
	static const char USE_SAVED_CURPOS[]; //!< 起動時に設定ファイルの現在地を送信するかどうかの XMLタグ名 
};

#endif   /*! CONFIG_DATA_HEADER_ */
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
