/*!
 * @file config_data.cpp
 * @brief Holder for the configuration variables.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#include <iostream>
#include <vector>
#include <fstream>
#include <string>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>

#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/util/PlatformUtils.hpp>

#include "XmlHandler.h"
//#include "base64.h"
#include "debug_utils.h"

#include "config_data.h"

using namespace std;
using namespace boost;

namespace /*unnamed  */{
const double PI = 3.14159265358979323846264338327950288;
double
deg2rad(double deg)
{
	return PI * deg / 180.0;
}
double
rad2deg(double rad)
{
	return 180.0 * rad / PI;
}

class XmlParserRAII {
public:
	XmlParserRAII(const string& filename) : 
		filename_(filename)
	{
		InitXmlParser();
	}
	~XmlParserRAII() {
		TerminateXmlParser();
	}
	// rtc_config を探していないので，ルーズなパーサ

	template <typename T>
	bool
	find_entry(const string& key, const string& attr, T& result)
	{
		if (!XmlParser(filename_, key, attr, &strs_)) {
			cerr << "cannot find " << key << " in " << filename_ << endl;
			return false;
		} else if (strs_.empty()) {
			cerr << "empty strs" << endl;
			return false;
		}

		try {
			result = lexical_cast<T>(strs_[0]);
		} catch (const bad_lexical_cast& e) {
			cerr << "Cannot parse " << strs_[0]
				 <<  " for " << key
				 << " in file: " << filename_ << endl;
			return false;
		}
		return true;
	}
private:
	vector<string> strs_;
	string filename_;
};

template <typename T>
size_t
readSequence(const string& str, vector<T>& values, const char* delim=", \t")
{
	vector<string> v;
	split(v, str, is_any_of(delim), token_compress_on);
	size_t n_read = 0;
	try {
		for (size_t i = 0, imax = v.size(); i < imax; ++i) {
			values.push_back(lexical_cast<T>(v[i]));
			++n_read;
		}
	} catch (const bad_lexical_cast& e) {
		cout << "parse error while parsing : " << str << "\n";
	}
	return n_read;
}

} /* unnamed namespace */

ConfigData::ConfigData()
	: n_inports_(1),
	  obs_thresh_(0.5),
	  have_dest_(false),
	  have_curpos_(false),
	  pgrid_depth_(256)
{
}
const char ConfigData::INPUT_NUMBER[] = "input_number";
const char ConfigData::DESTINATION[] = "destination";
const char ConfigData::OBS_THRESHOLD[] = "obs_threshold";
const char ConfigData::CURRENT_POS[] = "current_pos";
const char ConfigData::PGRID_DEPTH[] = "pgrid_depth";
const char ConfigData::USE_SAVED_CURPOS[] = "use_saved_curpos";

bool
ConfigData::readFromFile(const string& filename)
{
	bool all_ok = true;
	XmlParserRAII parser(filename);

	if (!parser.find_entry(INPUT_NUMBER, "", n_inports_))
		all_ok = false;

	string tmp;
	if (parser.find_entry(DESTINATION, "", tmp)) {
		vector<double> v;
		if (readSequence(tmp, v) != 2) {
			cout << "cannot read destination\n";
			have_dest_ = false;
		} else {
			dest_.x = v[0];
			dest_.y = v[1];
			have_dest_ = true;
			cout << "Dest: = (" << dest_.x << ", " << dest_.y << ")\n";
		}
	} else {
		cout << "no destinatioin in " << filename << "\n";
		all_ok = false;
	}

	if (parser.find_entry(CURRENT_POS, "", tmp)) {
		vector<double> v;
		if (readSequence(tmp, v) != 3) {
			cout << "cannot read current position\n";
			have_curpos_ = false;
		} else {
			curpos_.pos.x = v[0];
			curpos_.pos.y = v[1];
			curpos_.theta = v[2];
			have_curpos_ = true;
			cout << "curpos: = (" 
				 << curpos_.pos.x << ", " 
				 << curpos_.pos.y 
				 << ")\n";
			cout << "theta = " << curpos_.theta << "[rad] = " 
				 << rad2deg(curpos_.theta) << "[deg]\n";
		}
	} else {
		cout << "no current position in " << filename << "\n";
		all_ok = false;
	}

	int pgrid_depth;
	if (!parser.find_entry(PGRID_DEPTH, "", pgrid_depth)) {
		cout << "pgrid_depth was not found\n";
		all_ok = false;
	} else if (pgrid_depth < 2 || 65536 < pgrid_depth) {
		all_ok = false;
	} else {
		pgrid_depth_ = pgrid_depth;
		cout << "depth Potential Grid: " << pgrid_depth_ << "\n";
	}

	if (!parser.find_entry(OBS_THRESHOLD, "", obs_thresh_)) {
		cout << OBS_THRESHOLD << " was not found\n";
		all_ok = false;
	} else if (obs_thresh_ < 0.0) {
		obs_thresh_ = 0.0;
		cout << "obs_threshold must be between 0.0 and 1.0\n";
		all_ok = false;
	} else if (obs_thresh_ > 1.0) {
		obs_thresh_ = 1.0;
		cout << "obs_threshold must be between 0.0 and 1.0\n";
		all_ok = false;
	} else {
		cout << "Obstacle Threshold: " << obs_thresh_ << "\n";
	}

	if (!parser.find_entry(USE_SAVED_CURPOS, "", use_saved_curpos_)) {
		cout << USE_SAVED_CURPOS << " was not found\n";
		all_ok = false;
	} else {
		cout << "Use saved current position = " << use_saved_curpos_ << "\n";
	}
	

	return all_ok;
}

bool
ConfigData::getDestination(RGIS::PntC & dest) const
{
	if (have_dest_) {
		dest = dest_;
		return true;
	} else 
		return false;
}
void
ConfigData::setDestination(const RGIS::PntC & dest)
{
	dest_ = dest;
	have_dest_ = true;
}
bool
ConfigData::getCurrentPos(CurPos& curpos) const
{
	if (have_curpos_) {
		curpos = curpos_;
		return true;
	} else 
		return false;
}
void
ConfigData::setCurrentPos(const CurPos& curpos)
{
	curpos_ = curpos;
	have_curpos_ = true;
}

namespace /* unnamed */ {
string otag(const char* tag)
{
	ostringstream os;
	os << "<" << tag << ">";
	return os.str();
}
string etag(const char* tag)
{
	ostringstream os;
	os << "</" << tag << ">";
	return os.str();
}
} /* unnamed namespace */
bool
ConfigData::saveToFile(const std::string& filename)
{
	FUNC_TRACE;

	ostringstream os;
	ofstream ofs(filename.c_str());
	
	ofs << "<?xml version=\"1.0\"?>\n";
	ofs << "<rtc_config category=\"Navigation\" name=\"GlobalNavigation\">\n";

	ofs <<"  "
		<< otag(INPUT_NUMBER) 
		<< n_inports_
		<< etag(INPUT_NUMBER) << "\n";

	ofs <<"  "
		<< otag(OBS_THRESHOLD) 
		<< obs_thresh_
		<< etag(OBS_THRESHOLD) << "\n";

	ofs <<"  "
		<< otag(DESTINATION) 
		<< dest_.x << "," << dest_.y
		<< etag(DESTINATION)  << "\n";

	ofs <<"  "
		<< otag(CURRENT_POS) 
		<< curpos_.pos.x << "," << curpos_.pos.y << "," << curpos_.theta
		<< etag(CURRENT_POS)  << "\n";
	
	ofs <<"  "
		<< otag(PGRID_DEPTH)
		<< pgrid_depth_
		<< etag(PGRID_DEPTH) << "\n";

	ofs <<"  "
		<< otag(USE_SAVED_CURPOS)
		<< use_saved_curpos_
		<< etag(USE_SAVED_CURPOS) << "\n";


	ofs << "</rtc_config>\n";

	return true;
}
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
