// -*- C++ -*-
/*!
 * @file  GlobalNavigationRTC.cpp 
 * @brief RT component class for Global Navigation 
 */
#include <sys/time.h>
#include <algorithm>
#include <iostream>
#include <queue>
#include <sstream>
#include <iterator>
#include <string>
#include <vector>

using namespace std;

#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>

#include "GlobalNavigationRTC.h"
#include "debug_utils.h"
#include "gnav.h"
#include "baseRTC.h"
#include "navigation_data.h"
#include "Localization.h"
#include "Path.h"
#include "config_data.h"

namespace /* unnamed */ {

//! 現在時刻をdoubleとして返す。unix通算秒を実数にしたもの
double
now()
{
	struct timeval tv;
	struct timezone dummy;
	gettimeofday(&tv, &dummy);
	return double(tv.tv_sec) + tv.tv_usec * 1.0e-6;
}
/*! 現在時刻をval(OpenRTMのTimedXXを想定)に設定
 * @tparam TimedValue tm.sec tm.nsecメンバを持つ型
 * @param val TimedValue型オブジェクトへの参照
 */
template <class TimedValue>
void
setTimeStamp(TimedValue& val)
{
	struct timeval tv;
	struct timezone dummy;
	gettimeofday(&tv, &dummy);

	val.tm.sec  = tv.tv_sec;
	val.tm.nsec = tv.tv_usec * 1000;
}

//! 設定ファイル名の初期値
const char DEFAULT_CONF[] = "./GlobalNavigation.xml";

} /* unnamed namespace */



//! このコンポーネントの情報
static const char* GlobalNavigatioRTC_spec[] =
{
	"implementation_id", "GlobalNavigationRTC",
	"type_name",         "GlobalNavigationRTC",
	"description",       "GlobalNavigationRTC",
	"version",           "1.0.0",
	"vendor",            "fuRo at Chiba Institute of Technology",
	"category",          "Navigation",
	"activity_type",     "SPORADIC",
	"kind",              "DataFlowComponent",
	"max_instance",      "1",
	"language",          "C++",
	"lang_type",         "compile",
	// Configuration variables
	"conf.default.filename", DEFAULT_CONF,
	""
};
// </rtc-template>

GlobalNavigationRTC::GlobalNavigationRTC(RTC::Manager* manager)
    // <rtc-template block="initializer">
	: RTC::DataFlowComponentBase(manager),
	  m_outOut("out", m_out),
	  m_mapServicePort("MapService"),
	  m_routeServicePort("RouteService"),
	  gnav_(new GNav(m_mapService, m_routeService)),
	  config_(new ConfigData()),
	  initcmd_sent_(false)
	  // </rtc-template>
{
	// Registration: InPort/OutPort/Service
	// <rtc-template block="registration">
	// Set InPort buffers

	// Set OutPort buffer
	registerOutPort("out", m_outOut);

	// Set service provider to Ports

	// Set service consumers to Ports
	m_mapServicePort.registerConsumer("MapService", "MapService", m_mapService);
	m_routeServicePort.registerConsumer("RouteService", "RouteService", m_routeService);

	// Set CORBA Service Ports
	registerPort(m_mapServicePort);
	registerPort(m_routeServicePort);

	// </rtc-template>

}
GlobalNavigationRTC::~GlobalNavigationRTC()
{
}

/*!
 *  @details
 *  初期化時のフックルーチン。以下の処理を行う.
 *
 *  <ul>
 *    <li> コンフィグパラメータのバインド</li>
 *    <li> コンフィグ情報をファイルから読む</li>
 *    <li> 入力ポート数の設定</li>
 *    <li> @a GNav (バックエンド)の初期化</li>
 *  </ul>
 */
RTC::ReturnCode_t GlobalNavigationRTC::onInitialize()
{
	// <rtc-template block="bind_config">
	// Bind variables and configuration variable

	bindParameter("filename", filename, DEFAULT_CONF);

	config_->readFromFile(filename);

	int n_ports = config_->numberOfInPorts();
	cout << "Config: n_ports = " << n_ports << "\n";

	typedef boost::shared_ptr<TimedString> StringHolder_t;
	typedef boost::shared_ptr<InPort<TimedString> > PortHolder_t;
	for (int i = 0; i < n_ports; ++i) {
		char portname[16];
		snprintf(portname, sizeof portname, "in%d", i);

		m_ins.push_back(StringHolder_t(new TimedString()));
		PortHolder_t p(new InPort<TimedString>(portname, *m_ins[i]));
		m_inIns.push_back(p);

		registerInPort(portname, *(m_inIns[i].get()));
	}
	gnav_->readConfigFrom(*config_);
	gnav_->initialize();


	// </rtc-template>
	return RTC::RTC_OK;
}
/*!
 *  @details
 *  アクティベート時のフックルーチン
 *
 *  <ul>
 *    <li> コンフィグ情報をファイルから読む</li>
 *    <li> @a GNav (バックエンド)の初期化</li>
 *  </ul>
 */
RTC::ReturnCode_t GlobalNavigationRTC::onActivated(RTC::UniqueId ec_id)
{
	FUNC_TRACE;
	initcmd_sent_ = false;
	gnav_->initialize();
	config_->readFromFile(filename);// ここでも読むべきか？

	return RTC::RTC_OK;
}
/*!
 *  @details
 *  ディアクティベート時のフックルーチン
 *  <ul>
 *    <li> 出力queueをフラッシュ </li>
 *  </ul>
 */
RTC::ReturnCode_t GlobalNavigationRTC::onDeactivated(RTC::UniqueId ec_id)
{
	flushQueue();

	RGIS::PntC dest;
	if (gnav_->getDestination(dest)) {
		config_->setDestination(dest);
	}
	CurPos curpos;
	if (gnav_->getCurrentPos(curpos)) {
		config_->setCurrentPos(curpos);
	}

	config_->saveToFile(filename);
	initcmd_sent_ = false;

	return RTC::RTC_OK;
}
void GlobalNavigationRTC::flushQueue()
{
	while (!resultq_.empty()) {
		setTimeStamp(m_out);
		const string& result = resultq_.front();
		m_out.data = result.c_str();
		m_outOut.write();
		cout << "Sent:  "       << result << "\n"
			 << "TimeStamp: "   << m_out.tm.sec << "[s] "
			 << m_out.tm.nsec << "[ns]" << endl;

		resultq_.pop(); // ここでresultは無効になる
	}
}
/*!
 *  @details
 *  アクティブ時，周期的にコールされるルーチン
 *  <ol>
 *    <li> activate後最初のコールであれば，目的地を送信</li>
 *    <li> activate後最初のコールであって，
 *         初期設定ファイルのuse_cusaved_curposが真であれば，現在地を送信</li>
 *    <li> 全ての入力ポートをサーチし，入力があれば，@a gnav_に処理させる</li>
 *    <li> gnav_の出力はqueueに入るので，内容を出力ポートに出力する</li>
 *  </ol>
 */
RTC::ReturnCode_t GlobalNavigationRTC::onExecute(RTC::UniqueId ec_id)
{
	(void)ec_id;

	if (!initcmd_sent_) {
		if (!is_nil(m_mapService.getObject()) && 
			!is_nil(m_routeService.getObject())) 
		{
		
			RGIS::PntC dest;
			if (config_->getDestination(dest)) {
				gnav_->setDestination(dest, resultq_);
			}
			CurPos curpos;
			if (config_->useSavedCurpos()) {
				gnav_->setCurrentPos(curpos, resultq_);		
			}
		}
		initcmd_sent_ = true;
	}
	for (size_t i=0, imax=m_inIns.size(); i < imax; ++i) {
		if (m_inIns[i]->isNew()) {
			m_inIns[i]->read();
			string recv(m_ins[i]->data);

			cout << "On Input Port No. " << i << "\n"
				 << "Received:  "   << recv << "\n"
				 << "TimeStamp: "   << m_ins[i]->tm.sec << "[s] "
				 << m_ins[i]->tm.nsec << "[ns]" << endl;

			if (!gnav_->processXml(recv, resultq_)) {
				cout << "Error while processing input" << endl;
			}
		}
	}
	flushQueue();
	return RTC::RTC_OK;
}
extern "C"
{
	/*! このコンポーネントの初期化メソッド.
	 *  RTC::Managerにこのコンポーネントの情報，生成／破壊関数を登録する。
	 *  @param[in] manager managerへのポインタ
	 */
	void GlobalNavigationRTCInit(RTC::Manager* manager)
	{
		RTC::Properties profile(GlobalNavigatioRTC_spec);
		manager->registerFactory(profile,
								 RTC::Create<GlobalNavigationRTC>,
								 RTC::Delete<GlobalNavigationRTC>);
	}
  
};



/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
