/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.node;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import jp.sourceforge.expression_computer.CompileContext;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.command.FunctionCallCommand;

/**
 * <p>
 * FunctionExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class FunctionExpressionNode extends AbstractNode implements OperandNode {

    private FunctionExpressionNode.FunctionNameNode functionName;

    private FunctionExpressionNode.LeftBracketNode  leftBracket;

    private OperandNode[]                           arguments;

    private FunctionExpressionNode.CommaNode[]      argumentDelimiters;

    private FunctionExpressionNode.RightBracketNode rightBracket;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param functionName
     *            関数名。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。<br>
     *            識別子の形式ではない場合、{@link IllegalArgumentException}例外をスローします。
     * @param arguments
     *            引数の配列。<br>
     *            nullの場合、配列にnullがある場合、{@link NullPointerException}例外をスローします。
     */
    public FunctionExpressionNode(FunctionExpressionNode.FunctionNameNode functionName, OperandNode[] arguments) {
        if (functionName == null) {
            throw new NullPointerException("functionNameがnullです。");
        }
        if (arguments == null) {
            throw new NullPointerException("argumentsがnullです。");
        }
        for (int i = 0; i < arguments.length; i++) {
            if (arguments[i] == null) {
                throw new NullPointerException("arguments[" + i + "]がnullです。");
            }
        }
        this.functionName = functionName;
        this.leftBracket = new FunctionExpressionNode.LeftBracketNode();
        this.arguments = arguments;
        if (this.arguments.length > 0) {
            this.argumentDelimiters = new FunctionExpressionNode.CommaNode[this.arguments.length - 1];
            for (int i = 0; i < this.argumentDelimiters.length; i++) {
                this.argumentDelimiters[i] = new FunctionExpressionNode.CommaNode();
            }
        }
        this.rightBracket = new FunctionExpressionNode.RightBracketNode();
    }

    /**
     * {@inheritDoc}
     */
    public void compile(CompileContext context) {
        for (int i = 0; i < this.arguments.length; i++) {
            this.arguments[i].compile(context);
        }
        context.getCommandList().add(new FunctionCallCommand(this.functionName.getName(), this.arguments.length));
    }

    /**
     * {@inheritDoc}
     */
    public Node[] getChildren() {
        List nodeList = new ArrayList();
        nodeList.add(this.functionName);
        nodeList.add(this.leftBracket);
        for (int i = 0; i < this.arguments.length; i++) {
            if (i == 0) {
                nodeList.add(this.arguments[i]);
            } else {
                nodeList.add(this.argumentDelimiters[i - 1]);
                nodeList.add(this.arguments[i]);
            }
        }
        nodeList.add(this.rightBracket);
        return (Node[]) nodeList.toArray(new Node[0]);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + this.toChildrenString();
    }

    /**
     * <p>
     * FunctionExpressionの関数名を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class FunctionNameNode extends AbstractNode {

        private String name;

        /**
         * <p>
         * インスタンスを初期化します。
         * </p>
         * 
         * @param name
         *            関数名。nullの場合、{@link NullPointerException}例外をスローします。識別子の形式ではない場合、{@link IllegalArgumentException}例外をスローします。
         */
        public FunctionNameNode(String name) {
            if (name == null) {
                throw new NullPointerException("nameがnullです。");
            }
            if (!Pattern.matches("[\\u0024\\u0041-\\u005a\\u005f\\u0061-\\u007a\\u00c0-\\u00d6\\u00d8-\\u00f6\\u00f8-\\u00ff\\u0100-\\u1fff\\u3040-\\u318f\\u3300-\\u337f\\u3400-\\u3d2d\\u4e00-\\u9fff\\uf900-\\ufaff]([\\u0024\\u0041-\\u005a\\u005f\\u0061-\\u007a\\u00c0-\\u00d6\\u00d8-\\u00f6\\u00f8-\\u00ff\\u0100-\\u1fff\\u3040-\\u318f\\u3300-\\u337f\\u3400-\\u3d2d\\u4e00-\\u9fff\\uf900-\\ufaff]|[\\u0030-\\u0039\\u0660-\\u0669\\u06f0-\\u06f9\\u0966-\\u096f\\u09e6-\\u09ef\\u0a66-\\u0a6f\\u0ae6-\\u0aef\\u0b66-\\u0b6f\\u0be7-\\u0bef\\u0c66-\\u0c6f\\u0ce6-\\u0cef\\u0d66-\\u0d6f\\u0e50-\\u0e59\\u0ed0-\\u0ed9\\u1040-\\u1049])*", name)) {
                throw new IllegalArgumentException("nameが識別子の形式ではありません。");
            }
            this.name = name;
        }

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public Node[] getChildren() {
            return new Node[0];
        }

        /**
         * <p>
         * 関数名を返します。
         * </p>
         * 
         * @return 関数名。
         */
        public String getName() {
            return this.name;
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName() + "[" + this.name + "]";
        }

    }

    /**
     * <p>
     * FunctionExpressionの"("演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class LeftBracketNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public Node[] getChildren() {
            return new Node[0];
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * FunctionExpressionの")"演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class RightBracketNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public Node[] getChildren() {
            return new Node[0];
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * FunctionExpressionの","演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class CommaNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public Node[] getChildren() {
            return new Node[0];
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

}
