/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

import java.util.Iterator;

/**
 * <p>
 * 数式を解析し、計算を行います。数式は、実際はコマンドに分解され、それをスタックマシンとして実行します。
 * </p>
 * <p>
 * 使用する変数、関数は、{@link #compute(CommandList)}, {@link #compute(String)}メソッドを呼び出す前に定義しておいてください。定義されていない場合、処理中に{@link ComputeException}例外がスローされます。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class Computer {

    private ComputeContext context = new ComputeContext();

    /**
     * <p>
     * 変数に値を設定します。変数を宣言することにもなります。
     * </p>
     * 
     * @param name
     *            変数名。<br>
     *            nullの場合、{@link ComputeException}例外をスローします。
     * @param value
     *            設定する値。
     */
    public void setVariable(String name, double value) {
        this.context.setVariable(name, value);
    }

    /**
     * <p>
     * 変数の値を取得します。
     * </p>
     * 
     * @param name
     *            変数名。<br>
     *            指定した名前で変数が宣言されていない場合、{@link ComputeException}例外をスローします。
     * @return 変数の値。
     */
    public double getVariable(String name) {
        return this.context.getVariable(name);
    }

    /**
     * <p>
     * 宣言されている変数の名前の配列を返します。
     * </p>
     * 
     * @return 宣言されている変数の名前の配列。
     */
    public String[] getVariableNames() {
        return this.context.getVariableNames();
    }

    /**
     * <p>
     * 呼び出し可能な関数を追加します。この呼び出しは、関数を定義することと同義です。
     * </p>
     * 
     * @param name
     *            関数名。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param func
     *            呼び出される関数。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     */
    public void addFunction(String name, Function func) {
        this.context.addFunction(name, func);
    }

    /**
     * <p>
     * 計算を行います。
     * </p>
     * 
     * @param expression
     *            通常の数式を表す文字列。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。<br>
     *            構文解析に失敗した場合、{@link ParseException}例外をスローします。
     * @return 計算結果の値。
     */
    public double compute(String expression) {
        Parser parser = new Parser();
        Node node = parser.parse(expression);

        Compiler compiler = new Compiler();
        CommandList commandList = compiler.compile(node);

        return this.compute(commandList);
    }

    /**
     * <p>
     * 計算を行います。
     * </p>
     * 
     * @param commandList
     *            コマンド列。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @return 計算結果の値。
     */
    public double compute(CommandList commandList) {
        if (commandList == null) {
            throw new NullPointerException("commandListがnullです。");
        }

        for (Iterator i = commandList.iterator(); i.hasNext();) {
            Command c = (Command) i.next();
            c.execute(this.context);
        }

        if (this.context.sizeStack() == 0) {
            throw new ComputeException("計算終了後にスタックに結果の値が追加されていません。");
        }
        if (this.context.sizeStack() > 1) {
            throw new ComputeException("計算終了後にスタックに追加されている数が2つ以上あります。");
        }
        double resultValue = this.context.popStack();
        return resultValue;
    }

}
