/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

/**
 * <p>
 * 計算中の状態を保持するコンテキストです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class ComputeContext {

    private Map   variableMap = new HashMap();

    private Map   functionMap = new HashMap();

    private Stack stack       = new Stack();

    /**
     * <p>
     * スタックに格納されている要素の数を返します。
     * </p>
     * 
     * @return スタックに格納されている要素の数。
     */
    public int sizeStack() {
        return this.stack.size();
    }

    /**
     * <p>
     * スタックから値を取り出します。取り出した値は、スタックから削除されます。スタックが空の場合、{@link ComputeException}例外をスローします。
     * </p>
     * 
     * @return スタックから取り出した値。
     */
    public double popStack() {
        if (this.stack.empty()) {
            throw new ComputeException("スタックが空です。");
        }
        return ((Double) this.stack.pop()).doubleValue();
    }

    /**
     * <p>
     * スタックから値を取り出します。取り出した値は、スタックから削除されません。スタックが空の場合、{@link ComputeException}例外をスローします。
     * </p>
     * 
     * @return スタックから取り出した値。
     */
    public double peekStack() {
        if (this.stack.empty()) {
            throw new ComputeException("スタックが空です。");
        }
        return ((Double) this.stack.peek()).doubleValue();
    }

    /**
     * <p>
     * スタックに値を追加します。
     * </p>
     * 
     * @param value
     *            スタックに追加する値。
     */
    public void pushStack(double value) {
        this.stack.push(new Double(value));
    }

    /**
     * <p>
     * 変数に値を設定します。変数を宣言することにもなります。
     * </p>
     * 
     * @param name
     *            変数名。<br>
     *            nullの場合、{@link ComputeException}例外をスローします。
     * @param value
     *            設定する値。
     */
    public void setVariable(String name, double value) {
        if (name == null) {
            throw new NullPointerException("nameがnullです。");
        }
        this.variableMap.put(name, new Double(value));
    }

    /**
     * <p>
     * 変数の値を取得します。
     * </p>
     * 
     * @param name
     *            変数名。<br>
     *            指定した名前で変数が宣言されていない場合、{@link ComputeException}例外をスローします。
     * @return 変数の値。
     */
    public double getVariable(String name) {
        if (!this.variableMap.containsKey(name)) {
            throw new ComputeException("変数{" + name + "}は宣言されていません。");
        }
        double value = ((Double) this.variableMap.get(name)).doubleValue();
        return value;
    }

    /**
     * <p>
     * 宣言されている変数の名前の配列を返します。
     * </p>
     * 
     * @return 宣言されている変数の名前の配列。
     */
    public String[] getVariableNames() {
        String[] variableNames = (String[]) this.variableMap.keySet().toArray(new String[0]);
        return variableNames;
    }

    /**
     * <p>
     * 呼び出し可能な関数を追加します。この呼び出しは、関数を定義することと同義です。
     * </p>
     * 
     * @param name
     *            関数名。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param func
     *            呼び出される関数。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     */
    public void addFunction(String name, Function func) {
        if (name == null) {
            throw new NullPointerException("nameがnullです。");
        }
        if (func == null) {
            throw new NullPointerException("funcがnullです。");
        }
        this.functionMap.put(name, func);
    }

    /**
     * <p>
     * 呼び出し可能な関数を返します。
     * </p>
     * 
     * @param name
     *            関数名。<br>
     *            指定した名前で関数が定義されていない場合、{@link ComputeException}例外をスローします。
     * @return 関数。
     */
    public Function getFunction(String name) {
        if (!this.functionMap.containsKey(name)) {
            throw new ComputeException("関数{" + name + "}は定義されていません。");
        }
        Function func = (Function) this.functionMap.get(name);
        return func;
    }

}
