
package jp.sourceforge.rpn_computer.test;

import jp.sourceforge.rpn_computer.ParseException;
import jp.sourceforge.rpn_computer.RpnNode;
import jp.sourceforge.rpn_computer.RpnParser;
import junit.framework.TestCase;

/**
 * 構文解析のテストを行います。
 * 
 * @author uguu@users.sourceforge.jp
 */
public class RpnParserTest extends TestCase {

    /**
     * インスタンスを初期化します。
     * 
     * @param name
     *            テストの名前。
     */
    public RpnParserTest(String name) {
        super(name);
    }

    /**
     * 正常系テスト1。
     */
    public void testNormal1() {
        String exp = "12.34 + 56.78";

        RpnParser parser = new RpnParser();
        RpnNode node = parser.parse(exp);

        String str = "expression statement\n" + "\tadditive expression\n" + "\t\tmultiplicative expression\n" + "\t\t\tprimary expression\n" + "\t\t\t\tnumber[12.34]\n" + "\t\tadd[+]\n" + "\t\tadditive expression\n" + "\t\t\tmultiplicative expression\n" + "\t\t\t\tprimary expression\n" + "\t\t\t\t\tnumber[56.78]\n";

        assertEquals(str, node.toString());
    }

    /**
     * 正常系テスト2。
     */
    public void testNormal2() {
        String exp = "(1.2 + 3.4) * (5.6 - 7.8 / (9.0 + 0.1))";

        RpnParser parser = new RpnParser();
        RpnNode node = parser.parse(exp);

        String str = "expression statement\n" + "\tadditive expression\n" + "\t\tmultiplicative expression\n" + "\t\t\tprimary expression\n" + "\t\t\t\tleft bracket[(]\n" + "\t\t\t\tadditive expression\n" + "\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\tnumber[1.2]\n" + "\t\t\t\t\tadd[+]\n" + "\t\t\t\t\tadditive expression\n" + "\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\tnumber[3.4]\n" + "\t\t\t\tright bracket[)]\n" + "\t\t\tmultiply[*]\n" + "\t\t\tmultiplicative expression\n" + "\t\t\t\tprimary expression\n" + "\t\t\t\t\tleft bracket[(]\n" + "\t\t\t\t\tadditive expression\n" + "\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\tnumber[5.6]\n" + "\t\t\t\t\t\tsubtract[-]\n" + "\t\t\t\t\t\tadditive expression\n" + "\t\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\t\tnumber[7.8]\n" + "\t\t\t\t\t\t\t\tdivide[/]\n" + "\t\t\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\t\t\tleft bracket[(]\n" + "\t\t\t\t\t\t\t\t\t\tadditive expression\n" + "\t\t\t\t\t\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\t\t\t\t\t\tnumber[9.0]\n" + "\t\t\t\t\t\t\t\t\t\t\tadd[+]\n" + "\t\t\t\t\t\t\t\t\t\t\tadditive expression\n" + "\t\t\t\t\t\t\t\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\t\t\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\t\t\t\t\t\t\t\tnumber[0.1]\n" + "\t\t\t\t\t\t\t\t\t\tright bracket[)]\n" + "\t\t\t\t\tright bracket[)]\n";

        assertEquals(str, node.toString());
    }

    /**
     * 正常系テスト3。
     */
    public void testNormal3() {
        String exp = "1 + + 2";

        RpnParser parser = new RpnParser();
        RpnNode node = parser.parse(exp);

        String str = "expression statement\n" + "\tadditive expression\n" + "\t\tmultiplicative expression\n" + "\t\t\tprimary expression\n" + "\t\t\t\tnumber[1.0]\n" + "\t\tadd[+]\n" + "\t\tadditive expression\n" + "\t\t\tmultiplicative expression\n" + "\t\t\t\tprimary expression\n" + "\t\t\t\t\tpre add[+]\n" + "\t\t\t\t\tnumber[2.0]\n";

        assertEquals(str, node.toString());
    }

    /**
     * 正常系テスト4。
     */
    public void testNormal4() {
        String exp = "(+1)";

        RpnParser parser = new RpnParser();
        RpnNode node = parser.parse(exp);

        String str = "expression statement\n" + "\tadditive expression\n" + "\t\tmultiplicative expression\n" + "\t\t\tprimary expression\n" + "\t\t\t\tleft bracket[(]\n" + "\t\t\t\tadditive expression\n" + "\t\t\t\t\tmultiplicative expression\n" + "\t\t\t\t\t\tprimary expression\n" + "\t\t\t\t\t\t\tpre add[+]\n" + "\t\t\t\t\t\t\tnumber[1.0]\n" + "\t\t\t\tright bracket[)]\n";

        assertEquals(str, node.toString());
    }

    /**
     * 異常系テスト1。{@link RpnParser#parse(String)}メソッドのexpressionがnullのとき、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testFail1() {
        RpnParser parser = new RpnParser();
        try {
            parser.parse(null);
            fail();
        } catch (NullPointerException e) {
            assertEquals("expressionがnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト2。不正な構文を指定したとき、{@link ParseException}例外がスローされることを確認します。
     */
    public void testFail2() {
        String exp = "1 2";

        RpnParser parser = new RpnParser();
        try {
            parser.parse(exp);
            fail();
        } catch (ParseException e) {
            assertEquals("0行目、0列目でトークン\"2\"が出現しました。正しくは、次のパターンのトークンを期待しています。{}", e.getMessage());
        }
    }

    /**
     * 異常系テスト3。不正な構文を指定したとき、{@link ParseException}例外がスローされることを確認します。
     */
    public void testFail3() {
        String exp = "1 (1 + 2)";

        RpnParser parser = new RpnParser();
        try {
            parser.parse(exp);
            fail();
        } catch (ParseException e) {
            assertEquals("0行目、0列目でトークン\"(\"が出現しました。正しくは、次のパターンのトークンを期待しています。{}", e.getMessage());
        }
    }

    /**
     * 異常系テスト4。不正な構文を指定したとき、{@link ParseException}例外がスローされることを確認します。
     */
    public void testFail4() {
        String exp = "1 + * 1";

        RpnParser parser = new RpnParser();
        try {
            parser.parse(exp);
            fail();
        } catch (ParseException e) {
            assertEquals("期待したトークンが出現しませんでした。次のパターンのトークンを期待しています。{[0-9]+(\\.[0-9]+)?, \\(}", e.getMessage());
        }
    }

    /**
     * 異常系テスト5。不正な構文を指定したとき、{@link ParseException}例外がスローされることを確認します。
     */
    public void testFail5() {
        String exp = "1 * * 1";

        RpnParser parser = new RpnParser();
        try {
            parser.parse(exp);
            fail();
        } catch (ParseException e) {
            assertEquals("期待したトークンが出現しませんでした。次のパターンのトークンを期待しています。{[0-9]+(\\.[0-9]+)?, \\(}", e.getMessage());
        }
    }

    /**
     * 異常系テスト6。不正な構文を指定したとき、{@link ParseException}例外がスローされることを確認します。
     */
    public void testFail6() {
        String exp = "()";

        RpnParser parser = new RpnParser();
        try {
            parser.parse(exp);
            fail();
        } catch (ParseException e) {
            assertEquals("期待したトークンが出現しませんでした。次のパターンのトークンを期待しています。{[0-9]+(\\.[0-9]+)?, \\(}", e.getMessage());
        }
    }
}
