/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.rpn_computer.node;

import jp.sourceforge.rpn_computer.ParseException;
import jp.sourceforge.rpn_computer.RpnCompileContext;
import jp.sourceforge.rpn_computer.RpnNode;
import jp.sourceforge.rpn_computer.token.LeftBracketToken;
import jp.sourceforge.rpn_computer.token.LeftBracketTokenInfo;
import jp.sourceforge.rpn_computer.token.MinusToken;
import jp.sourceforge.rpn_computer.token.MinusTokenInfo;
import jp.sourceforge.rpn_computer.token.NumberToken;
import jp.sourceforge.rpn_computer.token.NumberTokenInfo;
import jp.sourceforge.rpn_computer.token.PlusToken;
import jp.sourceforge.rpn_computer.token.PlusTokenInfo;
import jp.sourceforge.rpn_computer.token.RightBracketTokenInfo;
import jp.sourceforge.rpn_computer.token.SkipTokenInfo;
import jp.sourceforge.tokenizer.Token;
import jp.sourceforge.tokenizer.TokenInfo;
import jp.sourceforge.tokenizer.Tokenizer;

/**
 * <p>
 * PrimaryExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class PrimaryExpressionNode extends AbstractNode {

    /**
     * {@inheritDoc}
     */
    public void parse(Tokenizer tokenizer) {
        RpnNode node;

        TokenInfo[] readTokenInfos = new TokenInfo[] { new PlusTokenInfo(), new MinusTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { new SkipTokenInfo() };
        if (tokenizer.read(readTokenInfos, skipTokenInfos)) {
            Token token = tokenizer.current();
            if (token instanceof PlusToken) {
                node = new PreAddNode();
            } else if (token instanceof MinusToken) {
                node = new PreSubtractNode();
            } else {
                throw new ParseException(token, readTokenInfos);
            }
            node.parse(tokenizer);
            this.addNode(node);
        }

        readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new LeftBracketTokenInfo(), };
        if (!tokenizer.read(readTokenInfos, skipTokenInfos)) {
            throw new ParseException(null, readTokenInfos);
        }
        Token token = tokenizer.current();
        if (token instanceof NumberToken) {
            node = new NumberNode();
            node.parse(tokenizer);
            this.addNode(node);
        } else if (token instanceof LeftBracketToken) {
            node = new LeftBracketNode();
            node.parse(tokenizer);
            this.addNode(node);

            node = new AdditiveExpressionNode();
            node.parse(tokenizer);
            this.addNode(node);

            readTokenInfos = new TokenInfo[] { new RightBracketTokenInfo() };
            if (!tokenizer.read(readTokenInfos, skipTokenInfos)) {
                throw new ParseException(null, readTokenInfos);
            }
            node = new RightBracketNode();
            node.parse(tokenizer);
            this.addNode(node);
        } else {
            throw new ParseException(token, readTokenInfos);
        }
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return "primary expression\n" + this.toChildrenString();
    }

    /**
     * {@inheritDoc}
     */
    public void compile(RpnCompileContext context) {
        RpnNode[] nodes = this.getChildren();

        RpnNode preOpeNode = null;

        int i = 0;
        if ((nodes[i] instanceof PreAddNode) || (nodes[i] instanceof PreSubtractNode)) {
            preOpeNode = nodes[i];
            i++;
        }

        if (nodes[i] instanceof NumberNode) {
            nodes[i].compile(context);
        } else {
            nodes[i + 1].compile(context);
        }

        if (preOpeNode != null) {
            preOpeNode.compile(context);
        }
    }
}
