<?php
/* ========================================================================
 - [rsl_pager.php]
 - 内容：ページ送り
 - 作成：高橋 裕志郎
 - ライセンス:
 -      This source file is subject to version 3.0 of the PHP license,
 -      that is available at http://www.php.net/license/3_0.txt
 -      If you did not receive a copy of the PHP license and are unable 
 -      to obtain it through the world-wide-web, please send a note to 
 -      license@php.net so we can mail you a copy immediately.  
 - 問い合わせ先：
 -      yujiro@rakuto.net
 -      http://www.rakuto.net/study/htdocs/
 -      Copyright (C) 2005 `rakuto.net'. All Rights Reserved.
 - 更新履歴：
 -      [2005/01/21] EXPLAINに対応
 -      [2005/01/08] SQLite用
 - ======================================================================== */    

/**
 * Pager移動可能ページの表示限度
 * @const RSL_PAGER_WIDTH
 */
define('RSL_PAGER_WIDTH',5);

/**
 * ページ送り一覧クラス
 *
 * @author 高橋 裕志郎 <yujiro@rakuto.net>
 * @package RSL_pager
 * @access public
 * @version 1.0.05.1.21
 */
class RSL_pager
{
    /* -=-=-= データメンバの宣言 =-=-=- */
    /**
     * DB接続オブジェクト
     * @var object
     */
    private $conn = NULL;
    /**
     * DATA取得開始行数
     * @var integer
     */
    private $offset = 0;
    /**
     * 最大DATA取得行数
     * @var integer
     */
    private $limit  = 20;
    /**
     * 最大DATA取得可能行数
     * @var integer
     */
    private $maxrow = null;

    /**
     * 解析後のSQL文配列
     * @var array
     */
    private $sql_parsed = array();

    /**
     * ページ情報格納
     * @var array
     */
    private $infos = null;

    /**
     * コンストラクタ
     *
     * @access public
     * @param object &$conn DB接続オブジェクト
     * @param integer $offset DATA取得開始行数
     * @param integer $limit 最大DATA取得行数
     * @return void
     */
    public function __construct(&$conn, $offset, $limit)
    {
        $this->conn   = &$conn;        // コピーの方が良いかも
        $this->offset = $offset;
        $this->limit  = $limit;
        $this->maxrow = null;
        $this->infos  = null;
    }

    /**
     * SQL文の設定
     *
     * @access public
     * @param string $sql SQL文（SELECT句）
     * @return void
     */
    public function setSql($sql)
    {
        $this->sql_parsed = RSL_db::parseSql($sql);

        if (!empty($this->sql_parsed['limit'])){
            $this->limit = $this->sql_parsed['limit'];
        }
        if (!empty($this->sql_parsed['offset'])){
            $this->offset = $this->sql_parsed['offset'];
        }
    }

    /**
     * 解析後のSQL文配列の改竄
     *
     * @access public
     * @param string $key   連想配列キー
     * @param string $value 改竄データ
     * @return void
     */
    public function setParsed($key,$value)
    {
        $this->sql_parsed[$key] = $value;
    }

    /**
     * 最大取得可能行数の設定
     *
     * @access public
     * @param integer $maxrow 最大取得可能行数
     * @return void
     */
    public function setMaxRow($maxrow)
    {
        $this->maxrow = $maxrow;
    }

    /**
     * 最大取得可能行数の設定
     *
     * @access public
     * @return integer 最大取得可能行数
     */
    public function getMaxRow()
    {
        if (!empty($this->sql_parsed['pragma']) || 
            !empty($this->sql_parsed['explain'])){
            return $this->maxrow = 0;
        }

        $sql = "SELECT ".
                        "count(*)".
                    " FROM ".
                        $this->sql_parsed['table'];
        if (!empty($this->sql_parsed['where'])){
            $sql .=    " WHERE ".
                            $this->sql_parsed['where'];
        }

        /* データの取得 */
        $stmt = $this->conn->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetchSingle();
        $this->maxrow = empty($result)?0:$result;
        
        return $this->maxrow;
    }

   /**
     * ページ情報配列の設定
     *
     * @access public
     * @return void
     */
    public function getInfo()
    {
        if ($this->maxrow === null) {
            $this->getMaxRow();
        }
        if (is_array($this->infos)){
            return $this->infos;
        }

        $this->infos = array();
        $this->infos['offset'] = $this->offset;
        $this->infos['prev'] = $this->offset - $this->limit;
        $this->infos['next'] = $this->offset + $this->limit;

        $this->infos['cur_page'] = intval(($this->offset / $this->limit) + 1);
        $this->infos['max_page'] = intval(($this->maxrow / $this->limit) + 0.9999);
        $this->infos['prev_bool'] = true;
        $this->infos['next_bool'] = true;
        if ($this->infos['prev'] < 0){
            $this->infos['prev_bool'] = false;
        }
        if ($this->infos['next'] >= $this->maxrow){
            $this->infos['next_bool'] = false;
        }
        return $this->infos;
    }

    /**
     * ページ送り情報配列の設定
     *
     * @access public
     * @return void
     */
    public function getPager()
    {
        $infos = $this->getInfo();
        $pages = array();
        
        /* 現在のページより左がわのページャーを取得 */
        $lefts = array();
        $page = $infos['cur_page'];
        for ($count=0;$count<RSL_PAGER_WIDTH;$count++){
            $page--;
            if ($page <= 0){
                break;
            }
            $lefts[] = array(
                'page'=> $page,
                'offset'=> $page * $this->limit - $this->limit,
                'class'=> 'pager'
            );
        }

        /* 配列の順番をひっくり返す */
        for ($row=(count($lefts)-1);$row>=0;$row--){
            $pages[] = $lefts[$row];
        }

        /* 現在のページャーを取得 */
        $pages[] = array(
            'page'=> $infos['cur_page'],
            'offset'=> $infos['cur_page'] * $this->limit - $this->limit,
            'class'=> 'curr'
        );

        /* 現在のページより右がわのページャーを取得 */
        $page = $infos['cur_page'];
        for ($count=0;$count<RSL_PAGER_WIDTH;$count++){
            $page++;
            if ($page > $infos['max_page']){
                break;
            }
            $pages[] = array(
                'page'=> $page,
                'offset'=> $page * $this->limit - $this->limit,
                'class'=> 'pager'
            );
        }

        return $pages;
    }

    /**
     * DBからデータ読み込み
     *
     * @access public
     * @return array
     */
    public function getValues()
    {
        $res = null;
        $row = 0;
        if ($this->maxrow === null) {
            $this->getMaxRow();
        }
        $offset = ($this->maxrow < $this->offset) ? 0:$this->offset;

        $sql = "";
        if (!empty($this->sql_parsed['field'])){
            $sql = "SELECT ".
                    $this->sql_parsed['field'];
        }
        if (!empty($this->sql_parsed['table'])){
            $sql .= " FROM ".
                    $this->sql_parsed['table'];
        }
        if (!empty($this->sql_parsed['where'])){
            $sql .= " WHERE ".
                        $this->sql_parsed['where'];
        }
        if (!empty($this->sql_parsed['order'])){
            $sql .= " ORDER BY ".
                        $this->sql_parsed['order'];
        }
        if (!empty($this->sql_parsed['group'])){
            $sql .= " GROUP BY ".
                        $this->sql_parsed['group'];
        }
        $sql .= " LIMIT ".$this->limit;
        $sql .= " OFFSET ".$offset;

        if (!empty($this->sql_parsed['pragma'])){
            $sql = "PRAGMA ".
                    $this->sql_parsed['pragma'];
        }

        if (!empty($this->sql_parsed['explain'])){
            $sql = "EXPLAIN ".$sql;
        }

        /* データの取得 */
        $stmt = $this->conn->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetchAll(PDO_FETCH_ASSOC);

        return $result;
    }
}
// RSL_pagerの終了
?>
