<?php
/* SVN FILE: $Id: rkt_db.php 440 2008-05-20 08:50:50Z bb_yujiro $ */
/**
 * データベース接続
 *
 * PHP versions 5
 *
 *      hitSuji : Social Network Service <http://rakuto.net/rktSNS/>
 *      Copyright (c) 2007 Yujiro Takahashi
 *
 * Licensed under The MIT License
 * Redistributions of files must retain the above copyright notice.
 *
 * @filesource
 * @package         hitSuji
 * @copyright       Copyright (c) 2007 Yujiro Takahashi
 * @link            http://rakuto.net/
 * @author          Yujiro Takahashi <yujiro@rakuto.net>
 * @version         $Revision: 440 $
 * @modifiedby      $LastChangedBy: bb_yujiro $
 * @lastmodified    $Date: 2008-05-20 17:50:50 +0900 (火, 20 5 2008) $
 * @license         http://opensource.org/licenses/mit-license.php The MIT License
 */

/**
 * RKT_DBクラス
 *
 * @category        hitSuji Core
 * @package         hitSuji
 * @copyright       Copyright (c) 2007 Yujiro Takahashi
 * @link            http://rakuto.net/
 * @author          Yujiro Takahashi <yujiro@rakuto.net>
 * @version         $Revision: 440 $
 * @modifiedby      $LastChangedBy: bb_yujiro $
 * @lastmodified    $Date: 2008-05-20 17:50:50 +0900 (火, 20 5 2008) $
 * @license         http://opensource.org/licenses/mit-license.php The MIT License
 */
class RKT_DB
{
    /**
     * PDOインスタンス
     * @var object
     */
    private static $_pdo;

    /**
     * 評価式変換
     * @var array
     */
    private static $_exprs = array(
            '='=>    'through',
            '!='=>   'through',
            '>'=>    'through',
            '<'=>    'through',
            '>='=>   'through',
            '<='=>   'through',
            'LIKE'=> 'through',
            '&'=>    'bitFlag',
        );

    /**
     * コンストラクタ
     *
     * @access private
     * @return void
     */
    private function __construct()
    {
    }

    /**
     * Singleton メソッド
     *
     * @access public
     * @return PDO object
     *
     * RKT_DB::exec()->getAccount();
     *
     */
    public static function exec()
    {
        if (isset(self::$_pdo)) {
            return self::$_pdo;
        }
        return self::getInstance();
    }

    /**
     * Singleton メソッド
     *
     * @access public
     * @param string $dsn
     * @return PDO object
     */
    public static function getInstance($dsn=DB_DSN)
    {
        if (!isset(self::$_pdo)) {
            /* PDO データベースハンドルの生成 */
            self::$_pdo = new PDO($dsn, DB_USER, DB_PASSWORD);

            $driver = self::$_pdo->getAttribute(PDO::ATTR_DRIVER_NAME);

            /* エラーモードの設定（例外処理を有効） */
            self::$_pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            self::$_pdo->setAttribute(PDO::ATTR_TIMEOUT, 4);

            $func_names = array(
                0=> array('RKT_DB','dummy'),        // NAMES OFF 0
                1=> array('RKT_DB','setNames')      // NAMES ON  1
            );
            call_user_func($func_names[USE_SET_NAMES]);
        }
        return self::$_pdo;
    }

    /**
     * 文字コードの設定
     *
     * @access public
     * @return void
     */
    public function setNames()
    {
        if (isset(self::$_pdo)) {
            self::$_pdo->exec("SET NAMES 'utf8'");
        }
    }

    /**
     * ダミー
     *
     * @access public
     * @return void
     */
    public function dummy()
    {
    }

    /**
     * テーブルリストの取得
     *
     * @access public
     * @param string $table
     * @param mixed $condition
     * @param string $column
     * @param string $order
     * @return array
     */
    public static function getList($table, $condition=null, $column='*', $order=null, $limit=null)
    {
        $objdb  = self::getInstance();
        $where  = '';
        $list   = array();
        $wheres = array();

        /* WHERE句の作成 */
        if (is_array($condition)){
            foreach ($condition as $key=> $row){
                $operator = '=';
                $value    = $row;
                if (is_array($row)){
                    if (isSet($row[2])){
                        $key      = $row[0];
                        $operator = $row[1];
                        $value    = $row[2];
                    } else {
                        $operator = $row[0];
                        $value    = $row[1];
                    }
                }
                $expr     = $key.' '.$operator.' '.$objdb->quote($value);
                $wheres[] = call_user_func(array(__CLASS__, self::$_exprs[$operator]), $expr);
            }
        } // if (is_array($condition))
        if (count($wheres)){
            $where = 'WHERE '. implode(' AND ',$wheres) .' ';
        }
        if (!empty($order)){
            $order = 'ORDER BY '. $order.' ';
        }
        if (!empty($limit)){
            $limit = 'LIMIT '. $limit;
        }

        $sql = 'SELECT '.$column.' FROM '.DB_PREFIX.$table.' '.
               $where.$order.$limit;
        $stmt = $objdb->prepare($sql);
        $stmt->execute(); 
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (empty($result)){
            return $list;
        }
        
        return $result;
    }

    /**
     * テーブル行の取得
     *
     * @access public
     * @param string $table
     * @param mixed $condition
     * @param string $column
     * @return array
     */
    public static function getRow($table, $condition, $column='*')
    {
        $objdb  = self::getInstance();
        $where  = '';
        $wheres = array();

        /* WHERE句の作成 */
        if (is_numeric($condition)){
            $wheres[] = 'id = '.$objdb->quote($condition);
        } elseif (is_array($condition)){
            foreach ($condition as $key=> $row){
                $value      = $row;
                $operator = '=';
                if (is_array($row)){
                    if (isSet($row[2])){
                        $key      = $row[0];
                        $operator = $row[1];
                        $value    = $row[2];
                    } else {
                        $operator = $row[0];
                        $value    = $row[1];
                    }
                }
                $expr     = $key.' '.$operator.' '.$objdb->quote($value);
                $wheres[] = call_user_func(array(__CLASS__, self::$_exprs[$operator]), $expr);
            }
        } // if (is_numeric($condition))
        if (count($wheres)){
            $where = 'WHERE '. implode(' AND ',$wheres) .' ';
        }

        $sql = 'SELECT '.$column.' FROM '.DB_PREFIX.$table.' '.$where;
        $stmt = $objdb->prepare($sql);
        $stmt->execute(); 
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stmt->closeCursor();

        return $result;
    }

    /**
     * 現在の時刻のフォーマット
     *
     * @access public
     * @return string
     */
    public static function now()
    {
        $formats = array(
            'mysql'=>  'now()',
            'sqlite'=> "date('now')",
            'pgsql'=>  'now()',
        );

        return $formats[DB_DRIVER];
    }

    /**
     * 年齢取得のフォーマット
     * 
     * @access public
     * @param string $column
     * @return string
     */
    public static function age($column)
    {
        /* (今日の日付-誕生日)/10000の小数点以下切捨て */
        $formats = array(
            'mysql'=>  'YEAR(DATE_SUB(now(),INTERVAL to_days('.$column.') DAY))',
            'sqlite'=> "((julianday('now') - julianday(".$column.")) * 0.00273785)",
            'pgsql'=>  "date_part('year', age(".$column."))",
        );

        return $formats[DB_DRIVER];
    }

    /**
     * ビットフラグ式
     *
     * @access public
     * @param string $expression
     *               $expression = 'flag & 10';
     * @return string
     */
    public static function bitFlag($expression)
    {
        $formats = array(
            'mysql'=>  $expression,
            'sqlite'=> $expression,
            'pgsql'=>  'CASE WHEN ('.$expression.') != 0 THEN true ELSE false END',
        );

        return $formats[DB_DRIVER];
    }

    /**
     * ダミー用式変換
     *
     * @access string
     * @param string $expression
     * @return void
     */
    public function through($expression)
    {
        return $expression;
    }

    /**
     * 日付型変換
     *
     * @access public
     * @param string $column
     * @return string
     */
    public static function toDate($column)
    {
        $formats = array(
            'mysql'=>  'DATE_FORMAT('.$column.", '%Y-%m-%d')",
            'sqlite'=> 'date('.$column.')',
            'pgsql'=>  'to_char('.$column.", 'YYYY-MM-DD')",
        );

        return $formats[DB_DRIVER];
    }

    /**
     * 日付フォーマット
     *
     * $format の変換指定子は strftime() 関数を参照
     *
     * @access public
     * @param string $column
     * @param string $format
     * @return string
     */
    public static function dateFormat($column, $format)
    {
        $func = array('RKT_DB', DB_DRIVER.'Format');
        $format = call_user_func($func, $format);

        $formats = array(
            'mysql'=>  'DATE_FORMAT('.$column.",'".$format."')",
            'sqlite'=> "strftime('".$format."', ".$column.")",
            'pgsql'=>  'to_char('.$column.", '".$format."')",
        );

        return $formats[DB_DRIVER];
    }

    /**
     * 日付フォーマットをMySql用に変換
     *
     * @access private
     * @param string $format
     * @return string
     */
    public static function mysqlFormat($format)
    {
        $format = preg_replace('/%M/','%i', $format);
        return $format;
    }

    /**
     * 日付フォーマットをSQLite用に変換
     *
     * @access private
     * @param string $format
     * @return string
     */
    public static function sqliteFormat($format)
    {
        return $format;
    }

    /**
     * 日付フォーマットをPosqgreSQL用に変換
     *
     * @access private
     * @param string $format
     * @return string
     */
    public static function pgsqlFormat($format)
    {
        $format = preg_replace('/%Y/','YYYY', $format);
        $format = preg_replace('/%m/','MM', $format);
        $format = preg_replace('/%d/','DD', $format);

        $format = preg_replace('/%H/','HH24', $format);
        $format = preg_replace('/%M/','MI', $format);
        $format = preg_replace('/%S/','SS', $format);

        return $format;
    }
} // RKT_DBの終了
?>