#!ruby
# -*- tab-width: 8; -*-

# finder -- walk a file hierarchy like unix find command
# Copyright (C) 2008-2010  pegacorn
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA


class Dir
  class Finder
    class ArgumentError < ::ArgumentError
      def initialize(argument, error)
	super("#{argument}: #{error}")
      end
    end

    class PrimaryError < ArgumentError
      def initialize(key, value = nil)
	primary = value ? "#{key} => #{value}" : key
	super(primary, 'invalid primary')
      end
    end

    class OptionError < ArgumentError
      def initialize(key, value = nil)
	option = value ? "#{key} => #{value}" : key
	super(option, 'invalid option')
      end
    end

   protected
    PRIMARY_KEYS = [
      :name,
      :type
    ].freeze

    OPTIONS = {
      :depth? => false,
      :sort? => false,
      :sub? => false,
      :together? => false
    }.freeze

    def initialize(primaries = {}, options = {})
      self.primaries = primaries
      self.options = options
    end

   public
    def primaries=(primaries)
      invalid_primaries = primaries.keys - PRIMARY_KEYS
      if invalid_primaries.size.nonzero?
	raise PrimaryError.new(invalid_primaries.first)
      end

      @primaries = primaries.dup.freeze
    end

    def options=(options)
      invalid_options = options.keys - OPTIONS.keys
      if invalid_options.size.nonzero?
	raise OptionError.new(invalid_options.first)
      end

      @options = OPTIONS.merge(options).freeze
    end

    attr_reader :primaries, :options

    def find(top_dir, block = Proc.new)
      find_sub(top_dir, nil, block)
    end

   protected
    def find_sub(top_dir, sub_dir, block = Proc.new)
      dpath = add_path(top_dir, sub_dir)
      begin
	files = Dir.entries(dpath).delete_if do |entry|
	  /\A\.\.?\z/ =~ entry
	end
      rescue IOError, SystemCallError => io_error
	on_raise_io_error(dpath, io_error)
	return
      end

      if @options[:sort?]
	files.sort!
      end

      paths = {}
      dirs = []
      files.each do |file|
	path = File.join(dpath, file)
	paths[file] = path
	dirs << file if FileTest.directory? path
      end

      if @options[:depth?]
	dirs.each do |file|
	  sub_dpath = add_path(sub_dir, file)
	  find_sub(top_dir, sub_dpath, block)
	end
      end

      if @options[:together?]
	target_files = files.find_all do |file|
	  target?(paths[file], dpath, file)
	end
	if @options[:sub?]
	  block.call(sub_dir, target_files)
	else
	  block.call(dpath, target_files)
	end
      else
	files.each do |file|
	  path = paths[file]
	  next unless target?(path, dpath, file)
	  if @options[:sub?]
	    block.call(add_path(sub_dir, file))
	  else
	    block.call(path)
	  end
	end
      end

      unless @options[:depth?]
	dirs.each do |file|
	  sub_dpath = add_path(sub_dir, file)
	  find_sub(top_dir, sub_dpath, block)
	end
      end
    end # def find

    def target?(path, dir, file)
      is_target = true
      @primaries.each do |key, val|
	case key.to_sym
	when :name
	  is_target = (val =~ file)
	when :type
	  case val.to_sym
	  when :d, :directory
	    is_target = FileTest.directory?(path)
	  when :f, :regular
	    is_target = FileTest.file?(path)
	  else
	    raise PrimaryError.new(key, val)
	  end
	else
	  raise PrimaryError.new(key)
	end
	break unless is_target
      end
      return is_target
    end # def target?

    def on_raise_io_error(file, error)
      $stderr.puts "error: #{file}: #{error}"
    end

    def add_path(dir, file)
      if file.nil?
	dir.dup
      elsif dir.nil?
	file.dup
      else
	File.join(dir, file)
      end
    end
  end # class Finder
end # class Dir
