/*
 * Decompiled with CFR 0.152.
 */
package io.remme.java.utils;

import io.remme.java.enums.KeyType;
import io.remme.java.enums.Patterns;
import io.remme.java.error.RemmeKeyException;
import io.remme.java.error.RemmeValidationException;
import io.remme.java.utils.Certificate;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EllipticCurve;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import net.i2p.crypto.eddsa.EdDSASecurityProvider;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.sec.SECNamedCurves;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.ECPointUtil;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECNamedCurveSpec;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;

public class Functions {
    public static String generateAddress(String familyName, String data) {
        return DigestUtils.sha512Hex((String)familyName).substring(0, 6) + DigestUtils.sha512Hex((String)data).substring(0, 64);
    }

    public static String generateAddress(String familyName, byte[] data) {
        return DigestUtils.sha512Hex((String)familyName).substring(0, 6) + DigestUtils.sha512Hex((byte[])data).substring(0, 64);
    }

    public static String generateSettingsAddress(String key) {
        List<String> keyParts = Arrays.asList(key.split("\\.", 4));
        List addressParts = keyParts.stream().map(v -> DigestUtils.sha256Hex((String)v).substring(0, 16)).collect(Collectors.toList());
        while (4 - addressParts.size() != 0) {
            addressParts.add(DigestUtils.sha256Hex((String)"").substring(0, 16));
        }
        return "000000" + String.join((CharSequence)"", addressParts);
    }

    public static String publicKeyToPem(PublicKey publicKey) {
        return "-----BEGIN PUBLIC KEY-----\n" + Base64.encodeBase64String((byte[])publicKey.getEncoded()) + "\n-----END PUBLIC KEY-----";
    }

    public static String privateKeyToPem(PrivateKey privateKey) {
        return "-----BEGIN PRIVATE KEY-----\n" + Base64.encodeBase64String((byte[])privateKey.getEncoded()) + "\n-----END PRIVATE KEY-----";
    }

    public static PublicKey getPublicKeyFromPEM(String pem) throws IOException {
        SubjectPublicKeyInfo pubInfo = (SubjectPublicKeyInfo)new PEMParser((Reader)new StringReader(pem)).readObject();
        return Functions.getPublicKeyFromBytesArray(KeyType.RSA, pubInfo.getEncoded());
    }

    public static PublicKey getPublicKeyFromPEM(File pem) throws IOException {
        SubjectPublicKeyInfo pubInfo = (SubjectPublicKeyInfo)new PEMParser((Reader)new FileReader(pem)).readObject();
        return Functions.getPublicKeyFromBytesArray(KeyType.RSA, pubInfo.getEncoded());
    }

    public static PrivateKey getPrivateKeyFromPEM(File pem) throws IOException {
        PrivateKeyInfo privKeyInfo = (PrivateKeyInfo)new PEMParser((Reader)new FileReader(pem)).readObject();
        return Functions.getPrivateKeyFromBytesArray(KeyType.RSA, privKeyInfo.getEncoded());
    }

    public static PrivateKey getPrivateKeyFromPEM(String pem) throws IOException {
        PrivateKeyInfo privKeyInfo = (PrivateKeyInfo)new PEMParser((Reader)new StringReader(pem)).readObject();
        return Functions.getPrivateKeyFromBytesArray(KeyType.RSA, privKeyInfo.getEncoded());
    }

    public static PublicKey getPublicKeyFromBytesArray(KeyType keyType, byte[] encoded) {
        try {
            switch (keyType) {
                case RSA: {
                    KeyFactory factory = KeyFactory.getInstance("RSA", "BC");
                    return factory.generatePublic(new X509EncodedKeySpec(encoded));
                }
                case ECDSA: {
                    return Functions.getECDSAPublicKeyFromBytes(encoded);
                }
                case EdDSA: {
                    KeyFactory factory = KeyFactory.getInstance("EdDSA", "EdDSA");
                    return factory.generatePublic(new X509EncodedKeySpec(encoded));
                }
            }
            throw new RemmeKeyException("Unsupported key type!");
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidKeySpecException e) {
            throw new RemmeKeyException(e);
        }
    }

    public static PrivateKey getPrivateKeyFromBytesArray(KeyType keyType, byte[] encoded) {
        try {
            switch (keyType) {
                case RSA: {
                    KeyFactory factory = KeyFactory.getInstance("RSA", "BC");
                    return factory.generatePrivate(new PKCS8EncodedKeySpec(encoded));
                }
                case ECDSA: {
                    return Functions.generateECDSAPrivateKey(encoded);
                }
                case EdDSA: {
                    KeyFactory factory = KeyFactory.getInstance("EdDSA", "EdDSA");
                    return factory.generatePrivate(new PKCS8EncodedKeySpec(encoded));
                }
            }
            throw new RemmeKeyException("Unsupported key type!");
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidKeySpecException e) {
            throw new RemmeKeyException(e);
        }
    }

    public static PrivateKey generateECDSAPrivateKey(byte[] keyBin) {
        try {
            ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec((String)"secp256k1");
            KeyFactory kf = KeyFactory.getInstance("ECDSA", "BC");
            ECNamedCurveSpec params = new ECNamedCurveSpec("secp256k1", spec.getCurve(), spec.getG(), spec.getN());
            ECPrivateKeySpec privKeySpec = new ECPrivateKeySpec(new BigInteger(keyBin), (ECParameterSpec)params);
            return kf.generatePrivate(privKeySpec);
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidKeySpecException e) {
            throw new RemmeKeyException(e);
        }
    }

    public static PublicKey getECDSAPublicKeyFromHex(String pubHex) {
        try {
            String hexX = pubHex.substring(0, pubHex.length() / 2);
            String hexY = pubHex.substring(pubHex.length() / 2);
            ECPoint point = new ECPoint(new BigInteger(hexX, 16), new BigInteger(hexY, 16));
            ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec((String)"secp256k1");
            KeyFactory kf = KeyFactory.getInstance("ECDSA", "BC");
            ECNamedCurveSpec params = new ECNamedCurveSpec("secp256k1", spec.getCurve(), spec.getG(), spec.getN());
            ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(point, (ECParameterSpec)params);
            return kf.generatePublic(pubKeySpec);
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidKeySpecException e) {
            throw new RemmeKeyException(e);
        }
    }

    public static String ecdsaPrivateKeyToHex(PrivateKey privateKey) {
        return ((BCECPrivateKey)privateKey).getS().toString(16);
    }

    public static String ecdsaPublicKeyToHex(PublicKey publicKey, boolean compressed) {
        byte[] w = ((BCECPublicKey)publicKey).getQ().getEncoded(compressed);
        return Hex.encodeHexString((byte[])w);
    }

    public static byte[] hexToBytes(String hex) {
        return new BigInteger(hex, 16).toByteArray();
    }

    public static String compress(PublicKey publicKey) {
        org.bouncycastle.math.ec.ECPoint point = ((BCECPublicKey)publicKey).getQ();
        byte[] x = point.getAffineXCoord().toBigInteger().toByteArray();
        byte[] y = point.getAffineYCoord().toBigInteger().toByteArray();
        byte[] xy = new byte[x.length + y.length];
        System.arraycopy(x, 0, xy, 0, x.length);
        System.arraycopy(y, 0, xy, x.length, y.length);
        BigInteger pubKey = new BigInteger(xy);
        return Functions.compress(pubKey);
    }

    public static String compress(BigInteger pubKey) {
        String pubKeyYPrefix = pubKey.testBit(0) ? "03" : "02";
        String pubKeyHex = pubKey.toString(16);
        System.out.println(pubKeyHex.length());
        String pubKeyX = pubKeyHex.substring(0, 64);
        return pubKeyYPrefix + pubKeyX;
    }

    public static byte[] uncompressPoint(byte[] compressed) {
        return SECNamedCurves.getByName((String)"secp256k1").getCurve().decodePoint(compressed).getEncoded(false);
    }

    public static PublicKey getECDSAPublicKeyFromBytes(byte[] bytes) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("ECDSA", "BC");
            ECNamedCurveParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec((String)"secp256k1");
            ECNamedCurveSpec params = new ECNamedCurveSpec("secp256k1", ecParameterSpec.getCurve(), ecParameterSpec.getG(), ecParameterSpec.getN());
            ECPoint publicPoint = ECPointUtil.decodePoint((EllipticCurve)params.getCurve(), (byte[])bytes);
            ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(publicPoint, (ECParameterSpec)params);
            return keyFactory.generatePublic(pubKeySpec);
        }
        catch (Exception e) {
            throw new RemmeKeyException(e);
        }
    }

    public static void checkAddress(String address) {
        if (address == null || address.isEmpty()) {
            throw new RemmeValidationException("Address was not provided, please set the address");
        }
        if (!address.matches(Patterns.ADDRESS.getPattern())) {
            throw new RemmeValidationException("Given address is not a valid");
        }
    }

    public static void checkPublicKey(String publicKey) {
        if (publicKey == null || publicKey.isEmpty()) {
            throw new RemmeValidationException("Public Key was not provided, please set the address");
        }
        if (!publicKey.matches(Patterns.PUBLIC_KEY.getPattern())) {
            throw new RemmeValidationException("Given public key is not a valid");
        }
    }

    public static String certificateToPEM(Certificate certificate, boolean withPrivateKey) {
        StringWriter sw = new StringWriter();
        try (JcaPEMWriter pw = new JcaPEMWriter((Writer)sw);){
            pw.writeObject((Object)certificate.getCert());
            if (withPrivateKey) {
                pw.writeObject((Object)certificate.getPrivateKey());
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return sw.toString();
    }

    public static Certificate certificateFromPEM(String certificatePEM) {
        StringReader sr = new StringReader(certificatePEM);
        Certificate certificate = new Certificate();
        try {
            PemReader reader = new PemReader((Reader)sr);
            PemObject object = reader.readPemObject();
            if (object != null) {
                do {
                    if (object.getType().toUpperCase().contains("CERTIFICATE")) {
                        certificate.setCert((X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(object.getContent())));
                    }
                    if (!object.getType().toUpperCase().contains("PRIVATE")) continue;
                    certificate.setPrivateKey(Functions.getPrivateKeyFromBytesArray(KeyType.RSA, object.getContent()));
                } while ((object = reader.readPemObject()) != null);
            }
            return certificate;
        }
        catch (Exception e) {
            throw new RemmeKeyException(e);
        }
    }

    public static void checkSha256(String data) {
        if (data == null || !data.matches(Patterns.SHA256.getPattern())) {
            throw new RemmeValidationException("Value should be SHA-256");
        }
    }

    public static void checkSha512(String data) {
        if (data == null || !data.matches(Patterns.SHA512.getPattern())) {
            throw new RemmeValidationException("Value should be SHA-512");
        }
    }

    public static void checkSha(String data) {
        if (data == null || !data.matches(Patterns.SHA256.getPattern()) && !data.matches(Patterns.SHA512.getPattern())) {
            throw new RemmeValidationException("Value should be SHA-256 or SHA-512");
        }
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
        Security.addProvider((Provider)new EdDSASecurityProvider());
    }
}

