/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.httpd;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/09/28
 */
public class HTTPServer {

	private static final int PORT = 9393;
	private static final int BUFSIZE = 1024;

	private static Pattern PN1 = Pattern.compile(
			"(/([^\\?]+/)*([^/\\.\\?]+(\\.([^/\\.\\?]+))?)?)(\\?.*)?$");

	static HTTPLogger log = new HTTPLoggerFactory.L();

	private Map<SelectionKey, HTTPRequestContinuableParser> wait =
			new HashMap<SelectionKey, HTTPRequestContinuableParser>();
	private Map<SelectionKey, Ecrire> ecrire =
			new HashMap<SelectionKey, Ecrire>();
	private Map<String, HTTPDispatch> dispatchers =
			new HashMap<String, HTTPDispatch>();
	private Charset cset;
	private String root;

	/**
	 * 
	 * @param root
	 */
	public HTTPServer(String root) {
		this.root = root;
		this.cset = Charset.defaultCharset();
	}

	Ecrire dispatch(SelectionKey k, HTTPRequest q) throws IOException {
		HTTPDispatch d;
		Class<?> c;
		Matcher m;
		String s;

		log.finest("method: %s", q.getMethod());
		log.finest("path: %s", q.getPath());
		log.finest("version: %s", q.getVersion());
		try {
			s = q.getPath();
			if(!(m = PN1.matcher(s)).find()) {
				log.finest("400: %s", s);
				return HTTPUtils.MESSAGE400;
			} else if(m.group(3) == null) {
				// index.html
				log.finest("directory: %s", s);
				return new ResourceEcrire(root + s + "index.html");
			} else if(m.group(4) != null) {
				// resource
				log.finest("file: %s", s);
				return new ResourceEcrire(root + s);
			} else {
				// dynamic
				log.finest("dynamic: %s", s);
				s = (root + s).replace('/', '.');
				c = Class.forName(s);
				if((d = dispatchers.get(s)) == null) {
					d = (HTTPDispatch)c.newInstance();
					dispatchers.put(s, d);
				}
				return new DynamicEcrire(d, q, c);
			}
		} catch(Exception e) {
			log.finest("exception read(): %s", e.toString());
			return HTTPUtils.MESSAGE500;
		}
	}

	void accept(Selector s,
			ServerSocketChannel c) throws IOException {
		SocketChannel d;

		d = c.accept();
		d.configureBlocking(false);
		d.register(s, SelectionKey.OP_READ);
	}

	void read(SelectionKey k) throws IOException {
		SocketChannel c = (SocketChannel)k.channel();
		ByteBuffer b = ByteBuffer.allocate(BUFSIZE);
		HTTPRequestContinuableParser s = null;
		HTTPRequest q;
		Ecrire w;
		byte[] a;

		s = wait.get(k);
		while(c.read(b) > 0) {
			if(s == null) {
				s = new HTTPRequestContinuableParser(cset);
				wait.put(k, s);
			}
			b.flip();  a = new byte[b.limit()];  b.get(a);
			log.finest("\"%s\"", new String(a, cset));
			s.readPartial(new ByteArrayInputStream(a));
		}

		if(s == null) {
			k.cancel();  c.close();
		} else {
			log.finest("read end of request");
			wait.remove(k);
			q = s.getRequest();
			w = dispatch(k, q);
			k.interestOps(k.interestOps() | SelectionKey.OP_WRITE);
			ecrire.put(k, w);
			log.finest("ready to response");
		}
	}

	void write(SelectionKey k) throws IOException {
		SocketChannel c = (SocketChannel)k.channel();
		Ecrire w;

		try {
			if((w = ecrire.get(k)) != null) {
				log.finest("ecrire found");
				w.write(c);
			} else {
				log.finest("ecrire not found");
			}
			log.finest("responded");
		} finally {
			k.cancel();  c.close();
		}
	}

	public void serve() throws IOException {
		ServerSocketChannel c = null;
		Iterator<SelectionKey> i;
		SelectionKey k;
		Selector s;

		try {
			s = Selector.open();
			c = ServerSocketChannel.open();
			c.configureBlocking(false);
			c.socket().bind(new InetSocketAddress(PORT));
			c.register(s, SelectionKey.OP_ACCEPT);
			while(s.select() > 0) {
				try {
					i = s.selectedKeys().iterator();
					while(i.hasNext()) {
						k = i.next();  i.remove();
						if(!k.isValid()) {
							// do nothing
						} else if(k.isAcceptable()) {
							accept(s, (ServerSocketChannel)k.channel());
						} else if(k.isReadable()) {
							log.finest("begin to read");
							read(k);
						} else if(k.isWritable()) {
							log.finest("begin to write");
							write(k);
						}
					}
				} catch(Exception e) {
					e.printStackTrace(System.err);
				}
			}
		} finally {
			if(c != null && c.isOpen())  c.close();
		}
	}

}
