package net.morilib.db.jdbc;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import net.morilib.db.expr.RelationExpression;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.Rational;
import net.morilib.db.relations.Relation;
import net.morilib.db.sqlcs.dml.SqlDml;
import net.morilib.db.sqlcs.dml.SqlSetExpression;

public class RelationsPreparedStatement extends RelationsStatement
implements PreparedStatement {

	private Object stmt;
	private SortedMap<Integer, Object> param;

	RelationsPreparedStatement(RelationsConnection c, Object s) {
		super(c);
		stmt = s;
		param = new TreeMap<Integer, Object>();
	}

	private List<Object> paramToList() throws SQLException {
		List<Object> l = new ArrayList<Object>();
		int j;

		if(param.size() > 0) {
			j = param.lastKey().intValue();
			for(int i = 0; i <= j; i++) {
				if(param.containsKey(i)) {
					l.add(param.get(i));
				} else {
					throw ErrorBundle.getDefault(20006);
				}
			}
		}
		return l;
	}

	@Override
	public ResultSet executeQuery() throws SQLException {
		Relation r;

		if(stmt instanceof SqlSetExpression) {
			try {
				r = conn.engine.visit((SqlSetExpression)stmt,
						paramToList());
				return new RelationsResultSet(this, r);
			} catch (IOException e) {
				throw ErrorBundle.getIODefault(e);
			}
		} else {
			throw ErrorBundle.getDefault(10038);
		}
	}

	@Override
	public int executeUpdate() throws SQLException {
		Object r;

		if(stmt instanceof SqlSetExpression) {
			throw ErrorBundle.getDefault(20005);
		} else {
			try {
				r = conn.engine.visit(stmt,
						paramToList());
				return r instanceof Number ?
						((Number)r).intValue() : 0;
			} catch (IOException e) {
				throw ErrorBundle.getIODefault(e);
			}
		}
	}

	private void checkparam(int p) throws SQLException {
		checkclose();
		if(!(stmt instanceof SqlDml)) {
			throw ErrorBundle.getDefault(20007);
		} else if(p < 1 || p > ((SqlDml)stmt).getMaxParameter()) {
			throw ErrorBundle.getDefault(20007);
		}
	}

	@Override
	public void setNull(int parameterIndex,
			int sqlType) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, "");
	}

	@Override
	public void setBoolean(int parameterIndex,
			boolean x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, x ? Rational.ONE : Rational.ZERO);
	}

	@Override
	public void setByte(int parameterIndex,
			byte x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setShort(int parameterIndex,
			short x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setInt(int parameterIndex,
			int x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setLong(int parameterIndex,
			long x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setFloat(int parameterIndex,
			float x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setDouble(int parameterIndex,
			double x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	@Override
	public void setBigDecimal(int parameterIndex,
			BigDecimal x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, Rational.valueOf(x));
	}

	private String _esc(String s) {
		return s;
	}

	private void setstr(int p, String s) {
		param.put(p, _esc(s));
	}

	@Override
	public void setString(int parameterIndex,
			String x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, x);
	}

	@Override
	public void setBytes(int parameterIndex,
			byte[] x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, new String(x));
	}

	@Override
	public void setDate(int parameterIndex,
			Date x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, new java.util.Date(x.getTime()));
	}

	@Override
	public void setTime(int parameterIndex,
			Time x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, new java.util.Date(x.getTime()));
	}

	@Override
	public void setTimestamp(int parameterIndex,
			Timestamp x) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, new java.util.Date(x.getTime()));
	}

	private String tostr(Reader x) throws SQLException {
		StringBuffer b = new StringBuffer();
		char[] a = new char[1024];
		int l;

		try {
			while((l = x.read(a)) >= 0) {
				b.append(new String(a, 0, l));
			}
			return b.toString();
		} catch (IOException e) {
			throw ErrorBundle.getIODefault(e);
		}
	}

	private String tostr(InputStream x) throws SQLException {
		return tostr(new InputStreamReader(x));
	}

	private String tostr(Reader x, int len) throws SQLException {
		StringBuffer b = new StringBuffer();
		char[] a = new char[len];
		int l;

		try {
			if((l = x.read(a)) >= 0) {
				b.append(new String(a, 0, l));
			}
			return b.toString();
		} catch (IOException e) {
			throw ErrorBundle.getIODefault(e);
		}
	}

	private String tostr(InputStream x, int len) throws SQLException {
		StringBuffer b = new StringBuffer();
		byte[] a = new byte[len];
		int l;

		try {
			if((l = x.read(a)) >= 0) {
				b.append(new String(a, 0, l));
			}
			return b.toString();
		} catch (IOException e) {
			throw ErrorBundle.getIODefault(e);
		}
	}

	@Override
	public void setAsciiStream(int parameterIndex, InputStream x,
			int length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setUnicodeStream(int parameterIndex, InputStream x,
			int length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setBinaryStream(int parameterIndex, InputStream x,
			int length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void clearParameters() throws SQLException {
		param.clear();
	}

	@Override
	public void setObject(int parameterIndex, Object x,
			int targetSqlType) throws SQLException {
		setObject(parameterIndex, x);
	}

	@Override
	public void setObject(int parameterIndex,
			Object x) throws SQLException {
		Object o;

		if(x == null) {
			o = RelationExpression.NULL;
		} else if(x instanceof BigDecimal) {
			o = Rational.valueOf((BigDecimal)x);
		} else if(x instanceof BigInteger) {
			o = Rational.valueOf((BigInteger)x);
		} else if(x instanceof Double) {
			o = Rational.valueOf(((Double)x).doubleValue());
		} else if(x instanceof Float) {
			o = Rational.valueOf(((Float)x).doubleValue());
		} else if(x instanceof Number) {
			o = Rational.valueOf(((Number)x).longValue());
		} else if(x instanceof java.util.Date) {
			o = new java.util.Date(((java.util.Date)x).getTime());
		} else {
			o = _esc(x.toString());
		}
		param.put(parameterIndex - 1, o);
	}

	@Override
	public boolean execute() throws SQLException {
		Object o;

		checkclose();
		try {
			o = conn.engine.visit(stmt, paramToList());
			if(o instanceof Relation) {
				select0 = (Relation)o;
				return true;
			} else if(o instanceof Number) {
				count0 = ((Number)o).intValue();
				return false;
			} else {
				count0 = 0;
				return false;
			}
		} catch (IOException e) {
			throw ErrorBundle.getIODefault(e);
		}
	}

	@Override
	public void addBatch() throws SQLException {
		// TODO
		// ignore it
	}

	@Override
	public void setCharacterStream(int parameterIndex, Reader x,
			int length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setRef(int parameterIndex, Ref x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setBlob(int parameterIndex,
			Blob x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setClob(int parameterIndex,
			Clob x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setArray(int parameterIndex,
			Array x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public ResultSetMetaData getMetaData() throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setDate(int parameterIndex, Date x,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(x);
		setDate(parameterIndex, new Date(c.getTimeInMillis()));
	}

	@Override
	public void setTime(int parameterIndex, Time x,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(x);
		setDate(parameterIndex, new Date(c.getTimeInMillis()));
	}

	@Override
	public void setTimestamp(int parameterIndex, Timestamp x,
			Calendar cal) throws SQLException {
		Calendar c;

		(c = (Calendar)cal.clone()).setTime(x);
		setDate(parameterIndex, new Date(c.getTimeInMillis()));
	}

	@Override
	public void setNull(int parameterIndex, int sqlType,
			String typeName) throws SQLException {
		checkparam(parameterIndex);
		param.put(parameterIndex - 1, "");
	}

	@Override
	public void setURL(int parameterIndex,
			URL x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public ParameterMetaData getParameterMetaData() throws SQLException {
		if(stmt instanceof SqlDml) {
			return new RelationsParameterMetaData(
					((SqlDml)stmt).getMaxParameter());
		} else {
			return new RelationsParameterMetaData(0);
		}
	}

	@Override
	public void setRowId(int parameterIndex,
			RowId x) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setNString(int parameterIndex,
			String x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, x);
	}

	@Override
	public void setNCharacterStream(int parameterIndex, Reader x,
			long length) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setNClob(int parameterIndex,
			NClob value) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setClob(int parameterIndex, Reader reader,
			long length) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setBlob(int parameterIndex, InputStream inputStream,
			long length) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setNClob(int parameterIndex, Reader reader,
			long length) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setSQLXML(int parameterIndex,
			SQLXML xmlObject) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setObject(int parameterIndex, Object x,
			int targetSqlType,
			int scaleOrLength) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setAsciiStream(int parameterIndex, InputStream x,
			long length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x, (int)length));
	}

	@Override
	public void setBinaryStream(int parameterIndex, InputStream x,
			long length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x, (int)length));
	}

	@Override
	public void setCharacterStream(int parameterIndex, Reader x,
			long length) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x, (int)length));
	}

	@Override
	public void setAsciiStream(int parameterIndex,
			InputStream x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setBinaryStream(int parameterIndex,
			InputStream x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setCharacterStream(int parameterIndex,
			Reader x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setNCharacterStream(int parameterIndex,
			Reader x) throws SQLException {
		checkparam(parameterIndex);
		setstr(parameterIndex - 1, tostr(x));
	}

	@Override
	public void setClob(int parameterIndex,
			Reader reader) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setBlob(int parameterIndex,
			InputStream inputStream) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public void setNClob(int parameterIndex,
			Reader reader) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

}
