unit UMsnPluginMain;

interface

uses
  Windows, Messages, SysUtils, Classes, UNsmConsts, UNsmTypes, UMsnMsgr,
  UMemberList, UNsmProtocolPluginMain, UMsnUtils, StrUtils, UNsmInfo,
  UWideUtils, USimpleTimer, Md5, UCTimeUtils, ShellApi;

const
  // vgR
  NMP_PROTOCOLNAME = 'MSN';

  // vOC
  NMP_INFO_APIVER       = NSM_API_VERSION;
  NMP_INFO_MODULENAME   = NMM_PROTOCOL + '/' + NMP_PROTOCOLNAME;
  NMP_INFO_TITLE        = 'MSN Messenger Protocol';
  NMP_INFO_DESCRIPTION  = 'MSN Messenger / Windows Messenger Protocol for Regnessem';
  NMP_INFO_AUTHOR       = 'Yamaneko';
  NMP_INFO_COPYRIGHT    = 'Copyright (c) 2001-2003 Yamaneko';
  NMP_INFO_PLUGINVER    = '0.5.8';

type
  // CNX
  TMsnPluginMain = class(TNsmProtocolPluginMain)
  private
    FConnections: TMsnConnectionList;
    FSessions: TMsnSessionList;
    FNsmInfo: TNsmSetInfo;
    FPingTimer: TSimpleTimer;

    procedure ApplyConfig;
    procedure ShowNotify(Text: WideString; ImageIndex: Integer;
      CallbackProc: TNotifyClickCallback = nil; Data: Integer = 0);
    procedure SendMessageEx(Session: TMsnSession; Msg: WideString);
    procedure SetConnectionCaption(Connection: TMsnConnection);
    procedure SetSessionCaption(Session: TMsnSession);

    procedure ConMemberAddition(Sender: TObject; ListKind: TListKind;
      Member: TMsnMember);
    procedure ConMemberDeletion(Sender: TObject; ListKind: TListKind;
      Member: TMsnMember);
    procedure ConMemberStatusChange(Sender: TObject;  Member: TMsnMemberBase;
      OldStatus: TMsnMemberStatus; InitList: Boolean);
    procedure ConMemberNameChange(Sender: TObject;  Member: TMsnMemberBase);
    procedure ConMemberGroupChange(Sender: TObject;  Member: TMsnMemberBase);
    procedure ConMemberListUpdated(Sender: TObject; List: TListKind);
    procedure ConGroupAddition(Sender: TObject; Group: TMsnGroup);
    procedure ConGroupDeletion(Sender: TObject; Group: TMsnGroup);
    procedure ConGroupNameChange(Sender: TObject; Group: TMsnGroup);

    procedure ConSignOut(Sender: TObject; SignOutType: TSignOutType);
    procedure ConLog(Sender: TObject;  LogStr: String);
    procedure ConError(Sender: TObject;  ErrorKind: TErrorKind;
      ErrorCode: Integer);
    procedure ConSwitchBoard(Sender: TObject;  TrID: Integer; SBAddress,
      Cookie: String);
    procedure ConCalled(Sender: TObject;  SessionID, SBAddress, Cookie,
      CallingUserAccount: String; CallingUserName: WideString);
    procedure ConNewMail(Sender: TObject; FromName: WideString; FromAddr: String);
    procedure ConUnreadMailChange(Sender: TObject; Init: Boolean);
    procedure ConUrl(Sender: TObject; rru, passport: String);

    procedure SesConnect(Sender: TObject);
    procedure SesDisconnect(Sender: TObject);
    procedure SesJoinMember(Sender: TObject; Member: TMsnMemberBase);
    procedure SesByeMember(Sender: TObject; Member: TMsnMemberBase);
    procedure SesMemberListChange(Sender: TObject);
    procedure SesReceiveMessage(Sender: TObject; Header: UTF8String;
      FromAccount: String; FromName, Msg: WideString);
  protected
    procedure DoInitialize; override;
    procedure DoTerminate; override;

    function DoShowOptionDialog(Parent: HWND): Integer; override;
    function DoConnect(LoginInfo: TLogInInfo): Integer; override;
    function DoDisconnect(CHandle: HNsmConnection): Integer; override;
    function DoOpenSession(CHandle: HNsmConnection; OSInfo: TOpenSessionInfo): Integer; override;
    function DoCloseSession(SHandle: HNsmSession): Integer; override;
    function DoInviteMember(SHandle: HNsmSession; Account: String): Integer; override;
    function DoSendMessage(SHandle: HNsmSession; MsgInfo: TMessageInfo): Integer; override;
    function DoChangeStatus(CHandle: HNsmConnection; StatusInfo: TUserStatusInfo): Integer; override;
    function DoChangeUserName(CHandle: HNsmConnection; NameInfo: TUserNameInfo): Integer; override;
    function DoAddMember(CHandle: HNsmConnection; AddMemberInfo: TAddMemberInfo): Integer; override;
    function DoRemoveMember(CHandle: HNsmConnection; RemoveMemberInfo: TRemoveMemberInfo): Integer; override;
    function DoChangeMemberGroup(CHandle: HNsmConnection; MemberGroupInfo: TMemberGroupInfo): Integer; override;
    function DoAddGroup(CHandle: HNsmConnection; GroupName: WideString): Integer; override;
    function DoRemoveGroup(CHandle: HNsmConnection; GroupId: Integer): Integer; override;
    function DoChangeGroupName(CHandle: HNsmConnection; GroupNameInfo: TGroupNameInfo): Integer; override;
  public
    constructor Create; override;
    destructor Destroy; override;

    property Connections: TMsnConnectionList read FConnections;
    property Sessions: TMsnSessionList read FSessions;
  end;

implementation

uses
  UConfig, UNsmPlugin, UDlgOption;

var
  PluginMain: TMsnPluginMain;

const
  TEMP_HTMLNAME = 'temp.html';

procedure PingTimerProc(hWnd: HWND; uMsg, idEvent: Cardinal; dwTime: DWORD); stdcall; forward;
procedure OnMailNotifyClick(nData: Integer); stdcall; forward;

// TMsnPluginMain --------------------------------------------------------------

constructor TMsnPluginMain.Create;
begin
  inherited;
  FConnections := TMsnConnectionList.Create;
  FSessions := TMsnSessionList.Create;
  FPingTimer := TSimpleTimer.Create;
  FPingTimer.OnTimer := PingTimerProc;

  PluginInfo[NMPI_APIVER]        := NMP_INFO_APIVER;
  PluginInfo[NMPI_MODULENAME]    := NMP_INFO_MODULENAME;
  PluginInfo[NMPI_TITLE]         := NMP_INFO_TITLE;
  PluginInfo[NMPI_DESCRIPTION]   := NMP_INFO_DESCRIPTION;
  PluginInfo[NMPI_AUTHOR]        := NMP_INFO_AUTHOR;
  PluginInfo[NMPI_COPYRIGHT]     := NMP_INFO_COPYRIGHT;
  PluginInfo[NMPI_PLUGINVER]     := NMP_INFO_PLUGINVER;
end;

destructor TMsnPluginMain.Destroy;
begin
  FConnections.Free;
  FSessions.Free;
  FPingTimer.Free;
  if FileExists(WorkDir + TEMP_HTMLNAME) then
    DeleteFile(WorkDir + TEMP_HTMLNAME);
  inherited;
end;

procedure TMsnPluginMain.DoInitialize;
begin
  inherited;
  FNsmInfo := TNsmSetInfo.Create(InitInfo);
  Config.LoadFromFile(IniFileName);
  ApplyConfig;
end;

procedure TMsnPluginMain.DoTerminate;
begin
  inherited;
  Config.SaveToFile(IniFileName);
  FNsmInfo.Free;
end;

function TMsnPluginMain.DoShowOptionDialog(Parent: HWND): Integer;
begin
  Result := ShowOptionDialog(Parent);
  if Result = IDOK then
    PluginMain.ApplyConfig;
end;

// RlNV쐬
function TMsnPluginMain.DoConnect(LoginInfo: TLogInInfo): Integer;
var
  CHandle: HNsmConnection;
  Connection: TMsnConnection;
begin
  CHandle := CallService(NMS_SYSTEM_CONNECTION_CREATE,
      Cardinal(PChar(NMP_PROTOCOLNAME)), 0);
  if CHandle > 0 then
  begin
    FNsmInfo.SetConnectionInfo(CHandle, NMCI_STATUS, NMCS_CONNECTING);
    FNsmInfo.SetConnectionInfo(CHandle, NMCI_USER_ACCOUNT, String(LoginInfo.lpAccount));
    FNsmInfo.SetConnectionInfo(CHandle, NMCI_USER_NAME, String(LoginInfo.lpAccount));
    FNsmInfo.SetConnectionInfo(CHandle, NMCI_USER_STATUS, NMST_OFFLINE);
    Connection := FConnections.Add;
    with Connection do
    begin
      NsmHandle := CHandle;
      DsHost := Config.MsnHost;
      DsPort := Config.MsnPort;
      OnMemberAddition := ConMemberAddition;
      OnMemberDeletion := ConMemberDeletion;
      OnMemberStatusChange := ConMemberStatusChange;
      OnMemberNameChange := ConMemberNameChange;
      OnMemberGroupChange := ConMemberGroupChange;
      OnGroupAddition := ConGroupAddition;
      OnGroupDeletion := ConGroupDeletion;
      OnGroupNameChange := ConGroupNameChange;
      OnSignOut := ConSignOut;
      OnLog := ConLog;
      OnError := ConError;
      OnSwitchBoard := ConSwitchBoard;
      OnCalled := ConCalled;
      OnMemberListUpdated := ConMemberListUpdated;
      OnNewMail := ConNewMail;
      OnUnreadMailChange := ConUnreadMailChange;
      OnUrl := ConUrl;

      SignIn(AnsiLowerCase(LoginInfo.lpAccount), LoginInfo.lpPassword, '',
          TMsnMemberStatus(LoginInfo.nStatus));
    end;
    SetConnectionCaption(Connection);
  end;
  Result := CHandle;
end;

// RlNV폜
function TMsnPluginMain.DoDisconnect(CHandle: HNsmConnection): Integer;
var
  Idx: Integer;
begin
  Idx := FConnections.IndexOfNsmHandle(CHandle);
  if Idx > -1 then
  begin
    FConnections.Delete(Idx);
    CallService(NMS_SYSTEM_CONNECTION_DELETE, CHandle, 0);
  end;
  Result := Idx + 1;
end;

// ZbV쐬
function TMsnPluginMain.DoOpenSession(CHandle: HNsmConnection; OSInfo: TOpenSessionInfo): Integer;
var
  SHandle: HNsmSession;
  Connection: TMsnConnection;
  Session: TMsnSession;
  ToMember: TMsnMember;
  cIdx: Integer;
begin
  Result := 0;
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    Connection := FConnections[cIdx];
    ToMember := Connection.Members.Find(OSInfo.lpToAccount);
    if AnsiSameText(Connection.User.Account, OSInfo.lpToAccount) then
    begin
      ShowNotify('ɃbZ[W𑗐M邱Ƃ͂ł܂B', NMNI_EXCLAMATION);
      Exit;
    end else
    if Connection.User.Status = usHDN then
    begin
      ShowNotify('sɐݒ肳Ă̂ŃbZ[W𑗐Mł܂B', NMNI_EXCLAMATION);
      Exit;
    end else
    if Assigned(ToMember) and (ToMember.Status in [usFLN, usHDN]) then
    begin
      ShowNotify('ItC̃oɃbZ[W𑗐M邱Ƃ͂ł܂B', NMNI_EXCLAMATION);
      Exit;
    end;

    SHandle := CallService(NMS_SYSTEM_SESSION_CREATE, CHandle, 0);
    if SHandle > 0 then
    begin
      FNsmInfo.SetSessionInfo(SHandle, NMSI_STATUS, NMCS_CONNECTING);
      Session := FSessions.Add;
      with Session do
      begin
        User.Assign(Connection.User);
        NsmHandle := SHandle;
        RequestID := Connection.SwitchBoardRequest;
        OnConnect := SesConnect;
        OnDisConnect := SesDisconnect;
        OnJoinMember := SesJoinMember;
        OnByeMember := SesByeMember;
        OnMemberListChange := SesMemberListChange;
        OnRecieveMessage := SesReceiveMessage;
        OnLog := ConLog;
        OnError := ConError;
        // o
        CallMember(OSInfo.lpToAccount);
      end;
      SetSessionCaption(Session);
      CallService(NMS_SYSTEM_SESSION_MEMBERS_ADD,
          Session.NsmHandle, Cardinal(PChar(OSInfo.lpToAccount)));
      FNsmInfo.SetSessionMemberInfo(Session.NsmHandle,
          OSInfo.lpToAccount, NMMI_NAME, OSInfo.lpToAccount);
      FNsmInfo.SetSessionMemberInfo(Session.NsmHandle,
          OSInfo.lpToAccount, NMMI_STATUS, NMST_OFFLINE);
    end;
    Result := SHandle;
  end;
end;

// ZbV폜
function TMsnPluginMain.DoCloseSession(SHandle: HNsmSession): Integer;
var
  Idx: Integer;
begin
  Idx := FSessions.IndexOfNsmHandle(SHandle);
  if Idx > -1 then
  begin
    FSessions.Delete(Idx);
    CallService(NMS_SYSTEM_SESSION_DELETE, SHandle, 0);
  end;
  Result := Idx + 1;
end;

// o
function TMsnPluginMain.DoInviteMember(SHandle: HNsmSession; Account: String): Integer;
var
  Idx: Integer;
begin
  Idx := FSessions.IndexOfNsmHandle(SHandle);
  if Idx > -1 then
  begin
    if AnsiSameText(FSessions[Idx].User.Account, Account) then
      ShowNotify('ɃbZ[W𑗐M邱Ƃ͂ł܂B', NMNI_EXCLAMATION)
    else
      FSessions[Idx].CallMember(Account);
  end;
  Result := Idx + 1;
end;

// bZ[W𑗐M
function TMsnPluginMain.DoSendMessage(SHandle: HNsmSession; MsgInfo: TMessageInfo): Integer;
  function MakeFontAttrStr(Attr: TTextAttributeInfo): String;
  var
    EF, CO, FN, CS, FS, BC: String;
  begin
    if (NMFS_BOLD and Attr.nStyles) <> 0 then
      EF := EF + 'B';
    if (NMFS_ITALIC and Attr.nStyles) <> 0  then
      EF := EF + 'I';
    if (NMFS_UNDERLINE and Attr.nStyles) <> 0  then
      EF := EF + 'U';
    if (NMFS_STRIKEOUT and Attr.nStyles) <> 0  then
      EF := EF + 'S';

    if Attr.nFontColor <> -1 then
      CO := IntToHex(Attr.nFontColor, 1);
    if Attr.lpFontName <> nil then
      FN := UrlEncode(Utf8Encode(Attr.lpFontName));
    if Attr.nCharSet <> -1 then
      CS := IntToHex(Attr.nCharSet, 1);
    Result := Format('X-MMS-IM-Format: FN=%s; EF=%s; CO=%s; CS=%s;',
        [FN, EF, CO, CS]);

    // NSMSGSgtH[}bgǉ
    if Attr.nFontSize <> -1 then
      FS := IntToStr(Attr.nFontSize);
    if Attr.nBgColor <> -1 then
      BC := IntToHex(Attr.nBgColor, 1);
    Result := Result + #13#10 + Format('X-NSM-IM-Format: FS=%s; BC=%s;',
        [FS, BC]);
  end;
  function AddMsgHeader(const Msg: WideString; Attr: TTextAttributeInfo): WideString;
  var
    Head: WideString;
  begin
    Head := 'MIME-Version: 1.0'#13#10 +
            'Content-Type: text/plain; charset=UTF-8'#13#10 +
            MakeFontAttrStr(Attr) + #13#10#13#10;
    Result := Head + Msg;
  end;
var
  Idx: Integer;
begin
  Idx := FSessions.IndexOfNsmHandle(SHandle);
  if Idx > -1 then
  begin
    SendMessageEx(FSessions[Idx], AddMsgHeader(MsgInfo.lpBody,
        PTextAttributeInfo(MsgInfo.lpTextAttribute)^));
  end;
  Result := Idx + 1;
end;

function TMsnPluginMain.DoChangeStatus(CHandle: HNsmConnection; StatusInfo: TUserStatusInfo): Integer;
var
  cIdx: Integer;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    FConnections[cIdx].ChangeUserStatus(TMsnMemberStatus(StatusInfo.nStatus));
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoChangeUserName(CHandle: HNsmConnection; NameInfo:
  TUserNameInfo): Integer;
var
  cIdx: Integer;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    FConnections[cIdx].RenameMember(FConnections[cIdx].User.Account,
        NameInfo.lpName);
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoAddMember(CHandle: HNsmConnection; AddMemberInfo: TAddMemberInfo): Integer;
var
  cIdx: Integer;
  Target: String;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    Target := AddMemberInfo.lpAccount;
    FConnections[cIdx].AddMember(TListKind(AddMemberInfo.nListKind), Target);
    
    // ɋ֎~ĂȂ΋Xgɒǉ
    if (AddMemberInfo.nListKind = NMLK_FORWARDLIST) then
      if not FConnections[cIdx].AllowMembers.Contains(Target) and
         not FConnections[cIdx].BlockMembers.Contains(Target) then
        FConnections[cIdx].AddMember(lkAL, Target);
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoRemoveMember(CHandle: HNsmConnection; RemoveMemberInfo: TRemoveMemberInfo): Integer;
var
  cIdx: Integer;
  Target: String;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    Target := RemoveMemberInfo.lpAccount;
    FConnections[cIdx].RemoveMember(TListKind(RemoveMemberInfo.nListKind), Target);

    // ΌXgȂꍇAXg폜
    if RemoveMemberInfo.nListKind = NMLK_FORWARDLIST then
      if not FConnections[cIdx].ReverceMembers.Contains(Target) and
             FConnections[cIdx].AllowMembers.Contains(Target) then
        FConnections[cIdx].RemoveMember(lkAL, Target);
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoChangeMemberGroup(CHandle: HNsmConnection; MemberGroupInfo: TMemberGroupInfo): Integer;
var
  cIdx, I: Integer;
  Member: TMsnMember;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    Member := FConnections[cIdx].Members.Find(MemberGroupInfo.lpAccount);
    if Assigned(Member) then
    begin
      FConnections[cIdx].AddMember(lkFL, Member.Account, MemberGroupInfo.nGroupId);    
      for I := 0 to Member.Groups.Count - 1 do
        FConnections[cIdx].RemoveMember(lkFL, Member.Account, Member.Groups[I]);
    end;
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoAddGroup(CHandle: HNsmConnection; GroupName: WideString): Integer;
var
  cIdx: Integer;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    FConnections[cIdx].AddGroup(GroupName);
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoRemoveGroup(CHandle: HNsmConnection; GroupId: Integer): Integer;
var
  cIdx: Integer;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    FConnections[cIdx].RemoveGroup(GroupId);
  end;
  Result := cIdx + 1;
end;

function TMsnPluginMain.DoChangeGroupName(CHandle: HNsmConnection; GroupNameInfo: TGroupNameInfo): Integer;
var
  cIdx: Integer;
begin
  cIdx := FConnections.IndexOfNsmHandle(CHandle);
  if cIdx >  -1 then
  begin
    FConnections[cIdx].RenameGroup(GroupNameInfo.nGroupId, GroupNameInfo.lpName);
  end;
  Result := cIdx + 1;
end;

// private ֐ ----------------------------------------------------------------

procedure TMsnPluginMain.ApplyConfig;
begin
  FPingTimer.Active := False;
  FPingTimer.Interval := Config.PingInterval * 1000 * 60;
  FPingTimer.Active := Config.SendPing;
end;

// ėpbZ[WM
procedure TMsnPluginMain.SendMessageEx(Session: TMsnSession; Msg: WideString);
var
  Idx: Integer;
begin
  // s݂Ȃ珵
  if (Session.SignInStage = ssSignIn) and (Session.Members.Count = 0) then
  begin
    Session.SendMessage(Msg);
    Session.CallReservedMembers;
  end else
  // ڑȂڑv
  if (Session.SignInStage = ssUnConnect) then
  begin
    Session.SendMessage(Msg);
    Idx := FConnections.IndexOfUser(Session.User.Account);
    if Idx > -1 then
      Session.RequestID := FConnections[Idx].SwitchBoardRequest;
    FNsmInfo.SetSessionInfo(Session.NsmHandle, NMSI_STATUS, NMCS_CONNECTING);
  end else
  // ʏ푗M
    Session.SendMessage(Msg);
end;

procedure TMsnPluginMain.ShowNotify(Text: WideString; ImageIndex: Integer;
  CallbackProc: TNotifyClickCallback = nil; Data: Integer = 0);
var
  NtfyInfo: TNotifyInfo;
begin
  with NtfyInfo do
  begin
    cbSize := SizeOf(TNotifyInfo);
    lpText := PWideChar(Text);
    nTimeToLive := -1;
    nIcon := ImageIndex;
    lpLinkTo := nil;
    lpOnClick := CallbackProc;
    nData := Data;
  end;
  CallService(NMS_UI_NOTIFY, Cardinal(@NtfyInfo), 0);
end;

procedure TMsnPluginMain.SetConnectionCaption(Connection: TMsnConnection);
var
  Caption: WideString;
begin
  Caption := Connection.User.Name;
  if Length(Caption) = 0 then
    Caption := WideFormat('%s(%d)', [NMP_PROTOCOLNAME, FConnections.Count]);
  FNsmInfo.SetConnectionInfo(Connection.NsmHandle, NMCI_CAPTION, Caption);
end;

procedure TMsnPluginMain.SetSessionCaption(Session: TMsnSession);
var
  Caption: WideString;
  I: Integer;
begin
  for I := 0 to Session.Members.Count - 1 do
  begin
    if Length(Caption) > 0 then
      Caption := Caption + ', ';
    Caption := Caption + Session.Members[I].Name;
  end;
  if Length(Caption) = 0 then
    Caption := WideFormat('%s(%d)', [NMP_PROTOCOLNAME, FSessions.Count]);
  FNsmInfo.SetSessionInfo(Session.NsmHandle, NMSI_CAPTION, Caption);
end;

// TMsnConnection Object ̃Cxgnh ------------------------------------

procedure TMsnPluginMain.ConMemberAddition(Sender: TObject; ListKind: TListKind;
  Member: TMsnMember);
var
  AMInfo: TAddMemberInfo;
begin
  with AMInfo do
  begin
    cbSize := SizeOf(TAddMemberInfo);
    nListKind := Ord(ListKind);
    lpAccount := PChar(Member.Account);
  end;
  CallService(NMS_SYSTEM_CONNECTION_MEMBERS_ADD,
      TMsnConnection(Sender).NsmHandle, Cardinal(@AMInfo));
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
      NMMI_NAME, Member.Name, Ord(ListKind));
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
      NMMI_STATUS, Ord(Member.Status), Ord(ListKind));
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
      NMMI_GROUPID, Member.Groups.Last, Ord(ListKind));
end;

procedure TMsnPluginMain.ConMemberDeletion(Sender: TObject; ListKind: TListKind;
  Member: TMsnMember);
var
  RMInfo: TRemoveMemberInfo;
begin
  with RMInfo do
  begin
    cbSize := SizeOf(TRemoveMemberInfo);
    nListKind := Ord(ListKind);
    lpAccount := PChar(Member.Account);
  end;

  CallService(NMS_SYSTEM_CONNECTION_MEMBERS_REMOVE,
      TMsnConnection(Sender).NsmHandle, Cardinal(@RMInfo));
end;

procedure TMsnPluginMain.ConMemberStatusChange(Sender: TObject;
  Member: TMsnMemberBase; OldStatus: TMsnMemberStatus; InitList: Boolean);
var
  Flags: Integer;
begin
  Flags := 0;
  if InitList then
    Flags := NMIF_INITLIST;
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
    NMMI_NAME, Member.Name, NMLK_FORWARDLIST, Flags);
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
    NMMI_STATUS, Ord(Member.Status), NMLK_FORWARDLIST, Flags);
  if Member = TMsnMember(TMsnConnection(Sender).User) then
  begin
    FNsmInfo.SetConnectionInfo(TMsnConnection(Sender).NsmHandle, NMCI_USER_STATUS,
      Ord(Member.Status));
  end;
end;

procedure TMsnPluginMain.ConMemberNameChange(Sender: TObject;
  Member: TMsnMemberBase);
begin
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
    NMMI_NAME, Member.Name, NMLK_FORWARDLIST);

  if Member.Account = TMsnConnection(Sender).User.Account then
    FNsmInfo.SetConnectionInfo(TMsnConnection(Sender).NsmHandle, NMCI_USER_NAME,
      Member.Name);
  SetConnectionCaption(TMsnConnection(Sender));
end;

procedure TMsnPluginMain.ConMemberListUpdated(Sender: TObject; List: TListKind);
begin
  if TMsnConnection(Sender).Members.Updated and
     TMsnConnection(Sender).ReverceMembers.Updated and
     TMsnConnection(Sender).AllowMembers.Updated and
     TMsnConnection(Sender).BlockMembers.Updated then
  begin
    FNsmInfo.SetConnectionInfo(TMsnConnection(Sender).NsmHandle, NMCI_STATUS,
      NMCS_CONNECTED);
    FNsmInfo.SetConnectionInfo(TMsnConnection(Sender).NsmHandle, NMCI_USER_NAME,
      TMsnConnection(Sender).User.Name);
    SetConnectionCaption(TMsnConnection(Sender));
  end;
end;

procedure TMsnPluginMain.ConGroupAddition(Sender: TObject; Group: TMsnGroup);
begin
  CallService(NMS_SYSTEM_CONNECTION_GROUPS_ADD,
      TMsnConnection(Sender).NsmHandle, Group.Id);
  FNsmInfo.SetGroupInfo(TMsnConnection(Sender).NsmHandle, Group.Id,
      NMGI_NAME, Group.Name);
  FNsmInfo.SetGroupInfo(TMsnConnection(Sender).NsmHandle, Group.Id,
      NMGI_EXPANDED, True);
end;

procedure TMsnPluginMain.ConGroupDeletion(Sender: TObject; Group: TMsnGroup);
begin
  CallService(NMS_SYSTEM_CONNECTION_GROUPS_REMOVE,
      TMsnConnection(Sender).NsmHandle, Group.Id);
end;

procedure TMsnPluginMain.ConGroupNameChange(Sender: TObject; Group: TMsnGroup);
begin
  FNsmInfo.SetGroupInfo(TMsnConnection(Sender).NsmHandle, Group.Id,
    NMGI_NAME, Group.Name);
end;

procedure TMsnPluginMain.ConMemberGroupChange(Sender: TObject;
  Member: TMsnMemberBase);
begin
  FNsmInfo.SetMemberInfo(TMsnConnection(Sender).NsmHandle, Member.Account,
    NMMI_GROUPID, TMsnMember(Member).Groups[0], NMLK_FORWARDLIST);
end;

procedure TMsnPluginMain.ConSignOut(Sender: TObject; SignOutType: TSignOutType);
begin
  if SignOutType <> otXFR then
  begin
    FNsmInfo.SetConnectionInfo(TMsnConnection(Sender).NsmHandle, NMCI_STATUS,
      NMCS_DISCONNECTED);
    case SignOutType of
    otOTH     : ShowNotify('ق̏ꏊŃTCC߁AT[oؒf܂B', NMNI_STOP);
    otSSD     : ShowNotify('eiX̂߁AT[oؒf܂B', NMNI_STOP);
    end;
  end;
end;

procedure TMsnPluginMain.ConLog(Sender: TObject;  LogStr: String);
begin
//  CallService(NMS_SYSTEM_DEBUG_PRINT, Cardinal(PChar(LogStr)), 0);
//  ShowNotify(WideString(LogStr), NMNI_STOP);
end;

procedure TMsnPluginMain.ConError(Sender: TObject;  ErrorKind: TErrorKind;
  ErrorCode: Integer);
var
  Msg: String;
begin
  if (ErrorKind = ekSocketError) and (ErrorCode = 10053) then  // WSAECONNABORTED
    Exit;
  case ErrorKind of
  ekMsnError    :
    Msg := 'Protocol Error #' + IntToStr(ErrorCode) + #13#10 +
           GetMsnErrorMessage(ErrorCode);
  ekSocketError :
    Msg := 'Socket Error #' + IntToStr(ErrorCode);
  end;
  ShowNotify(WideString(Msg), NMNI_STOP);
end;

procedure TMsnPluginMain.ConSwitchBoard(Sender: TObject;  TrID: Integer; SBAddress,
  Cookie: String);
var
  sIdx: Integer;
  Host, Port: String;
begin
  Host := Copy(SBAddress, 1, AnsiPos(':', SBAddress) - 1);
  Port := Copy(SBAddress, AnsiPos(':', SBAddress) + 1, Length(SBAddress));
  sIdx := FSessions.IndexOfRequestID(TrID);
  if sIdx > -1 then
    FSessions[sIdx].Connect(Host, StrToIntDef(Port, Config.MsnPort),
        FSessions[sIdx].User.Account, Cookie, '');
end;

procedure TMsnPluginMain.ConCalled(Sender: TObject;  SessionID, SBAddress, Cookie,
  CallingUserAccount: String; CallingUserName: WideString);
var
  Connection: TMsnConnection;
  Host, Port: String;
begin
  Host := Copy(SBAddress, 1, AnsiPos(':', SBAddress) - 1);
  Port := Copy(SBAddress, AnsiPos(':', SBAddress) + 1, Length(SBAddress));
  Connection := TMsnConnection(Sender);

  with FSessions.Add do
  begin
    User.Assign(Connection.User);
    NsmHandle := 0;
    OnConnect := SesConnect;
    OnDisConnect := SesDisconnect;
    OnJoinMember := SesJoinMember;
    OnByeMember := SesByeMember;
    OnMemberListChange := SesMemberListChange;    
    OnRecieveMessage := SesReceiveMessage;
    OnLog := ConLog;
    OnError := ConError;
    Connect(Host, StrtoIntDef(Port, Config.MsnPort), Connection.User.Account,
        Cookie, SessionID);
  end;
end;

procedure TMsnPluginMain.ConNewMail(Sender: TObject; FromName: WideString;
  FromAddr: String);
begin
  ShowNotify('Hotmail MgC ' + FromName +
      ' ̐V[͂܂B', NMNI_MAIL, OnMailNotifyClick,
      Integer(Sender));
end;

procedure TMsnPluginMain.ConUnreadMailChange(Sender: TObject; Init: Boolean);
var
  Msg: String;
begin
  if Init then
  begin
    Msg := '';
    if TMsnConnection(Sender).InboxUnread > 0 then
      Msg := Msg + 'MgC ' +
        IntToStr(TMsnConnection(Sender).InboxUnread) + ' ';
    if TMsnConnection(Sender).FoldersUnread > 0 then
    begin
      if Msg <> '' then
        Msg := Msg + 'A';
      Msg := Msg + 'tH_ ' +
        IntToStr(TMsnConnection(Sender).FoldersUnread) + ' ';
    end;

    ShowNotify('Hotmail ' + Msg + '̖ǃ[܂B', NMNI_MAIL,
        OnMailNotifyClick, Integer(Sender));
  end;
end;

procedure TMsnPluginMain.ConUrl(Sender: TObject; rru, passport: String);
var
  TmpLst: TStringList;
  TmpFileName: String;
  creds: String;
  PassportInfo: TMsnPassportInfo;
begin
  PassportInfo := TMsnConnection(Sender).PassportInfo;
  creds := Format('%s%d%s', [PassportInfo.mspauth,
      DateTimeToCTime(Now) - PassportInfo.LoginTime,
      TMsnConnection(Sender).User.Password]);
  creds := MD5Print(MD5String(creds));
  TmpLst := TStringList.Create;
  try
		TmpLst.Add('<html>');
		TmpLst.Add('<head>');
		TmpLst.Add('<noscript>');
		TmpLst.Add('<meta http-equiv=Refresh content="0; url=http://www.hotmail.com">');
		TmpLst.Add('</noscript>');
		TmpLst.Add('</head>');
		TmpLst.Add('');
		TmpLst.Add('<body onload="document.pform.submit(); ">');
		TmpLst.Add(Format('<form name="pform" action="%s" method="POST">', [passport]));
		TmpLst.Add('<input type="hidden" name="mode" value="ttl">');
		TmpLst.Add(Format('<input type="hidden" name="login" value="%s">', [TMsnConnection(Sender).User.Account]));
		TmpLst.Add(Format('<input type="hidden" name="username" value="%s">', [TMsnConnection(Sender).User.Account]));
		TmpLst.Add(Format('<input type="hidden" name="sid" value="%d">', [PassportInfo.sid]));
		TmpLst.Add(Format('<input type="hidden" name="kv" value="%d">', [PassportInfo.kv]));
		TmpLst.Add('<input type="hidden" name="id" value="2">');
		TmpLst.Add(Format('<input type="hidden" name="sl" value="%d">', [DateTimeToCTime(Now) - PassportInfo.LoginTime]));
		TmpLst.Add(Format('<input type="hidden" name="rru" value="%s">', [rru]));
		TmpLst.Add(Format('<input type="hidden" name="auth" value="%s">', [PassportInfo.mspauth]));
		TmpLst.Add(Format('<input type="hidden" name="creds" value="%s">', [creds]));
		TmpLst.Add('<input type="hidden" name="svc" value="mail">');
		TmpLst.Add('<input type="hidden" name="js" value="yes">');
		TmpLst.Add('</form></body>');
		TmpLst.Add('</html>');

    TmpFileName := WorkDir + TEMP_HTMLNAME;
    TmpLst.SaveToFile(TmpFileName);
    ShellExecute(0, nil, PChar(TmpFileName), nil, nil, SW_SHOW);
  finally
    TmpLst.Free;
  end;
end;

// TMsnSession Object ̃Cxgnh ---------------------------------------

procedure TMsnPluginMain.SesConnect(Sender: TObject);
var
  SHandle: HNsmSession;
  Connection: TMsnConnection;
  I, Idx: Integer;
  Member: TMsnMember;
begin
  if TMsnSession(Sender).NsmHandle = 0 then
  begin
  {
    if (TMsnSession(Sender).Members.Count = 1) then
    begin
      Idx := FSessions.IndexOfSingleMember(TMsnSession(Sender));
      if Idx > -1 then
      begin
        TMsnSession(Sender).NsmHandle := FSessions[Idx].NsmHandle;
        FSessions.Delete(Idx);
        FNsmInfo.SetSessionInfo(TMsnSession(Sender).NsmHandle,
            NMSI_STATUS, NMCS_CONNECTING);
        for I := 0 to TMsnSession(Sender).Members.Count - 1 do
        begin
          Member := TMsnSession(Sender).Members[I];
          FInitInfo.CallService(
              FInitInfo.GetService(NMS_SYSTEM_SESSION_MEMBERS_ADD),
              TMsnSession(Sender).NsmHandle, Cardinal(PChar(Member.Account)));
          FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle,
              Member.Account, NMMI_NAME, Member.Name);
          FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle,
              Member.Account, NMMI_STATUS, Ord(Member.Status));
        end;
        FNsmInfo.SetSessionInfo(TMsnSession(Sender).NsmHandle,
            NMSI_STATUS, NMCS_CONNECTED);
        SetSessionCaption(TMsnSession(Sender));
        Exit;
      end;
    end;
  }
    Idx := FConnections.IndexOfUser(TMsnSession(Sender).User.Account);
    if Idx > -1 then
    begin
      Connection := FConnections[Idx];
      SHandle := CallService(NMS_SYSTEM_SESSION_CREATE,
          Connection.NsmHandle, 0);
      if (SHandle > 0) then
      begin
        TMsnSession(Sender).NsmHandle := SHandle;
        FNsmInfo.SetSessionInfo(SHandle, NMSI_STATUS, NMCS_CONNECTING);
        for I := 0 to TMsnSession(Sender).Members.Count - 1 do
        begin
          Member := TMsnSession(Sender).Members[I];
          CallService(NMS_SYSTEM_SESSION_MEMBERS_ADD,
              TMsnSession(Sender).NsmHandle, Cardinal(PChar(Member.Account)));
          FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle,
              Member.Account, NMMI_NAME, Member.Name);
          FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle,
              Member.Account, NMMI_STATUS, Ord(Member.Status));
        end;
        FNsmInfo.SetSessionInfo(SHandle, NMSI_STATUS, NMCS_CONNECTED);
        SetSessionCaption(TMsnSession(Sender));
      end else
      begin
        TMsnSession(Sender).Free;
        Exit;
      end;
    end else
    begin
      TMsnSession(Sender).Free;
      Exit;
    end;
  end;
  FNsmInfo.SetSessionInfo(TMsnSession(Sender).NsmHandle, NMSI_STATUS,
    NMCS_CONNECTED);
end;

procedure TMsnPluginMain.SesDisconnect(Sender: TObject);
begin
  FNsmInfo.SetSessionInfo(TMsnSession(Sender).NsmHandle, NMSI_STATUS,
    NMCS_DISCONNECTED);
end;

procedure TMsnPluginMain.SesJoinMember(Sender: TObject;
  Member: TMsnMemberBase);
begin
  if TMsnSession(Sender).NsmHandle > 0 then
  begin
    CallService(NMS_SYSTEM_SESSION_MEMBERS_ADD,
        TMsnSession(Sender).NsmHandle, Cardinal(PChar(Member.Account)));
    FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle, Member.Account,
        NMMI_NAME, Member.Name);
    FNsmInfo.SetSessionMemberInfo(TMsnSession(Sender).NsmHandle, Member.Account,
        NMMI_STATUS, Ord(Member.Status));
    SetSessionCaption(TMsnSession(Sender));
  end;
end;

procedure TMsnPluginMain.SesByeMember(Sender: TObject;
  Member: TMsnMemberBase);
begin
  if (TMsnSession(Sender).ReservedMembers.Count = 0) and
     (TMsnSession(Sender).Members.Count = 1) then
     TMsnSession(Sender).ReservedMembers.Add(Member.Account)
  else
    CallService(NMS_SYSTEM_SESSION_MEMBERS_REMOVE,
        TMsnSession(Sender).NsmHandle, Cardinal(PChar(Member.Account)));
end;

procedure TMsnPluginMain.SesMemberListChange(Sender: TObject);
begin
  SetSessionCaption(TMsnSession(Sender));
end;

procedure TMsnPluginMain.SesReceiveMessage(Sender: TObject; Header: UTF8String;
  FromAccount: String; FromName, Msg: WideString);
  procedure GetTextAttr(FontStr: String; var Attr: TTextAttributeInfo; var FontName: String);
  var
    I: Integer;
    ParamLst: TStringList;
    CO, EF, FN, CS: String;
  begin
    ParamLst := TStringList.Create;
    try
      Split(ParamLst, FontStr, ';');

      for I := 0 to ParamLst.Count - 1 do
        ParamLst[I] := Trim(ParamLst[I]);

      CO := ParamLst.Values['CO'];
      EF := ParamLst.Values['EF'];
      FN := Utf8ToAnsi(UrlDecode(ParamLst.Values['FN']));
      CS := ParamLst.Values['CS'];

      Attr.nStyles := 0;
      if Pos('B', EF) > 0 then
        Attr.nStyles := Attr.nStyles or NMFS_BOLD;
      if Pos('I', EF) > 0 then
        Attr.nStyles := Attr.nStyles or NMFS_ITALIC;
      if Pos('U', EF) > 0 then
        Attr.nStyles := Attr.nStyles or NMFS_UNDERLINE;
      if Pos('S', EF) > 0 then
        Attr.nStyles := Attr.nStyles or NMFS_STRIKEOUT;

      Attr.nFontColor := StrToIntDef('$' + CO, -1);
      Attr.nCharSet := StrToIntDef('$' + CS, DEFAULT_CHARSET);
      Attr.nBgColor := -1;
      Attr.nFontSize := -1;
      FontName := FN;
    finally
      ParamLst.Free;
    end;
  end;
  procedure GetNsmTextAttr(FontStr: String; var Attr: TTextAttributeInfo);
  var
    I: Integer;
    ParamLst: TStringList;
    FS, BC: String;
  begin
    ParamLst := TStringList.Create;
    try
      Split(ParamLst, FontStr, ';');

      for I := 0 to ParamLst.Count - 1 do
        ParamLst[I] := Trim(ParamLst[I]);

      FS := ParamLst.Values['FS'];
      BC := ParamLst.Values['BC'];

      Attr.nFontSize  := StrToIntDef(FS, -1);
      Attr.nBgColor   := StrToIntDef('$' + BC, -1);
    finally
      ParamLst.Free;
    end;
  end;
var
  TextAttr: TTextAttributeInfo;
  MsgInfo: TMessageInfo;
  ContentType, FontName: String;
  Headers: TStringList;
begin
  Headers := TStringList.Create;
  try
    // MIMEwb_𕪗
    SplitMimeHeader(Headers, Header);
    // wb_
    ContentType := Headers.Values['Content-Type'];
    // MIME Type ʂɏ
    if AnsiContainsStr(ContentType, 'text/x-msmsgscontrol') then
    begin

    end else
    if AnsiContainsStr(ContentType, 'text/x-msmsgsinvite') then
    begin

    end else
    begin
      GetTextAttr(Headers.Values['X-MMS-IM-Format'], TextAttr, FontName);
      GetNsmTextAttr(Headers.Values['X-NSM-IM-Format'], TextAttr);
      TextAttr.lpFontName := PWideChar(WideString(FontName));
      with MsgInfo do
      begin
        cbSize := SizeOf(MsgInfo);
        lpFrom := PChar(String(FromAccount));
        lpBody := PWideChar(Msg);
        lpTextAttribute := @TextAttr;
        nFlags := 0;
      end;
      CallService(NMS_SYSTEM_SESSION_RECEIVEMESSAGE,
          TMsnSession(Sender).NsmHandle, Cardinal(@MsgInfo));
    end;
  finally
    Headers.Free;
  end;
end;

// -----------------------------------------------------------------------------

procedure PingTimerProc(hWnd: HWND; uMsg, idEvent: Cardinal; dwTime: DWORD); stdcall;
var
  I: Integer;
begin
  for I := 0 to PluginMain.FConnections.Count - 1 do
    if PluginMain.FConnections[I].SignInStage = ssSignIn then
    begin
      PluginMain.FConnections[I].Ping;
    end;
end;

procedure OnMailNotifyClick(nData: Integer); stdcall;
begin
  TMsnConnection(nData).QueryUrl('INBOX');
end;

// -----------------------------------------------------------------------------

initialization
  PluginMain := TMsnPluginMain.Create;

finalization
  PluginMain.Free;

end.
