unit UMsnMsgr;
(* Msn Messenger Client Components *)

interface
uses
  Windows, SysUtils, Classes, StrUtils, UMemberList, ScktComp, Md5,
  UMsnUtils;

const
  MSN_DEFAULTHOST = 'messenger.hotmail.com';
  MSN_DEFAULTPORT = 1863;
  MSN_VERSION = 'MSNP7 MSNP6';

  ERR_SYNTAX_ERROR                = 200;
  ERR_INVALID_PARAMETER           = 201;
  ERR_INVALID_USER                = 205;
  ERR_FQDN_MISSING                = 206;
  ERR_ALREADY_LOGIN               = 207;
  ERR_INVALID_USERNAME            = 208;
  ERR_INVALID_FRIENDLY_NAME       = 209;
  ERR_LIST_FULL                   = 210;
  ERR_ALREADY_THERE               = 215;
  ERR_NOT_ON_LIST                 = 216;
  ERR_USER_NOT_ONLINE             = 217;
  ERR_ALREADY_IN_THE_MODE         = 218;
  ERR_ALREADY_IN_OPPOSITE_LIST    = 219;
  ERR_NOT_EXISTS_GROUP            = 231;
  ERR_SWITCHBOARD_FAILED          = 280;
  ERR_NOTIFY_XFR_FAILED           = 281;

  ERR_REQUIRED_FIELDS_MISSING     = 300;
  ERR_NOT_LOGGED_IN               = 302;
  ERR_INTERNAL_SERVER             = 500;
  ERR_DB_SERVER                   = 501;
  ERR_FILE_OPERATION              = 510;
  ERR_MEMORY_ALLOC                = 520;
  ERR_WRONG_CHL_VALUE             = 540;
  ERR_SERVER_BUSY                 = 600;
  ERR_SERVER_UNAVAILABLE          = 601;
  ERR_PEER_NS_DOWN                = 602;
  ERR_DB_CONNECT                  = 603;
  ERR_SERVER_GOING_DOWN           = 604;
  ERR_CREATE_CONNECTION           = 707;
  ERR_UNKNOWN_CVR_PARAMETERS      = 710;
  ERR_BLOCKING_WRITE              = 711;
  ERR_SESSION_OVERLOAD            = 712;
  ERR_USER_TOO_ACTIVE             = 713;
  ERR_TOO_MANY_SESSIONS           = 714;
  ERR_NOT_EXPECTED                = 715;
  ERR_BAD_FRIEND_FILE             = 717;
  ERR_AUTHENTICATION_FAILED       = 911;
  ERR_NOT_ALLOWED_WHEN_OFFLINE    = 913;
  ERR_NOT_ACCEPTING_NEW_USERS     = 920;
  ERR_PASSPORT_NOT_VERIFIED       = 924;

type
  TSignInStage  = (ssUnConnect, ssTryConnectServer, ssTrySignIn, ssSignIn);
  TListKind     = (lkFL, lkRL, lkAL, lkBL);
  TErrorKind    = (ekMsnError, ekSocketError);
  TSignOutType  = (otOTH, otSSD, otXFR, otUnKnown);
  TBLP          = (bpAL, bpBL);

  TMsnPassportInfo = record
    LoginTime: Integer;
    EmailEnabled: Boolean;
    MemberIdHigh: Integer;
    MemberIdLow: Integer;
    lang_preference: Integer;
    preferredEmail: String;
    country: String;
    PostalCode: String;
    Gender: String;
    Kid: Integer;
    Age: Integer;
    sid: Integer;
    kv: Integer;
    MSPAuth: String;
    sl: Integer;
  end;

  TMsnListMemberEvent = procedure (Sender: TObject; ListKind: TListKind;
    Member: TMsnMember) of Object;
  TMsnListEvent = procedure (Sender: TObject; List: TListKind) of Object;
  TMsnMemberEvent = procedure (Sender: TObject; Member: TMsnMemberBase) of Object;
  TMsnGroupEvent= procedure (Sender: TObject; Group: TMsnGroup) of Object;
  TMsnStatusChangeEvent = procedure (Sender: TObject; Member: TMsnMemberBase;
    OldStatus: TMsnMemberStatus; InitList: Boolean) of Object;
  TMsnSignOutEvent = procedure (Sender: TObject; SignOutType: TSignOutType) of Object;
  TMsnLogEvent = procedure (Sender: TObject; LogStr: String) of Object;
  TMsnErrorEvent = procedure (Sender: TObject; ErrorKind: TErrorKind;
    ErrorCode: Integer) of Object;
  TMsnReceiveMessageEvent = procedure (Sender: TObject; Header: UTF8String;
    FromAccount: String; FromName, Msg: WideString) of Object;
  TMsnSwitchBoardEvent = procedure (Sender: TObject; TrID: Integer;
    SBAddress, Cookie: String) of Object;
  TMsnCalledEvent = procedure (Sender: TObject; SessionID, SBAddress, Cookie,
    CallingUserAccount: String; CallingUserName: WideString) of Object;
  TMsnUrlEvent = procedure (Sender: TObject; rru, passport: String) of Object;
  TMsnNewMailEvent = procedure (Sender: TObject; FromName: WideString; FromAddr: String) of Object;
  TMsnUnreadMailChangeEvent = procedure (Sender: TObject; Init: Boolean) of Object;

  // ---------------------------------------------------------------------------
  TMsnConnection = class(TObject)
  private
    FNsmHandle: Cardinal;
    FSocket: TClientSocket;
    FHost: String;
    FPort: Integer;
    FUser: TMsnUser;
    FMembers: TMsnMemberList;
    FGroups: TMSNGroupList;
    FReverceMembers: TMsnMemberList;
    FAllowMembers: TMsnMemberList;
    FBlockMembers: TMsnMemberList;
    FSignInStage: TSignInStage;
    FGTC: Boolean;
    FBLP: TBLP;
    FTrID: Integer;
    FPassportInfo: TMsnPassportInfo;
    FInboxUnread: Integer;
    FFoldersUnread: Integer;

    FSignOutType: TSignOutType;
    FIncompleteCommand: UTF8String;

    // CxgvpeB̕ϐ
    FOnMemberAddition: TMsnListMemberEvent;
    FOnMemberDeletion: TMsnListMemberEvent;
    FOnMemberListChange: TMsnListEvent;
    FOnMemberStatusChange: TMsnStatusChangeEvent;
    FOnMemberNameChange: TMsnMemberEvent;
    FOnMemberGroupChange: TMsnMemberEvent;
    FOnMemberOnline: TMsnMemberEvent;
    FOnMemberOffline: TMsnMemberEvent;
    FOnGroupAddition: TMsnGroupEvent;
    FOnGroupDeletion: TMsnGroupEvent;
    FOnGroupListChange: TNotifyEvent;
    FOnGroupNameChange: TMsnGroupEvent;
    FOnSignIn: TNotifyEvent;
    FOnSignOut: TMsnSignOutEvent;
    FOnMemberListUpdated: TMsnListEvent;
    FOnLog: TMsnLogEvent;
    FOnError: TMsnErrorEvent;
    FOnSwitchBoard: TMsnSwitchBoardEvent;
    FOnCalled: TMsnCalledEvent;
    FOnReceiveMessage: TMsnReceiveMessageEvent;
    FOnUrl: TMsnUrlEvent;
    FOnNewMail: TMsnNewMailEvent;
    FOnUnreadMailChange: TMsnUnreadMailChangeEvent;

    // Ŏg葱Ȃ
    procedure ProcessSignIn(ParamLst: TStringList);
    procedure ProcessMemberList(ParamLst: TStringList);
    procedure ProcessGroupList(ParamLst: TStringList);
    procedure ProcessChangeStatus(ParamLst: TStringList);
    procedure ProcessChangeName(ParamLst: TStringList);
    procedure ProcessError(ParamLst: TStringList);
    procedure ProcessPrivacySetting(ParamLst: TStringList);
    procedure ProcessCHL(ParamLst: TStringList);
    procedure ProcessMessage(DataStr: UTF8String);
    procedure ProcessHotmail(Header: UTF8String; FromAccount: String;
      FromName, Msg: WideString);

    procedure CreateSocket;
    procedure DestroySocket;
    procedure SetGTC(Value: Boolean);
    procedure SetBLP(Value: TBLP);
    function ListKindToStr(List: TListKind): String;

    // \Pbg̃Cxgnh
    procedure SocketConnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketRead(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketDisconnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketConnecting(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketError(Sender: TObject; Socket: TCustomWinSocket;
      ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  protected
    // Events Trigger
    procedure DoMemberAddition(ListKind: TListKind; Member: TMsnMember);
    procedure DoMemberDeletion(ListKind: TListKind; Member: TMsnMember);
    procedure DoMemberListChange(ListKind: TListKind);
    procedure DoMemberStatusChange(Member: TMsnMemberBase; OldStatus: TMsnMemberStatus; InitList: Boolean);
    procedure DoMemberNameChange(Member: TMsnMemberBase);
    procedure DoMemberGroupChange(Member: TMsnMemberBase);
    procedure DoMemberOnline(Member: TMsnMemberBase);
    procedure DoMemberOffline(Member: TMsnMemberBase);
    procedure DoGroupAddition(Group: TMsnGroup);
    procedure DoGroupDeletion(Group: TMsnGroup);
    procedure DoGroupListChange;
    procedure DoGroupNameChange(Group: TMsnGroup);
    procedure DoSignIn;
    procedure DoSignOut(SignOutType: TSignOutType);
    procedure DoMemberListUpdated(ListKind: TListKind);
    procedure DoLog(Head, Str: String);
    procedure DoError(ErrorKind: TErrorKind; ErrorCode: Integer);
    procedure DoSwitchBoard(TrID: Integer; SBAddress, Cookie: String);
    procedure DoCalled(SessionID, SBAddress, Cookie, CallingUserAccount: String;
      CallingUserName: WideString);
    procedure DoReceiveMessage(Header: UTF8String; FromAccount: String;
      FromName, Msg: WideString);
    procedure DoUrl(rru, passport: String);
    procedure DoNewMail(FromName: WideString; FromAddr: String);
    procedure DoUnreadMailChange(Init: Boolean);
  public
    // \bh
    constructor Create;
    destructor Destroy; override;
    procedure SendCommand(Str: UTF8String);
    procedure SignIn(Account, Password: String; Name: WideString;
      Status: TMsnMemberStatus);
    procedure SignOut;
    function ChangeUserStatus(Status: TMsnMemberStatus): Integer;
    function RenameMember(Account: String; NewName: WideString): Integer;
    function AddMember(List: TListKind; Account: String; GroupId: Integer = -1): Integer;
    function RemoveMember(List: TListKind; Account: String; GroupId: Integer = -1): Integer;
    function AddGroup(Name: WideString): Integer;
    function RemoveGroup(GroupId: Integer): Integer;
    function RenameGroup(GroupId: Integer; NewName: WideString): Integer;
    function SwitchBoardRequest: Integer;
    procedure BlockMember(Account: String);
    procedure AllowMember(Account: String);
    function QueryUrl(Param: String): Integer;
    procedure Ping;

    // svpe
    property NsmHandle: Cardinal read FNsmHandle write FNsmHandle;
    property TrID: Integer read FTrID;
    property User: TMsnUser read FUser;
    property Members: TMsnMemberList read FMembers;
    property ReverceMembers: TMsnMemberList read FReverceMembers;
    property AllowMembers: TMsnMemberList read FAllowMembers;
    property BlockMembers: TMsnMemberList read FBlockMembers;
    property Groups: TMSNGroupList read FGroups;
    property Socket: TClientSocket read FSocket;
    property SignInStage: TSignInStage read FSignInStage;
    property GTC: Boolean read FGTC write SetGTC;
    property BLP: TBLP read FBLP write SetBLP;
    property DSHost: String read FHost write FHost;
    property DSPort: Integer read FPort write FPort;
    property PassportInfo: TMsnPassportInfo read FPassportInfo;
    property InboxUnread: Integer read FInboxUnread;
    property FoldersUnread: Integer read FFoldersUnread;

    // Cxg
    property OnMemberAddition: TMsnListMemberEvent read FOnMemberAddition write FOnMemberAddition;
    property OnMemberDeletion: TMsnListMemberEvent read FOnMemberDeletion write FOnMemberDeletion;
    property OnMemberListChange: TMsnListEvent read FOnMemberListChange write FOnMemberListChange;
    property OnMemberStatusChange: TMsnStatusChangeEvent read FOnMemberStatusChange write FOnMemberStatusChange;
    property OnMemberNameChange: TMsnMemberEvent read FOnMemberNameChange write FOnMemberNameChange;
    property OnMemberGroupChange: TMsnMemberEvent read FOnMemberGroupChange write FOnMemberGroupChange;
    property OnMemberOnline: TMsnMemberEvent read FOnMemberOnline write FOnMemberOnline;
    property OnMemberOffline: TMsnMemberEvent read FOnMemberOffline write FOnMemberOffline;
    property OnGroupAddition: TMsnGroupEvent read FOnGroupAddition write FOnGroupAddition;
    property OnGroupDeletion: TMsnGroupEvent read FOnGroupDeletion write FOnGroupDeletion;
    property OnGroupListChange: TNotifyEvent read FOnGroupListChange write FOnGroupListChange;
    property OnGroupNameChange: TMsnGroupEvent read FOnGroupNameChange write FOnGroupNameChange;
    property OnSignIn: TNotifyEvent read FOnSignIn write FOnSignIn;
    property OnSignOut: TMsnSignOutEvent read FOnSignOut write FOnSignOut;
    property OnMemberListUpdated: TMsnListEvent read FOnMemberListUpdated write FOnMemberListUpdated;
    property OnLog: TMsnLogEvent read FOnLog write FOnLog;
    property OnError: TMsnErrorEvent read FOnError write FOnError;
    property OnSwitchBoard: TMsnSwitchBoardEvent read FOnSwitchBoard write FOnSwitchBoard;
    property OnCalled: TMsnCalledEvent read FOnCalled write FOnCalled;
    property OnReceiveMessage: TMsnReceiveMessageEvent read FOnReceiveMessage write FOnReceiveMessage;
    property OnUrl: TMsnUrlEvent read FOnUrl write FOnUrl;
    property OnNewMail: TMsnNewMailEvent read FOnNewMail write FOnNewMail;
    property OnUnreadMailChange: TMsnUnreadMailChangeEvent read FOnUnreadMailChange write FOnUnreadMailChange;
  end;

  // ---------------------------------------------------------------------------
  TMsnSession = class(TObject)
  private
    // vpeBtB[h
    FNsmHandle: Cardinal;
    FRequestID: Integer;
    FSocket: TClientSocket;
    FHost: String;
    FPort: Integer;
    FUser: TMsnUser;
    FMembers: TMsnMemberList;
    FSignInStage: TSignInStage;
    FReservedMembers: TStringList;
    FReservedMessages: TStringList;

    FOnConnect: TNotifyEvent;
    FOnDisconnect: TNotifyEvent;
    FOnLog: TMsnLogEvent;
    FOnError: TMsnErrorEvent;
    FOnMemberListChange: TNotifyEvent;
    FOnRecieveMessage: TMsnReceiveMessageEvent;
    FOnJoinMember: TMsnMemberEvent;
    FOnByeMember: TMsnMemberEvent;

    FTrID: Integer;
    FCookie: String;
    FSessionID: String;
    FIncompleteCommand: String;    
    procedure SendCommand(Str: Utf8String);
    procedure ErrorHandler(ParamLst: TStringList);

    procedure SetMemberList(ParamLst: TStringList);
    procedure ReceiveMessage(DataStr: Utf8String);

    // Socket Events
    procedure SocketConnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketRead(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketDisconnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketConnecting(Sender: TObject; Socket: TCustomWinSocket);
    procedure SocketError(Sender: TObject; Socket: TCustomWinSocket;
      ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  protected
    procedure DoConnect;
    procedure DoDisconnect;
    procedure DoLog(Head, Str: String);
    procedure DoError(ErrorKind: TErrorKind; ErrorCode: Integer);
    procedure DoMemberListChange;
    procedure DoRecieveMessage(Header: UTF8String; FromAccount: String;
        FromName, Msg: WideString);
    procedure DoJoinMember(Member: TMsnMember);
    procedure DoByeMember(Member: TMsnMember);
  public
    // \bh
    constructor Create;
    destructor Destroy; override;
    procedure Connect(AHost: String; APort: Integer; Account, Cookie,
      SessionID: String);
    procedure Disconnect;
    procedure SendMessage(Msg: WideString);
    procedure CallMember(Account: String);
    procedure CallReservedMembers;
    procedure SendReservedMessages;
    function SingleMember(Account: String): Boolean;

    // svpeB
    property NsmHandle: Cardinal read FNsmHandle write FNsmHandle;
    property RequestID: Integer read FRequestID write FRequestID;
    property TrID: Integer read FTrID;    
    property User: TMsnUser read FUser;
    property Members: TMsnMemberList read FMembers;
    property Socket: TClientSocket read FSocket;
    property SignInStage: TSignInStage read FSignInStage;
    property ReservedMembers: TStringList read FReservedMembers;
    property ReservedMessages: TStringList read FReservedMessages;
  published
    // Cxg
    property OnConnect: TNotifyEvent read FOnConnect write FOnConnect;
    property OnDisconnect: TNotifyEvent read FOnDisconnect write FOnDisconnect;
    property OnLog: TMsnLogEvent read FOnLog write FOnLog;
    property OnError: TMsnErrorEvent read FOnError write FOnError;
    property OnRecieveMessage: TMsnReceiveMessageEvent read FOnRecieveMessage write FOnRecieveMessage;
    property OnJoinMember: TMsnMemberEvent read FOnJoinMember write FOnJoinMember;
    property OnByeMember: TMsnMemberEvent read FOnByeMember write FOnByeMember;
    property OnMemberListChange: TNotifyEvent
                read FOnMemberListChange write FOnMemberListChange;
  end;

  // RlNVXg
  TMsnConnectionList = class
  private
    FConnections: TList;
    function GetConnection(Index: Integer): TMsnConnection;
    function GetCount: Integer;
  public
    constructor Create;
    destructor Destroy; override;
    function Add: TMsnConnection;
    procedure Delete(Idx: Integer);
    procedure Clear;
    function IndexOf(AConnection: TMsnConnection): Integer;
    function IndexOfNsmHandle(NsmHandle: Cardinal): Integer;
    function IndexOfUser(Account: String): Integer;

    property Connections[Index: Integer]: TMsnConnection read GetConnection; default;
    property Count: Integer read GetCount;
  end;

  TMsnSessionList = class
  private
    FSessions: TList;
    function GetSession(Index: Integer): TMsnSession;
    function GetCount: Integer;
  public
    constructor Create;
    destructor Destroy; override;
    function Add: TMsnSession;
    procedure Delete(Idx: Integer);
    procedure Clear;
    function IndexOf(ASession: TMsnSession): Integer;
    function IndexOfNsmHandle(NsmHandle: Cardinal): Integer;
    function IndexOfRequestID(RequestID: Integer): Integer;
    function IndexOfSingleMember(ASession: TMsnSession): Integer;

    property Sessions[Index: Integer]: TMsnSession read GetSession; default;
    property Count: Integer read GetCount;
  end;


function GetMsnErrorMessage(ECode: Integer): String;

implementation

function SplitCommandStr(List: TStringList; const Str: UTF8String): UTF8String; forward;
procedure SplitParamStr(List: TStringList; const Str: UTF8String); forward;
function SS(List: TStringList; Idx: Integer): String; forward;
function Utf8ToAnsiEx(const S: Utf8String): String; forward;
function EncodeParam(const S: WideString): UTF8String; forward;
function DecodeParam(const S: UTF8String): WideString; forward;

// -----------------------------------------------------------------------------
constructor TMsnConnection.Create;
begin
  inherited;
  CreateSocket;
  FUser           := TMsnUser.Create;
  FMembers        := TMsnMemberList.Create;
  FReverceMembers := TMsnMemberList.Create;
  FAllowMembers   := TMsnMemberList.Create;
  FBlockMembers   := TMsnMemberList.Create;
  FGroups := TMSNGroupList.Create;

  FHost := MSN_DEFAULTHOST;
  FPort := MSN_DEFAULTPORT;
  FSignInStage := ssUnConnect;
  FUser.Status := usFLN;
end;

destructor TMsnConnection.Destroy;
begin
  FUser.Free;
  FMembers.Free;
  FReverceMembers.Free;
  FAllowMembers.Free;
  FBlockMembers.Free;
  FGroups.Free;
  DestroySocket;
  inherited;
end;

// R}hM
procedure TMsnConnection.SendCommand(Str: UTF8String);
begin
  FSocket.Socket.SendText(Str);
  DoLog('C', Utf8ToAnsi(Str));
end;

procedure TMsnConnection.DoMemberAddition(ListKind: TListKind; Member: TMsnMember);
begin
  if Assigned(FOnMemberAddition) then
    FOnMemberAddition(Self, ListKind, Member);
end;

procedure TMsnConnection.DoMemberDeletion(ListKind: TListKind; Member: TMsnMember);
begin
  if Assigned(FOnMemberDeletion) then
    FOnMemberDeletion(Self, ListKind, Member);
end;

procedure TMsnConnection.DoMemberListChange(ListKind: TListKind);
begin
  if Assigned(FOnMemberListChange) then
    FOnMemberListChange(Self, ListKind);
end;

procedure TMsnConnection.DoMemberStatusChange(Member: TMsnMemberBase; OldStatus: TMsnMemberStatus; InitList: Boolean);
begin
  if Assigned(FOnMemberStatusChange) then
    FOnMemberStatusChange(Self, Member, OldStatus, InitList);
end;

procedure TMsnConnection.DoMemberNameChange(Member: TMsnMemberBase);
begin
  if Assigned(FOnMemberNameChange) then
    FOnMemberNameChange(Self, Member);
end;

procedure TMsnConnection.DoMemberGroupChange(Member: TMsnMemberBase);
begin
  if Assigned(FOnMemberGroupChange) then
    FOnMemberGroupChange(Self, Member);
end;

procedure TMsnConnection.DoMemberOnline(Member: TMsnMemberBase);
begin
  if Assigned(FOnMemberOnline) then
    FOnMemberOnline(Self, Member);
end;

procedure TMsnConnection.DoMemberOffline(Member: TMsnMemberBase);
begin
  if Assigned(FOnMemberOffline) then
    FOnMemberOffline(Self, Member);
end;

procedure TMsnConnection.DoGroupAddition(Group: TMsnGroup);
begin
  if Assigned(FOnGroupAddition) then
    FOnGroupAddition(Self, Group);
end;

procedure TMsnConnection.DoGroupDeletion(Group: TMsnGroup);
begin
  if Assigned(FOnGroupDeletion) then
    FOnGroupDeletion(Self, Group);
end;

procedure TMsnConnection.DoGroupListChange;
begin
  if Assigned(FOnGroupListChange) then
    FOnGroupListChange(Self);
end;

procedure TMsnConnection.DoGroupNameChange(Group: TMsnGroup);
begin
  if Assigned(FOnGroupNameChange) then
    FOnGroupNameChange(Self, Group);
end;

procedure TMsnConnection.DoSignIn;
begin
  if Assigned(FOnSignIn) then
    FOnSignIn(Self);
end;

procedure TMsnConnection.DoSignOut(SignOutType: TSignOutType);
begin
  if Assigned(FOnSignOut) then
    FOnSignOut(Self, SignOutType);
end;

procedure TMsnConnection.DoMemberListUpdated(ListKind: TListKind);
begin
  if Assigned(FOnMemberListUpdated) then
    FOnMemberListUpdated(Self, ListKind);
end;

procedure TMsnConnection.DoLog(Head, Str: String);
begin
  if Assigned(FOnLog) then
    FOnLog(Self, Head + #9 + Str);
end;

procedure TMsnConnection.DoError(ErrorKind: TErrorKind; ErrorCode: Integer);
begin
  if Assigned(FOnError) then
    FOnError(Self, ErrorKind, ErrorCode);
end;

procedure TMsnConnection.DoSwitchBoard(TrID: Integer; SBAddress, Cookie: String);
begin
  if Assigned(FOnSwitchBoard) then
    FOnSwitchBoard(Self, TrID, SBAddress, Cookie);
end;

procedure TMsnConnection.DoCalled(SessionID, SBAddress, Cookie, CallingUserAccount: String;
  CallingUserName: WideString);
begin
  if Assigned(FOnCalled) then
    FOnCalled(Self, SessionID, SBAddress, Cookie, CallingUserAccount, CallingUserName);
end;

procedure TMsnConnection.DoReceiveMessage(Header: UTF8String;
  FromAccount: String; FromName, Msg: WideString);
begin
  if Assigned(FOnReceiveMessage) then
    FOnReceiveMessage(Self, Header, FromAccount, FromName, Msg);
end;

procedure TMsnConnection.DoUrl(rru, passport: String);
begin
  if Assigned(FOnUrl) then
    FOnUrl(Self, rru, passport);
end;

procedure TMsnConnection.DoNewMail(FromName: WideString; FromAddr: String);
begin
  if Assigned(FOnNewMail) then
    FOnNewMail(Self, FromName, FromAddr);
end;

procedure TMsnConnection.DoUnreadMailChange(Init: Boolean);
begin
  if Assigned(FOnUnreadMailChange) then
    FOnUnreadMailChange(Self, Init);
end;

// TCC
procedure TMsnConnection.SignIn(Account, Password: String; Name: WideString;
  Status: TMsnMemberStatus);
begin
  FSignOutType := otUnKnown;
  FSignInStage := ssTryConnectServer;
  // oXg
  FMembers.Clear;
  FReverceMembers.Clear;
  FAllowMembers.Clear;
  FBlockMembers.Clear;
  FGroups.Clear;
  FMembers.Updated := False;
  FReverceMembers.Updated := False;
  FAllowMembers.Updated := False;
  FBlockMembers.Updated := False;
  // [U
  FUser.Account := Account;
  FUser.Password := Password;
  FUser.Status := Status;
  FUser.Name := Name;
  // \Pbg
  FSocket.Close;
  FSocket.Address := '';
  FSocket.Host := FHost;
  FSocket.Port := FPort;
  FSocket.Open;
end;

// TCAEg
procedure TMsnConnection.SignOut;
begin
  SendCommand('OUT'#13#10);
end;

// [ȔԂύX
function TMsnConnection.ChangeUserStatus(Status: TMsnMemberStatus): Integer;
begin
 	Inc(FTrID);
  SendCommand(Format('CHG %d %s'#13#10, [FTrID, MemberStatusToStr(Status)]));
  Result := FTrID;
end;

// [UoύX
function TMsnConnection.RenameMember(Account: String; NewName: WideString): Integer;
begin
 	Inc(FTrID);
  SendCommand(Format('REA %d %s %s'#13#10,
      [FTrID, Account, EncodeParam(NewName)]));
  Result := FTrID;
end;

// oXgɒǉ
function TMsnConnection.AddMember(List: TListKind; Account: String;
  GroupId: Integer = -1): Integer;
var
  Name: UTF8String;
begin
  if FMembers.Contains(Account) then
    Name := EncodeParam(FMembers.Find(Account).Name)
  else
    Name := Account;

 	Inc(FTrID);
  if GroupId <> -1 then
    SendCommand(Format('ADD %d %s %s %s %d'#13#10,
        [FTrID, ListKindToStr(List), Account, Name, GroupId]))
  else
    SendCommand(Format('ADD %d %s %s %s'#13#10,
        [FTrID, ListKindToStr(List), Account, Name]));
  Result := FTrID;
end;

// oXg폜
function TMsnConnection.RemoveMember(List: TListKind; Account: String;
  GroupId: Integer = -1): Integer;
begin
 	Inc(FTrID);
  if GroupId = -1 then
    SendCommand(Format('REM %d %s %s'#13#10,
        [FTrID, ListKindToStr(List), Account]))
  else
    SendCommand(Format('REM %d %s %s %d'#13#10,
        [FTrID, ListKindToStr(List), Account, GroupId]));
  Result := FTrID;
end;

// O[vǉ
function TMsnConnection.AddGroup(Name: WideString): Integer;
begin
 	Inc(FTrID);
 	SendCommand(Format('ADG %d %s 0'#13#10, [FTrID, EncodeParam(Name)]));
  Result := FTrID;
end;

// O[v폜
function TMsnConnection.RemoveGroup(GroupId: Integer): Integer;
begin
 	Inc(FTrID);
  SendCommand(Format('RMG %d %d'#13#10, [FTrID, GroupId]));
  Result := FTrID;
end;

// O[vύX
function TMsnConnection.RenameGroup(GroupId: Integer; NewName: WideString): Integer;
begin
 	Inc(FTrID);
  SendCommand(Format('REG %d %d %s 0'#13#10,
      [FTrID, GroupId, EncodeParam(NewName)]));
  Result := FTrID;
end;

// SwitchBoard Server Љv𑗐M
function TMsnConnection.SwitchBoardRequest: Integer;
begin
 	Inc(FTrID);
  SendCommand(Format('XFR %d SB'#13#10, [FTrID]));
  Result := FTrID;
end;

// o֎~Xgɒǉ
procedure TMsnConnection.BlockMember(Account: String);
begin
  if FAllowMembers.Contains(Account) then
    RemoveMember(lkAL, Account);
  if not FBlockMembers.Contains(Account) then
    AddMember(lkBL, Account, -1);
end;

// oXgɒǉ
procedure TMsnConnection.AllowMember(Account: String);
begin
  if FBlockMembers.Contains(Account) then
    RemoveMember(lkBL, Account);
  if not FAllowMembers.Contains(Account) then
    AddMember(lkAL, Account, -1);
end;

// URL ₢킹
function TMsnConnection.QueryUrl(Param: String): Integer;
begin
 	Inc(FTrID);
 	SendCommand(Format('URL %d %s'#13#10, [FTrID, Param]));
  Result := FTrID;
end;

// Ping M
procedure TMsnConnection.Ping;
begin
 	SendCommand('PNG'#13#10);
end;

// GTC ݒ
procedure TMsnConnection.SetGTC(Value: Boolean);
begin
  Inc(FTrID);
  if Value then
    SendCommand(Format('GTC %d A'#13#10, [FTrID]))
  else
    SendCommand(Format('GTC %d N'#13#10, [FTrID]));
end;

// BLP ݒ
procedure TMsnConnection.SetBLP(Value: TBLP);
begin
  Inc(FTrID);
  case Value of
  bpAL:
    SendCommand(Format('BLP %d AL'#13#10, [FTrID]));
  bpBL:
    SendCommand(Format('BLP %d BL'#13#10, [FTrID]));
  end;
end;

// \PbgIuWFNg쐬
procedure TMsnConnection.CreateSocket;
begin
  if not Assigned(FSocket) then
  begin
    FSocket := TClientSocket.Create(nil);
    FSocket.OnConnect     := SocketConnect;
    FSocket.OnConnecting  := SocketConnecting;
    FSocket.OnDisconnect  := SocketDisconnect;
    FSocket.OnRead        := SocketRead;
    FSocket.OnError       := SocketError;
  end;
end;

// \PbgIuWFNgj
procedure TMsnConnection.DestroySocket;
begin
  if Assigned(FSocket) then
  begin
    FreeAndNil(FSocket);
  end;
end;

// \PbgڑCxgnh
procedure TMsnConnection.SocketConnecting(Sender: TObject;
  Socket: TCustomWinSocket);
var
  HostName: String;
begin
  if FSocket.Host <> '' then
    HostName := FSocket.Host
  else
    HostName := FSocket.Address;
  // T[oɐڑ
  FSignInStage := ssTryConnectServer;
  DoLog('-', Format('Connecting => %s:%d', [HostName, FSocket.Port]));
end;

// \PbgڑCxgnh
procedure TMsnConnection.SocketConnect(Sender: TObject;
  Socket: TCustomWinSocket);
begin
  FSignOutType := otUnKnown;
  FSignInStage := ssTrySignIn;
  // TCCJn
  FTrID := 0;
  SendCommand(Format('VER %d %s'#13#10, [FTrID, MSN_VERSION]));
end;

// \PbgؒfCxg
procedure TMsnConnection.SocketDisconnect(Sender: TObject;
  Socket: TCustomWinSocket);
begin
  // ڑ
  FSignInStage := ssUnConnect;
  DoLog('-', 'Disconnect');
  // oXg
  FMembers.Clear;
  FReverceMembers.Clear;
  FAllowMembers.Clear;
  FBlockMembers.Clear;
  FGroups.Clear;
  // TCAEgCxg
  DoSignOut(FSignOutType);
end;

// \PbgG[Cxgnh
procedure TMsnConnection.SocketError(Sender: TObject; Socket: TCustomWinSocket;
  ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
  // Socket G[
  DoLog('-', Format('Socket Error #%d',[ErrorCode]));
  // G[Cxg
  DoError(ekSocketError, ErrorCode);

  ErrorCode := 0;
  FSocket.Close;
  if FSignInStage <> ssUnConnect then
  begin
    FSignInStage := ssUnConnect;
    FSignOutType := otUnKnown;
    // TCAEgCxg
    DoSignOut(FSignOutType);
  end;
end;

// \PbgMCxgnh
procedure TMsnConnection.SocketRead(Sender: TObject; Socket: TCustomWinSocket);
var
  DataStr: Utf8String;
  CommandLst: TStringList;
  ParamLst: TStringList;
begin
  CommandLst := TStringList.Create;
  ParamLst := TStringList.Create;
  DataStr := Socket.ReceiveText;
  DoLog('S', Utf8ToAnsiEx(DataStr));
  DataStr := FIncompleteCommand + DataStr;

  FIncompleteCommand := SplitCommandStr(CommandLst, DataStr);
  while CommandLst.Count > 0 do
  begin
    SplitParamStr(ParamLst, CommandLst[0]);

    // G[
    if StrToIntDef(SS(ParamLst, 0), -1) <> -1 then
      ProcessError(ParamLst)
    // T[oؒfꂽ
    else if (SS(ParamLst, 0) = 'OUT') and (SS(ParamLst, 1) = 'OTH') then
      FSignOutType := otOTH
    else if (SS(ParamLst, 0) = 'OUT') and (SS(ParamLst, 1) = 'SSD') then
      FSignOutType := otSSD
    // ȍԂω
    else if (SS(ParamLst, 0) = 'CHG') or (SS(ParamLst, 0) = 'NLN') or
            (SS(ParamLst, 0) = 'FLN') or (SS(ParamLst, 0) = 'ILN') then
      ProcessChangeStatus(ParamLst)
    // o̖OύXꂽ
    else if (SS(ParamLst, 0) = 'REA') then
      ProcessChangeName(ParamLst)
    // bZ[WM
    else if SS(ParamLst, 0) = 'MSG' then
      ProcessMessage(CommandLst[0])
    // bɏ҂ꂽ
    else if (SS(ParamLst, 0) = 'RNG') then
      DoCalled(ParamLst[1], ParamLst[2], ParamLst[4], ParamLst[5], DecodeParam(ParamLst[6]))
    // SwitchBoard Server
    else if (SS(ParamLst, 0) = 'XFR') and (SS(ParamLst, 2) = 'SB') then
      DoSwitchBoard(StrToInt(ParamLst[1]), ParamLst[3], ParamLst[5])
    // oXgύX
    else if (SS(ParamLst, 0) = 'LST') or
            (SS(ParamLst, 0) = 'ADD') or (SS(ParamLst, 0) = 'REM') then
      ProcessMemberList(ParamLst)
    // O[vXgύX
    else if ((SS(ParamLst, 0) = 'LSG') and (SS(ParamLst, 4) <> '0')) or
            (SS(ParamLst, 0) = 'ADG') or (SS(ParamLst, 0) = 'REG') or
            (SS(ParamLst, 0) = 'RMG') then
      ProcessGroupList(ParamLst)
    // vCoV[ݒ̕ύX
    else if (SS(ParamLst, 0) = 'GTC') or (SS(ParamLst, 0) = 'BLP') then
      ProcessPrivacySetting(ParamLst)
    // OCԂ̃`FbNH
    else if (SS(ParamLst, 0) = 'CHL') then
      ProcessCHL(ParamLst)
    // URL
    else if (SS(ParamLst, 0) = 'URL') then
      DoUrl(ParamLst[2], ParamLst[3])
    else if FSignInStage <> ssSignIn then
      ProcessSignIn(ParamLst);

    CommandLst.Delete(0);
  end;

  ParamLst.Free;
  CommandLst.Free;
end;

// TCCV[PX
procedure TMsnConnection.ProcessSignIn(ParamLst: TStringList);
var
  MsgID: Integer;
  sTemp: String;
  P: Integer;
begin
  MsgID := StrToIntDef(SS(ParamLst, 1), -1);
	if MsgID = 0 then
  begin
    // OCJn
    Inc(FTrID);
    SendCommand(Format('INF %d'#13#10, [FTrID]));
	end
  else if MsgID = 1 then
  begin
    // [UF
    Inc(FTrID);
    SendCommand(Format('USR %d MD5 I %s'#13#10, [FTrID, FUser.Account]));
	end
	else if MsgID = 2 then
  begin
    // ʂ NS ɍĐڑ
    if ParamLst[0] = 'XFR' then
    begin
      FSignOutType := otXFR;
      FSocket.Close;
      DestroySocket;
      CreateSocket;
      P := AnsiPos(':', ParamLst[3]);
      FSocket.Address := LeftStr(ParamLst[3], P - 1);
      FSocket.Port := StrToIntDef(Copy(ParamLst[3], P + 1,
          Length(ParamLst[3]) - P), FPort);
      FSocket.Host := '';
      FSocket.Open;
  	end else
    begin
      // ÍpX[h𑗐M
      Inc(FTrID);
      sTemp := LowerCase(MD5Print(MD5String(Trim(ParamLst[4] + FUser.Password))));
      SendCommand(Format('USR %d MD5 S %s'#13#10, [FTrID, sTemp]));
    end;
	end
  else if MsgID = 3 then
  begin
    if (ParamLst[0] = 'USR') and (ParamLst[2] = 'OK') then
    begin
      FSignInStage := ssSignIn;
      // ŐV擾
     	Inc(FTrID);
      SendCommand(Format('SYN %d 0'#13#10, [FTrID]));
       // OύX
      if Length(FUser.Name) > 0 then
        RenameMember(FUser.Account, FUser.Name);
      // ԕύX
     	Inc(FTrID);
      SendCommand(Format('CHG %d %s'#13#10,
          [FTrID, MemberStatusToStr(FUser.Status)]));
      FUser.Name := DecodeParam(ParamLst[4]);
      FUser.Status := usFLN;
      // TCCCxg
      DoSignIn;
    end else
    begin
      // TCCsۂ
      FSocket.Close;
    end;
  end;
end;

// o̖OύX
procedure TMsnConnection.ProcessChangeName(ParamLst: TStringList);
var
  Member: TMsnMemberBase;
  NewName: WideString;
begin
  if ParamLst[0] = 'REA' then
  begin
    NewName := DecodeParam(ParamLst[4]);
    if ParamLst[3] = FUser.Account then
      FUser.Name := NewName;

    // SẴXgŖOύX
    Member := FMembers.Find(ParamLst[3]);
    if Member <> nil then
      Member.Name := NewName;
    Member := FReverceMembers.Find(ParamLst[3]);
    if Member <> nil then
      Member.Name := NewName;
    Member := FBlockMembers.Find(ParamLst[3]);
    if Member <> nil then
      Member.Name := NewName;
    Member := FAllowMembers.Find(ParamLst[3]);
    if Member <> nil then
      Member.Name := NewName;

    Member := FMembers.Find(ParamLst[3]);
    if not Assigned(Member) and AnsiSameStr(FUser.Account, ParamLst[3]) then
      Member := FUser;
    // oύXCxg
    if Assigned(Member) then
      DoMemberNameChange(Member);
  end
end;

// ȍԂύX
procedure TMsnConnection.ProcessChangeStatus(ParamLst: TStringList);
var
  Member: TMsnMemberBase;
  OldStatus: TMsnMemberStatus;
  OldName: WideString;
  InitList: Boolean;
begin
  Member := nil;
  OldStatus := usFLN;
  InitList := False;
  
  // [ȔԂύX
  if ParamLst[0] = 'CHG' then
  begin
    OldStatus := FUser.Status;
    FUser.Status := StrToMemberStatus(ParamLst[2]);
    Member := TMsnMember(FUser);
  end
  // oIC
  else if ParamLst[0] = 'NLN' then
  begin
    Member := FMembers.Find(ParamLst[2]);
    if Member <> nil then
    begin
      OldName := Member.Name;
      OldStatus := Member.Status;
      Member.Status := StrToMemberStatus(ParamLst[1]);
      Member.Name := DecodeParam(ParamLst[3]);
      // o^ύX
      if Member.Name <> OldName then
        RenameMember(Member.Account, Member.Name);
      // ICCxg
      if OldStatus = usFLN then
        DoMemberOnline(Member);
    end;
  end
  //@oItC
  else if ParamLst[0] = 'FLN' then
  begin
    Member := FMembers.Find(ParamLst[1]);
    if Member <> nil then
    begin
      OldStatus := Member.Status;
      Member.Status := usFLN;
      // ItCCxg
      if OldStatus <> usFLN then
        DoMemberOffline(Member);
    end;
  end
  //@ȍԂύX
  else if ParamLst[0] = 'ILN' then
  begin
    InitList := True;
    Member := FMembers.Find(ParamLst[3]);
    if Member <> nil then
    begin
      OldStatus := Member.Status;
      Member.Status := StrToMemberStatus(ParamLst[2]);
      Member.Name := DecodeParam(ParamLst[4]);
      // o^XV
      if (Member.Status <> usFLN) and (Member.Name <> OldName) then
        RenameMember(Member.Account, Member.Name);
    end;
  end;

  // oԕύXCxg
  if Member <> nil then
    DoMemberStatusChange(Member, OldStatus, InitList);
end;

// FBXg͂č쐬
procedure TMsnConnection.ProcessMemberList(ParamLst: TStringList);
var
  Idx: Integer;
  List: TMsnMemberList;
  ListKind: TListKind;
  Member: TMsnMember;
begin
  if ParamLst[2] = 'FL' then
  begin
    ListKind := lkFL;
    List := FMembers;
  end else
  if ParamLst[2] = 'RL' then
  begin
    ListKind := lkRL;
    List := FReverceMembers;
  end else
  if ParamLst[2] = 'AL' then
  begin
    ListKind := lkAL;
    List := FAllowMembers;
  end else
  if ParamLst[2] = 'BL' then
  begin
    ListKind := lkBL;
    List := FBlockMembers;
  end else
    Exit;

  if (ParamLst[0] = 'LST') and (ParamLst[4] <> '0') then
  begin
    Member := List.Add;
    Member.Status := usFLN;
    Member.Account := ParamLst[6];
    Member.Name := DecodeParam(ParamLst[7]);
    if ListKind = lkFL then
      Split(Member.Groups, SS(ParamLst, 8), ',');
    if Member.Groups.Count = 0 then
      Member.Groups.Add(0);
    // oǉCxg
    DoMemberAddition(ListKind, Member);
    List.Updated := (ParamLst[4] = ParamLst[5]);
    // SẴoMI
    if List.Updated then
      DoMemberListUpdated(ListKind);
  end else
  if (ParamLst[0] = 'LST') and (ParamLst[4] = '0') then
  begin
    List.Updated := True;
    DoMemberListUpdated(ListKind);
  end else
  // Xgɒǉ
  if (ParamLst[0] = 'ADD') then
  begin
    Member := List.Find(ParamLst[4]);
    if Member = nil then    // VKɃoǉ
    begin
      Member := List.Add;
      with Member do
      begin
        Status := usFLN;
        Account := ParamLst[4];
        Name := DecodeParam(ParamLst[5]);
        if ListKind = lkFL then
          Split(Groups, SS(ParamLst, 6), ',');
        if SS(ParamLst, 6) <> '' then
          Member.Groups.Add(StrToIntDef(SS(ParamLst, 6), 0));
        if Member.Groups.Count = 0 then
          Member.Groups.Add(0);
        // oǉCxg
        DoMemberAddition(ListKind, Member);
      end;
    end else                // ̃oɃO[vǉ
    begin
      if SS(ParamLst, 6) <> '' then
        Member.Groups.Add(StrToIntDef(SS(ParamLst, 6), 0));
      DoMemberGroupChange(Member);
    end;
  end else
  // Xg폜
  if (ParamLst[0] = 'REM') then
  begin
    Idx := List.IndexOf(ParamLst[4]);
    if Idx <> -1 then
    begin
      if SS(ParamLst, 5) = '' then    // o폜
      begin
        // o폜Cxg
        DoMemberDeletion(ListKind, List[Idx]);
        List.Delete(Idx);
      end else                        // ̃O[v폜
      begin
        List[Idx].Groups.Remove(StrToIntDef(ParamLst[5], -1));
        if List[Idx].Groups.Count = 0 then
        begin
          // o폜Cxg
          DoMemberDeletion(ListKind, List[Idx]);
          List.Delete(Idx);
        end else
          DoMemberGroupChange(List[Idx]);
      end;
    end;
  end;

  // oXgύXCxg
  DoMemberListChange(ListKind);
end;

// O[vXg쐬
procedure TMsnConnection.ProcessGroupList(ParamLst: TStringList);
var
  I: Integer;
  Group: TMsnGroup;
begin
  if ParamLst[0] = 'LSG' then
  begin
    Group := FGroups.Add;
    with Group do
    begin
      Id := StrToIntDef(ParamLst[5], 0);
      Name := DecodeParam(ParamLst[6]);
    end;
    DoGroupAddition(Group);
    DoGroupListChange;
  end else
  if ParamLst[0] = 'ADG' then
  begin
    Group := FGroups.Add;
    with Group do
    begin
      Id := StrtoIntDef(ParamLst[4], 0);
      Name := DecodeParam(ParamLst[3]);
    end;
    DoGroupAddition(Group);
    DoGroupListChange;
  end else
  if ParamLst[0] = 'REG' then
  begin
    I := FGroups.IndexOf(StrtoIntDef(ParamLst[3], -1));
    if I <> -1 then
    begin
      FGroups[I].Name := DecodeParam(ParamLst[4]);
      DoGroupNameChange(FGroups[I]);
    end;
  end else
  if ParamLst[0] = 'RMG' then
  begin
    I := FGroups.IndexOf(StrToIntDef(ParamLst[3], -1));
    if I <> -1 then
    begin
      DoGroupDeletion(FGroups[I]);
      FGroups.Delete(I);
      DoGroupListChange;
    end;
  end;
end;

// GTC BLPM
procedure TMsnConnection.ProcessPrivacySetting(ParamLst: TStringList);
begin
  if (ParamLst[0] = 'GTC') then
  begin
    if ParamLst[3] = 'A' then
      FGTC := True
    else if ParamLst[3] = 'N' then
      FGTC := False;
  end else
  if (ParamLst[0] = 'BLP') then
  begin
    if ParamLst[3] = 'AL' then
      FBLP := bpAL
    else if ParamLst[3] = 'BL' then
      FBLP := bpBL;
  end;
end;

// CHLɕԓ
procedure TMsnConnection.ProcessCHL(ParamLst: TStringList);
var
  sTemp: String;
begin
  Inc(FTrID);
  sTemp :=
      LowerCase(MD5Print(MD5String(Trim(ParamLst[2]) + 'Q1P7W2E4J9R8U3S5')));
  SendCommand(Format('QRY %d msmsgs@msnmsgr.com 32'#13#10, [FTrID]));
  SendCommand(sTemp);
end;

// G[ĂƂ
procedure TMsnConnection.ProcessError(ParamLst: TStringList);
begin
  DoError(ekMsnError, StrToInt(ParamLst[0]));
end;

// bZ[WM
procedure TMsnConnection.ProcessMessage(DataStr: UTF8String);
var
  Command, Header: UTF8String;
  Msg, FromName, FromAccount: WideString;
  ParamLst: TStringList;
  P: Integer;
begin
  P := Pos(#13#10, DataStr);
  if P > 0 then
    Command := Copy(DataStr, 1, P - 1)
  else
    Command := DataStr;
  ParamLst := TStringList.Create;
  try
    SplitParamStr(ParamLst, Command);

    FromAccount := UTF8Decode(ParamLst[1]);
    FromName := DecodeParam(ParamLst[2]);

    Delete(DataStr, 1, Length(Command) + 2);
    P := Pos(#13#10#13#10, DataStr);
    if P > 0 then
    begin
      Header := Copy(DataStr, 1, P - 1);
      Msg := Copy(DataStr, P + 4, Length(DataStr));
    end else
    begin
      Header := DataStr;
      Msg := '';
    end;
    // bZ[WMCxg
    DoReceiveMessage(Header, FromAccount, FromName, Msg);
    ProcessHotmail(Header, FromAccount, FromName, Msg);
  finally
    ParamLst.Free;
  end;
end;

procedure TMsnConnection.ProcessHotmail(Header: UTF8String; FromAccount: String;
  FromName, Msg: WideString);
var
  Lines: TStringList;
  ContentType: String;
  Init: Boolean;
  NewMail: Boolean;
  Active: Boolean;
  FromAddr, From, SrcFolder, DestFolder: String;
  MessageDelta: Integer;
begin
  Lines := TStringList.Create;
  SplitMimeHeader(Lines, Header + #13#10 +  Msg);

  ContentType := Lines.Values['Content-Type'];
  if AnsiContainsStr(ContentType, 'text/x-msmsgsprofile') then
  begin
    with FPassportInfo do
    begin
      LoginTime := StrToIntDef(Lines.Values['LoginTime'], 0);
      EmailEnabled := (StrToIntDef(Lines.Values['EmailEnabled'], 0) = 1);
      MemberIdHigh := StrToIntDef(Lines.Values['MemberIdHigh'], 0);
      MemberIdLow := StrToIntDef(Lines.Values['MemberIdLow'], 0);
      lang_preference := StrToIntDef(Lines.Values['lang_preference'], 0);
      preferredEmail := Lines.Values['preferredEmail'];
      country := Lines.Values['country'];
      PostalCode := Lines.Values['PostalCode'];
      Gender := Lines.Values['Gender'];
      Kid := StrToIntDef(Lines.Values['Kid'], 0);
      Age := StrToIntDef(Lines.Values['Age'], 0);
      sid := StrToIntDef(Lines.Values['sid'], 0);
      kv := StrToIntDef(Lines.Values['kv'], 0);
      MSPAuth := Lines.Values['MSPAuth'];
      sl := DateTimeToFileDate(Now);
    end;
  end else
  begin
    Init := AnsiContainsStr(ContentType, 'text/x-msmsgsinitialemailnotification');
    NewMail := AnsiContainsStr(ContentType, 'text/x-msmsgsemailnotification');
    Active := AnsiContainsStr(ContentType, 'text/x-msmsgsactivemailnotification');
    // init
    FInboxUnread := StrToIntDef(Lines.Values['Inbox-Unread'], FInboxUnread);
    FFoldersUnread := StrToIntDef(Lines.Values['Folders-Unread'], FFoldersUnread);
    // newmail
    From := DecodeParam(Lines.Values['From']);
    FromAddr := Lines.Values['From-Addr'];
    SrcFolder := Lines.Values['Src-Folder'];
    DestFolder := Lines.Values['Dest-Folder'];
    MessageDelta := StrToIntDef(Lines.Values['Message-Delta'], 0);

    // V[̍XV
    if Init then
    begin
      if FInboxUnread + FFoldersUnread > 0 then
        DoUnreadMailChange(True);
    end else
    if NewMail then
    begin
      if (DestFolder = 'ACTIVE') then
        Inc(FInboxUnread)
      else
        Inc(FFoldersUnread);
      DoUnreadMailChange(False);
      DoNewMail(From, FromAddr);
    end else
    if Active then
    begin
      if (SrcFolder = 'ACTIVE') then
        Dec(FInboxUnread, MessageDelta)
      else
        Dec(FFoldersUnread, MessageDelta);
      DoUnreadMailChange(False);
    end;
  end;
  Lines.Free;
end;

// TListKind^𕶎ɕϊ
function TMsnConnection.ListKindToStr(List: TListKind): String;
begin
  case List of
    lkFL : Result := 'FL';
    lkRL : Result := 'RL';
    lkAL : Result := 'AL';
    lkBL : Result := 'BL';
  end;
end;

// -----------------------------------------------------------------------------
constructor TMsnSession.Create;
begin
  FSocket := TClientSocket.Create(nil);
  FSocket.OnConnect     := SocketConnect;
  FSocket.OnConnecting  := SocketConnecting;
  FSocket.OnDisconnect  := SocketDisconnect;
  FSocket.OnRead        := SocketRead;
  FSocket.OnError       := SocketError;

  FUser := TMsnUser.Create;
  FMembers := TMsnMemberList.Create;
  FReservedMembers := TStringList.Create;
  FReservedMessages := TStringList.Create;

  FHost := '';
  FPort := MSN_DEFAULTPORT;
  FSignInStage := ssUnConnect;
end;

destructor TMsnSession.Destroy;
begin
  FUser.Free;
  FMembers.Free;
  FSocket.Free;
  FReservedMembers.Free;
  FReservedMessages.Free;
  inherited;
end;

procedure TMsnSession.DoConnect;
begin
  if Assigned(FOnConnect) then
    FOnConnect(Self);
end;

procedure TMsnSession.DoDisconnect;
begin
  if Assigned(FOnDisconnect) then
    FOnDisconnect(Self);
end;

procedure TMsnSession.DoLog(Head, Str: String);
begin
  if Assigned(FOnLog) then
    FOnLog(Self, Head + #9 + Str);
end;

procedure TMsnSession.DoError(ErrorKind: TErrorKind; ErrorCode: Integer);
begin
  if Assigned(FOnError) then
    FOnError(Self, ErrorKind, ErrorCode);
end;

procedure TMsnSession.DoMemberListChange;
begin
  if Assigned(FOnMemberListChange) then
    FOnMemberListChange(Self);
end;

procedure TMsnSession.DoRecieveMessage(Header: UTF8String; FromAccount: String;
    FromName, Msg: WideString);
begin
  if Assigned(FOnRecieveMessage) then
    FOnRecieveMessage(Self, Header, FromAccount, FromName, Msg);
end;

procedure TMsnSession.DoJoinMember(Member: TMsnMember);
begin
  if Assigned(FOnJoinMember) then
    FOnJoinMember(Self, Member);
end;

procedure TMsnSession.DoByeMember(Member: TMsnMember);
begin
  if Assigned(FOnByeMember) then
    FOnByeMember(Self, Member);
end;

// ڑ
procedure TMsnSession.Connect(AHost: String; APort: Integer; Account, Cookie,
    SessionID: String);
begin
  FUser.Account := Account;
  FCookie       := Cookie;
  FSessionID    := SessionID;
  FHost         := AHost;
  FPort         := APort;

  FSocket.Close;
  FSocket.Address := FHost;
  FSocket.Host    := '';
  FSocket.Port    := FPort;
  FSocket.Open;
end;

// ؒf
procedure TMsnSession.Disconnect;
begin
  FSocket.Close;
end;

// bZ[W𑗐M
procedure TMsnSession.SendMessage(Msg: WideString);
var
  Utf8Msg: UTF8String;
begin
  if (FSignInStage = ssSignIn) and (FMembers.Count > 0) then
  begin
    Inc(FTrID);
    Utf8Msg := UTF8Encode(Msg);
    SendCommand(Format('MSG %d U %d'#13#10, [FTrID, Length(Utf8Msg)]));
    SendCommand(Utf8Msg);
  end else
    FReservedMessages.Add(Msg);
end;

// o
procedure TMsnSession.CallMember(Account: String);
begin
  if FSignInStage = ssSignIn then
  begin
   	Inc(FTrID);
    SendCommand(Format('CAL %d %s'#13#10, [FTrID, Account]));
  end else
    FReservedMembers.Add(Account);
end;

function TMsnSession.SingleMember(Account: String): Boolean;
begin
  Result := ((FMembers.Count = 1) and (FMembers[0].Account = Account)) or
            ((FMembers.Count = 0) and (FReservedMembers.Count = 1) and
             (FReservedMembers[0] = Account));
end;

procedure TMsnSession.SocketConnecting(Sender: TObject;
  Socket: TCustomWinSocket);
begin
  // T[oɐڑ
  FSignInStage := ssTryConnectServer;
  DoLog('-', Format('Connecting => %s:%d', [FSocket.Address, FSocket.Port]));
end;

procedure TMsnSession.SocketConnect(Sender: TObject;
  Socket: TCustomWinSocket);
begin
  // TCCv
  FSignInStage := ssTrySignIn;

  FTrID := 0;
  // [UZbVmꍇ
  if FSessionID = '' then
    SendCommand(Format('USR %d %s %s'#13#10, [FTrID, FUser.Account, FCookie]))
  // bɏ҂ꂽꍇ
  else
    SendCommand(Format('ANS %d %s %s %s'#13#10,
        [FTrID, FUser.Account, FCookie, FSessionID]));
end;

procedure TMsnSession.SocketDisconnect(Sender: TObject;
  Socket: TCustomWinSocket);
begin
  // ڑ
  FSignInStage := ssUnConnect;
  DoLog('-', 'Disconnect');
  FMembers.Clear;

  // TCAEgCxg
  DoDisconnect;
end;

procedure TMsnSession.SocketError(Sender: TObject; Socket: TCustomWinSocket;
  ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
  // Socket G[
  DoLog('-', Format('Socket Error #%d', [ErrorCode]));
  DoError(ekSocketError, ErrorCode);

  ErrorCode := 0;
  if FSignInStage <> ssUnConnect then
  begin
    FMembers.Clear;
    DoDisconnect;
    FSignInStage := ssUnConnect;
  end;
end;

procedure TMsnSession.SocketRead(Sender: TObject; Socket: TCustomWinSocket);
var
  DataStr: Utf8String;
  CommandLst: TStringList;
  ParamLst: TStringList;
begin
  CommandLst := TStringList.Create;
  ParamLst := TStringList.Create;
  DataStr := Socket.ReceiveText;
  DoLog('S', Utf8ToAnsiEx(DataStr));
  DataStr := FIncompleteCommand + DataStr;

  FIncompleteCommand := SplitCommandStr(CommandLst, DataStr);
  while CommandLst.Count > 0 do
  begin
    SplitParamStr(ParamLst, CommandLst[0]);

    if ParamLst.Count = 0 then
      //
    // G[
    else if StrToIntDef(ParamLst[0], -1) <> -1 then
      ErrorHandler(ParamLst)
    else if ((ParamLst[0] = 'USR') and (ParamLst[1] = '0') and (ParamLst[2] = 'OK')) or
            ((ParamLst[0] = 'ANS') and (ParamLst[1] = '0') and (ParamLst[2] = 'OK')) then
    begin
      FSignInStage := ssSignIn;
      // ڑCxg
      DoConnect;
      CallReservedMembers;
    end
    // bQ҂̏
    else if (ParamLst[0] = 'IRO') or (ParamLst[0] = 'JOI') or
            (ParamLst[0] = 'BYE') then
      SetMemberList(ParamLst)
    // bZ[WM
    else if (ParamLst[0] = 'MSG') then
      ReceiveMessage(CommandLst[0]);
    CommandLst.Delete(0);
  end;

  ParamLst.Free;
  CommandLst.Free;
end;

// R}hM
procedure TMsnSession.SendCommand(Str: Utf8String);
begin
  FSocket.Socket.SendText(Str);
  DoLog('C', Utf8ToAnsi(Str));
end;

// oXgύX
procedure TMsnSession.SetMemberList(ParamLst: TStringList);
var
  Member: TMsnMember;
  I: Integer;
begin
  // bQ҂̏
  if (ParamLst[0] = 'IRO') then
  begin
    Member := FMembers.Add;
    with Member do
    begin
      Account := ParamLst[4];
      Name := DecodeParam(ParamLst[5]);
    end;
    // oQCxg
    DoJoinMember(Member);
  end
  // oQ
  else if (ParamLst[0] = 'JOI') then
  begin
    Member := FMembers.Add;
    Member.Account := ParamLst[1];
    Member.Name := DecodeParam(ParamLst[2]);
    // oQCxg
    DoJoinMember(Member);
  end
  // o[ގ
  else if (ParamLst[0] = 'BYE') then
  begin
    I := FMembers.IndexOf(ParamLst[1]);
    if I <> -1 then
    begin
      DoByeMember(FMembers[I]);
      FMembers.Delete(I);
    end;
  end;
  if (FMembers.Count > 0) and (FSignInStage = ssSignIn) then
    SendReservedMessages;

  DoMemberListChange;
end;

// bZ[WM
procedure TMsnSession.ReceiveMessage(DataStr: Utf8String);
var
  Command, Header: UTF8String;
  Msg, FromName, FromAccount: WideString;
  ParamLst: TStringList;
  P: Integer;
begin
  P := Pos(#13#10, DataStr);
  if P > 0 then
    Command := Copy(DataStr, 1, P - 1)
  else
    Command := DataStr;
  ParamLst := TStringList.Create;
  SplitParamStr(ParamLst, Command);

  FromAccount := UTF8Decode(ParamLst[1]);
  FromName := DecodeParam(ParamLst[2]);

  Delete(DataStr, 1, Length(Command) + 2);
  P := Pos(#13#10#13#10, DataStr);
  if P > 0 then
  begin
    Header := Copy(DataStr, 1, P - 1);
    Msg := UTF8Decode(Copy(DataStr, P + 4, Length(DataStr)));
  end else
  begin
    Header := DataStr;
    Msg := '';
  end;

  // bZ[WMCxg
  DoRecieveMessage(Header, FromAccount, FromName, Msg);

  ParamLst.Free;
end;

procedure TMsnSession.CallReservedMembers;
begin
  while FReservedMembers.Count > 0 do
  begin
    CallMember(FReservedMembers[0]);
    FReservedMembers.Delete(0);
  end;
end;

procedure TMsnSession.SendReservedMessages;
begin
  while FReservedMessages.Count > 0 do
  begin
    SendMessage(FReservedMessages[0]);
    FReservedMessages.Delete(0);
  end;
end;

// G[ĂƂ
procedure TMsnSession.ErrorHandler(ParamLst: TStringList);
begin
  // G[Cxg
  DoError(ekMsnError, StrToInt(ParamLst[0]));
end;

// -----------------------------------------------------------------------------
constructor TMsnConnectionList.Create;
begin
  FConnections := TList.Create;
end;

destructor TMsnConnectionList.Destroy;
begin
  Clear;
  FConnections.Free;
end;

function TMsnConnectionList.GetConnection(Index: Integer): TMsnConnection;
begin
  Result := TMsnConnection(FConnections[Index]);
end;

function TMsnConnectionList.GetCount: Integer;
begin
  Result := FConnections.Count;
end;

// ǉ
function TMsnConnectionList.Add: TMsnConnection;
begin
  Result := TMsnConnection.Create;
  FConnections.Add(Result);
end;


// 폜
procedure TMsnConnectionList.Delete(Idx: Integer);
begin
  TMsnConnection(FConnections[Idx]).Free;
  FConnections.Delete(Idx);
end;

// S
procedure TMsnConnectionList.Clear;
begin
  while FConnections.Count > 0 do
    Delete(0);
end;

// 
function TMsnConnectionList.IndexOf(AConnection: TMsnConnection): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FConnections.Count - 1 do
  begin
    if TMsnConnection(FConnections[I]) = AConnection then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TMsnConnectionList.IndexOfNsmHandle(NsmHandle: Cardinal): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FConnections.Count - 1 do
  begin
    if TMsnConnection(FConnections[I]).NsmHandle = NsmHandle then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TMsnConnectionList.IndexOfUser(Account: String): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FConnections.Count - 1 do
  begin
    if TMsnConnection(FConnections[I]).User.Account = Account then
    begin
      Result := I;
      Break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
constructor TMsnSessionList.Create;
begin
  FSessions := TList.Create;
end;

destructor TMsnSessionList.Destroy;
begin
  Clear;
  FSessions.Free;
end;

function TMsnSessionList.GetSession(Index: Integer): TMsnSession;
begin
  Result := TMsnSession(FSessions[Index]);
end;

function TMsnSessionList.GetCount: Integer;
begin
  Result := FSessions.Count;
end;

// ǉ
function TMsnSessionList.Add: TMsnSession;
begin
  Result := TMsnSession.Create;
  FSessions.Add(Result);
end;

// 폜
procedure TMsnSessionList.Delete(Idx: Integer);
begin
  TMsnSession(FSessions[Idx]).Free;
  FSessions.Delete(Idx);
end;

// S
procedure TMsnSessionList.Clear;
begin
  while FSessions.Count > 0 do
    Delete(0);
end;

// 
function TMsnSessionList.IndexOf(ASession: TMsnSession): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FSessions.Count - 1 do
  begin
    if TMsnSession(FSessions[I]) = ASession then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TMsnSessionList.IndexOfNsmHandle(NsmHandle: Cardinal): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FSessions.Count - 1 do
  begin
    if TMsnSession(FSessions[I]).NsmHandle = NsmHandle then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TMsnSessionList.IndexOfRequestID(RequestID: Integer): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FSessions.Count - 1 do
  begin
    if TMsnSession(FSessions[I]).RequestID = RequestID then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TMsnSessionList.IndexOfSingleMember(ASession: TMsnSession): Integer;
var
  I: Integer;
begin
  Result := -1;
  if (ASession.Members.Count <> 1) then Exit;
  for I := 0 to FSessions.Count - 1 do
  begin
    if (TMsnSession(FSessions[I]) <> ASession) and
       TMsnSession(FSessions[I]).SingleMember(ASession.Members[0].Account) then
    begin
      Result := I;
      Break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// G[R[hG[bZ[Wɕϊ
function GetMsnErrorMessage(ECode: Integer): String;
begin
  case ECode of
    ERR_SYNTAX_ERROR             : Result := '@G[łB';
    ERR_INVALID_PARAMETER        : Result := 'p[^łB';
    ERR_INVALID_USER             : Result := 'ȃ[UłB';
    ERR_FQDN_MISSING             : Result := 'hCsSłB';
    ERR_ALREADY_LOGIN            : Result := 'łɃOCς݂łB';
    ERR_INVALID_USERNAME         : Result := 'TCCłB';
    ERR_INVALID_FRIENDLY_NAME    : Result := '\łB';
    ERR_LIST_FULL                : Result := 'ȏナXgɒǉł܂B';
    ERR_ALREADY_THERE            : Result := 'łɑ݂܂B';
    ERR_NOT_ON_LIST              : Result := 'Xgɑ݂܂B';
    ERR_ALREADY_IN_THE_MODE      : Result := 'łɎw肳ꂽ[hłB';
    ERR_ALREADY_IN_OPPOSITE_LIST : Result := 'łɋt̃Xgɑ݂܂B';
    ERR_NOT_EXISTS_GROUP         : Result := 'ȃO[vłB';
    ERR_SWITCHBOARD_FAILED       : Result := 'ERR_SWITCHBOARD_FAILED';
    ERR_NOTIFY_XFR_FAILED        : Result := 'ERR_NOTIFY_XFR_FAILED';
    ERR_REQUIRED_FIELDS_MISSING  : Result := 'ERR_REQUIRED_FIELDS_MISSING';
    ERR_NOT_LOGGED_IN            : Result := 'OCĂ܂B';
    ERR_INTERNAL_SERVER          : Result := 'T[oG[łB';
    ERR_DB_SERVER                : Result := 'f[^x[XT[oG[łB';
    ERR_FILE_OPERATION           : Result := 't@CG[łB';
    ERR_MEMORY_ALLOC             : Result := '[mۂł܂B';
    ERR_WRONG_CHL_VALUE          : Result := 'ERR_WRONG_CHL_VALUE';
    ERR_SERVER_BUSY              : Result := 'T[orW[ԂłB';
    ERR_SERVER_UNAVAILABLE       : Result := 'T[opł܂B';
    ERR_PEER_NS_DOWN             : Result := 'ڑNST[oĂ܂B';
    ERR_DB_CONNECT               : Result := 'f[^x[Xɐڑł܂B';
    ERR_SERVER_GOING_DOWN        : Result := 'ԂȂT[o~܂B';
    ERR_CREATE_CONNECTION        : Result := 'RlNVmł܂B';
    ERR_UNKNOWN_CVR_PARAMETERS   : Result := 'CVRp[^łB';
    ERR_BLOCKING_WRITE           : Result := '݂֎~Ă܂B';
    ERR_SESSION_OVERLOAD         : Result := 'ERR_SESSION_OVERLOAD';
    ERR_USER_TOO_ACTIVE          : Result := '[UI܂B';
    ERR_TOO_MANY_SESSIONS        : Result := 'ȏZbVmł܂B';
    ERR_NOT_EXPECTED             : Result := '\ʃG[łB';
    ERR_BAD_FRIEND_FILE          : Result := 'ERR_BAD_FRIEND_FILE';
    ERR_AUTHENTICATION_FAILED    : Result := 'TCC܂̓pX[hԈĂ܂B';
    ERR_NOT_ALLOWED_WHEN_OFFLINE : Result := 'ItCɂ͎słȂłB';
    ERR_NOT_ACCEPTING_NEW_USERS  : Result := 'ERR_NOT_ACCEPTING_NEW_USERS';
    ERR_PASSPORT_NOT_VERIFIED    : Result := 'ERR_PASSPORT_NOT_VERIFIED';
  else
    Result := 'sȃG[܂B';
  end;
end;

// T[o瑗ĂR}hɕ
function SplitCommandStr(List: TStringList; const Str: UTF8String): UTF8String;
var
  P: Integer;
  Temp: UTF8String;
  Delimiter: UTF8String;
  ParamLst: TStringList;
begin
  ParamLst := TStringList.Create;
  List.Clear;
  Delimiter := #13#10;
  Result := '';

  Temp := Str;
  P := Pos(Delimiter, Temp);
  while P <> 0 do
  begin
    Split(ParamLst, Copy(Temp, 1, P - 1), ' ');
    if (SS(ParamLst, 0) = 'MSG') and (StrToIntDef(SS(ParamLst, 3), 0) <> 0) and
       (Length(Temp) < (P + Length(Delimiter) + StrToIntDef(SS(ParamLst, 3), 0)) - 1)then
    begin
      Break;
    end else
    if (SS(ParamLst, 0) = 'MSG') and (StrToIntDef(SS(ParamLst, 3), 0) <> 0) then
    begin
      P := P + Length(Delimiter) + StrToIntDef(SS(ParamLst, 3), 0);
      List.Add(Copy(Temp, 1, P - 1));
      Delete(Temp, 1, P - 1);
    end else
    begin
      List.Add(Copy(Temp, 1, P - 1));
      Delete(Temp, 1, P + Length(Delimiter) - 1);
    end;
    P := Pos(Delimiter, Temp);
  end;
  Result := Temp;

  ParamLst.Free;
end;

// R}hp[^ɕ
procedure SplitParamStr(List: TStringList; const Str: UTF8String);
var
  P: Integer;
  Temp: UTF8String;
begin
  List.Clear;

  // ŏ̉sȍ~͖
  Temp := Str;
  P := Pos(#13#10, Temp);
  if P > 0 then
    Temp := Copy(Temp, 1, P);

  Split(List, Temp, ' ');
end;

// CfbNXᔽ΍i
function SS(List: TStringList; Idx: Integer): String;
begin
  Result := '';
  if (Idx >= 0) and (Idx < List.Count) then
    Result := List[Idx];
end;

// ϊsꍇ͕̂܂ܕԂ Utf8ToAnsi
function Utf8ToAnsiEx(const S: Utf8String): String;
var
  Temp: String;
begin
  Temp := Utf8ToAnsi(S);
  if Temp = '' then
    Result := S
  else
    Result := Temp;
end;

// \ȂǂGR[h
function EncodeParam(const S: WideString): UTF8String;
begin
  Result := EncodeSpace(UTF8Encode(S));
end;

// \ȂǂfR[h
function DecodeParam(const S: UTF8String): WideString;
begin
  Result := UTF8Decode(UrlDecode(S));
end;

end.
