unit UNotify;

interface

uses
  Windows, MMSystem, SysUtils, UConfig, USkinResource, IniFiles, UNsmUtils,
  UNsmTypes;

type
  TEventNotifyType = (
    ntExclamation,      // x              NMNI_EXCLAMATION
    ntStop,             // G[            NMNI_STOP
    ntInformation,      //               NMNI_INFORMATION
    ntQuestion,         // ₢킹        NMNI_QUESTION

    ntReceiveMessage,   // bZ[WM    NMNI_MESSAGE
    ntSendMessage,
    ntReceiveMail,      // [M        NMNI_MAIL
    ntReceiveFile,      // t@CM      NMNI_FILE
    ntFinishFileSend,
    ntFinishFileReceive,
    ntConnect,          // ڑ              NMNI_CONNECT
    ntDisconnect,       // ؒf              NMNI_DISCONNECT
    ntMemberOnline,     // oIC  NMNI_ONLINE
    ntMemberOffline     // oItC  NMNI_OFFLINE
  );

  TEventNotifyItem = class(TObject)
  private
    FNotifyType: TEventNotifyType;
    FSoundFileName: String;
    FSoundEnabled: Boolean;
    FNotifyEnabled: Boolean;
  public
    constructor Create(ANotifyType: TEventNotifyType);
    procedure PlaySound(IgnoreEnabled: Boolean = False);
    procedure ShowNotify(Text: String; TTL: Integer = -1; IgnoreEnabled: Boolean = False; LinkTo: String = ''; OnClick: TNotifyClickCallback = nil; Data: Integer = 0);
    procedure ShowDefaultMsg(const Args: array of const; TTL: Integer = -1; IgnoreEnabled: Boolean = False);
    procedure Assign(Source: TEventNotifyItem);

    property NotifyType: TEventNotifyType read  FNotifyType;
    property SoundFileName: String read FSoundFileName write FSoundFileName;
    property SoundEnabled: Boolean read FSoundEnabled write FSoundEnabled;
    property NotifyEnabled: Boolean read FNotifyEnabled write FNotifyEnabled;
  end;

  TEventNotifyList = class(TObject)
  private
    FItems: array[ntExclamation..ntMemberOffline] of TEventNotifyItem;
    function GetItem(NotifyType: TEventNotifyType): TEventNotifyItem;
    function GetCount: Integer;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Init;
    procedure Assign(Source: TEventNotifyList);
    procedure SaveToFile(const FileName: String);
    procedure LoadFromFile(const FileName: String);
    property Notify[NotifyType: TEventNotifyType]: TEventNotifyItem read GetItem; default;
    property Count: Integer read GetCount;
  end;

implementation

uses UFrmMain, UFrmNotify;

const
  INISECTION_EVENTNOTIFY = 'EventNotify';

constructor TEventNotifyItem.Create(ANotifyType: TEventNotifyType);
begin
  FNotifyType := ANotifyType;
end;

procedure TEventNotifyItem.Assign(Source: TEventNotifyItem);
begin
  FSoundFileName := Source.SoundFileName;
  FSoundEnabled := Source.SoundEnabled;
  FNotifyEnabled := Source.NotifyEnabled;
end;

procedure TEventNotifyItem.PlaySound(IgnoreEnabled: Boolean = False);
begin
  if FSoundEnabled or IgnoreEnabled then
  begin
    if FileExists(FSoundFileName) then
      MMSystem.PlaySound(PChar(FSoundFileName), 0, SND_ASYNC or SND_FILENAME)
    else
      case FNotifyType of
        ntExclamation : MessageBeep(MB_ICONEXCLAMATION);
        ntStop        : MessageBeep(MB_ICONHAND);
        ntInformation : MessageBeep(MB_ICONASTERISK);
        ntQuestion    : MessageBeep(MB_ICONQUESTION);
      else
        MessageBeep(MB_OK);
      end;
  end;
end;

procedure TEventNotifyItem.ShowNotify(Text: String; TTL: Integer = -1;
  IgnoreEnabled: Boolean = False; LinkTo: String = ''; OnClick:
  TNotifyClickCallback = nil; Data: Integer = 0);
var
  Item: TNotifyItem;
  Idx: Integer;
begin
  if not(FNotifyEnabled or IgnoreEnabled) then Exit;

  if not Assigned(FrmNotify) then
    FrmNotify := TFrmNotify.Create(FrmMain);

  if TTL = -1 then
    TTL := Config.NotifyTime;

  Item := FrmNotify.NotifyList.Add(Text, TTL, nil);
  case FNotifyType of
    ntExclamation       : Idx := 0;
    ntStop              : Idx := 1;
    ntInformation       : Idx := 2;
    ntQuestion          : Idx := 3;

    ntReceiveMessage    : Idx := 4;
    ntSendMessage       : Idx := 4;
    ntReceiveMail       : Idx := 5;
    ntReceiveFile       : Idx := 6;
    ntFinishFileSend    : Idx := 6;
    ntFinishFileReceive : Idx := 6;
    ntConnect           : Idx := 7;
    ntDisconnect        : Idx := 8;

    ntMemberOnline      : Idx := 0;
    ntMemberOffline     : Idx := 3;
  else
    Idx := 0;
  end;
  case FNotifyType of
  ntMemberOnline .. ntMemberOffline:
    SkinResources.StateImages.GetBitmap(Idx, Item.Bitmap);
  else
    SkinResources.CommonImages.GetBitmap(Idx, Item.Bitmap);
  end;
  Item.LinkTo := LinkTo;
  Item.OnClick := OnClick;
  Item.Data := Data;
end;

procedure TEventNotifyItem.ShowDefaultMsg(const Args: array of const;
  TTL: Integer = -1; IgnoreEnabled: Boolean = False);
var
  Msg: String;
begin
  Msg := '';
  case FNotifyType of
    ntExclamation       : Msg := 'x'#13#10'%s';
    ntStop              : Msg := 'G['#13#10'%s';
    ntInformation       : Msg := ''#13#10'%s';
    ntQuestion          : Msg := '₢킹'#13#10'%s';

    ntReceiveMessage    : Msg := '%s ̔ :'#13#10'%s';
    ntSendMessage       : Msg := '%s ̔ :'#13#10'%s';
    ntReceiveMail       : Msg := '%s ̐V[͂܂';
    ntReceiveFile       : Msg := '%s  "%s" 𑗐M悤ƂĂ܂';
    ntFinishFileSend    : Msg := '"%s" ̓]܂';
    ntFinishFileReceive : Msg := '"%s" ̓]܂';
    ntConnect           : Msg := '%s ɐڑ܂';
    ntDisconnect        : Msg := '%s ؒf܂';

    ntMemberOnline      : Msg := '%s ICɂȂ܂';
    ntMemberOffline     : Msg := '%s ItCɂȂ܂';
  end;
  Msg := Format(Msg, Args);
  ShowNotify(Msg, TTL, IgnoreEnabled);
  PlaySound;
end;
// -----------------------------------------------------------------------------

constructor TEventNotifyList.Create;
var
  I: Integer;
begin
  for I := 0 to Length(FItems) - 1 do
    FItems[TEventNotifyType(I)] := TEventNotifyItem.Create(TEventNotifyType(I));
  Init;
end;

destructor TEventNotifyList.Destroy;
var
  I: Integer;
begin
  for I := 0 to Length(FItems) - 1 do
    FItems[TEventNotifyType(I)].Free;
  inherited;
end;

procedure TEventNotifyList.Init;
begin
  Fitems[ntExclamation].SoundEnabled := True;
  Fitems[ntStop].SoundEnabled := True;
  Fitems[ntInformation].SoundEnabled := True;
  Fitems[ntQuestion].SoundEnabled := True;
  Fitems[ntReceiveMessage].SoundEnabled := True;
  Fitems[ntSendMessage].SoundEnabled := False;
  Fitems[ntReceiveMail].SoundEnabled := True;
  Fitems[ntReceiveFile].SoundEnabled := True;
  Fitems[ntFinishFileSend].SoundEnabled := True;
  Fitems[ntFinishFileReceive].SoundEnabled := True;
  Fitems[ntConnect].SoundEnabled := False;
  Fitems[ntDisconnect].SoundEnabled := False;
  Fitems[ntMemberOnline].SoundEnabled := True;
  Fitems[ntMemberOffline].SoundEnabled := False;
  
  Fitems[ntExclamation].NotifyEnabled := True;
  Fitems[ntStop].NotifyEnabled := True;
  Fitems[ntInformation].NotifyEnabled := True;
  Fitems[ntQuestion].NotifyEnabled := True;
  Fitems[ntReceiveMessage].NotifyEnabled := True;
  Fitems[ntSendMessage].NotifyEnabled := False;
  Fitems[ntReceiveMail].NotifyEnabled := True;
  Fitems[ntReceiveFile].NotifyEnabled := True;
  Fitems[ntFinishFileSend].NotifyEnabled := True;
  Fitems[ntFinishFileReceive].NotifyEnabled := True;
  Fitems[ntConnect].NotifyEnabled := False;
  Fitems[ntDisconnect].NotifyEnabled := False;
  Fitems[ntMemberOnline].NotifyEnabled := True;
  Fitems[ntMemberOffline].NotifyEnabled := False;
end;

procedure TEventNotifyList.Assign(Source: TEventNotifyList);
var
  I: Integer;
begin
  for I := 0 to Length(FItems) - 1 do
    FItems[TEventNotifyType(I)].Assign(Source.Notify[TEventNotifyType(I)]);
end;

function TEventNotifyList.GetItem(NotifyType: TEventNotifyType): TEventNotifyItem;
begin
  Result := FItems[NotifyType];
end;

function TEventNotifyList.GetCount: Integer;
begin
  Result := Length(FItems);
end;

procedure TEventNotifyList.SaveToFile(const FileName: String);
var
  Ini: TIniFile;
  I: TEventNotifyType;
  Value: String;
begin
  Ini := TIniFile.Create(FileName);
  try
    Ini.EraseSection(INISECTION_EVENTNOTIFY);
    for I := TEventNotifyType(0) to TEventNotifyType(Length(FItems) - 1) do
    begin
      Value := Format('%d %d %s', [
                Ord(FItems[I].SoundEnabled),
                Ord(FItems[I].NotifyEnabled),
                EncodeSpace(FItems[I].SoundFileName)]);
      Ini.WriteString(INISECTION_EVENTNOTIFY, IntToStr(Ord(I)), Value);
    end;
  finally
    Ini.Free;
  end;
end;

procedure TEventNotifyList.LoadFromFile(const FileName: String);
var
  Ini: TIniFile;
  I: TEventNotifyType;
  Value: String;
const
  SprtChar = ' ';
begin
  Ini := TIniFile.Create(FileName);
  try
    for I := TEventNotifyType(0) to TEventNotifyType(Length(FItems) - 1) do
    begin
      Value := Ini.ReadString(INISECTION_EVENTNOTIFY, IntToStr(Ord(I)), '');
      with FItems[I] do
      begin
        SoundEnabled :=
            Boolean(StrToIntDef(GetNextParam(Value, SprtChar),
            Ord(SoundEnabled)));
        NotifyEnabled :=
            Boolean(StrToIntDef(GetNextParam(Value, SprtChar),
            Ord(NotifyEnabled)));
        SoundFileName := DecodeSpace(GetNextParam(Value, SprtChar));
      end;
    end;
  finally
    Ini.Free;
  end;
end;

end.
 