unit USkinStyles;

interface

uses
  SysUtils, Types, Classes, Graphics, IniFiles, StrUtils, Controls;

type
  TSkinTextAlign = (taNone, taLeft, taCenter, taRight);
  TSkinBorderStyle = (sbNone, sbSolid, sbDashed, sbDotted, sbDouble, sbInset, sbOutset, sbGroove, sbRidge, sbImage, sbSystem);
  TSkinBackgroundMode = (bmRepeat, bmRepeatX, bmRepeatY, bmNoRepeat, bmStretch);

  TSkinStyles = class;
  TSkinIniFile = class;

  TSkinStyle = class(TObject)
  private
    FOwner: TSkinStyles;
    // 摜
    FTopImage: TBitmap;
    FBottomImage: TBitmap;
    FLeftImage: TBitmap;
    FRightImage: TBitmap;
    FTopLeftImage: TBitmap;
    FTopRightImage: TBitmap;
    FBottomLeftImage: TBitmap;
    FBottomRightImage: TBitmap;
    FBackgroundImage: TBitmap;
  public
    // tHg
    FontColor: TColor;
    FontFamily: String;
    FontStyle: TFontStyles;
    FontSize: Integer;
    // eLXg
    TextAlign: TSkinTextAlign;
    PaddingLeft: Integer;
    PaddingRight: Integer;
    PaddingTop: Integer;
    PaddingBottom: Integer;
    // {bNX
    BorderStyle: TSkinBorderStyle;
    BorderWidth: Integer;
    BorderColor: TColor;
    // wi
    BackgroundColor: TColor;
    TransparentColor: TColor;
    BackgroundMode: TSkinBackgroundMode;
    // ̑
    Left: Integer;
    Top: Integer;
    Right: Integer;
    Bottom: Integer;
    Width: Integer;
    Height: Integer;
    Visibility: Boolean;

    property TopImage: TBitmap read FTopImage;
    property BottomImage: TBitmap read FBottomImage;
    property LeftImage: TBitmap read FLeftImage;
    property RightImage: TBitmap read FRightImage;
    property TopLeftImage: TBitmap read FTopLeftImage;
    property TopRightImage: TBitmap read FTopRightImage;
    property BottomLeftImage: TBitmap read FBottomLeftImage;
    property BottomRightImage: TBitmap read FBottomRightImage;
    property BackgroundImage: TBitmap read FBackgroundImage;

    constructor Create(AOwner: TSkinStyles);
    destructor Destroy; override;
    procedure Assign(Source: TSkinStyle);
    procedure LoadFromFile(Ini: TSkinIniFile; const Section: String);
    procedure Draw(ACanvas: TCanvas; R: TRect; Transparent: Boolean = False);
    procedure DrawText(ACanvas: TCanvas; R: TRect; Text: WideString);
    procedure FillBackGround(ACanvas: TCanvas; R: TRect);
    procedure DrawBorder(ACanvas: TCanvas; R: TRect; Transparent: Boolean = False);
    procedure AdjustClientRect(var R: TRect);
    function GetBackgroundRect(const R: TRect): TRect;
    procedure AssignToFont(AFont: TFont);
    procedure SetControlSize(AControl: TControl; DefWidth, DefHeight: Integer);
    procedure SetControlBounds(AControl: TControl; DefLeft, DefTop, DefWidth, DefHeight: Integer);
  end;

  TSkinIniFile = class(TIniFile)
  public
    function ReadColor(const Section, Ident: String; Default: TColor): TColor;
    function ReadFontStyles(const Section, Ident: String; Default: TFontStyles): TFontStyles;
    function ReadTextAlign(const Section, Ident: String; Default: TSkinTextAlign): TSkinTextAlign;
    function ReadBorderStyle(const Section, Ident: String; Default: TSkinBorderStyle): TSkinBorderStyle;
    function ReadBackgroundMode(const Section, Ident: String; Default: TSkinBackgroundMode): TSkinBackgroundMode;
  end;

  TSkinStyles = class(TObject)
  private
    FItems: THashedStringList;
    FFileName: String;
    FIniFile: TSkinIniFile;
    FDefaultStyle: TSkinStyle;
    function GetItem(Idx: Integer): TSkinStyle;
    function GetCount: Integer;
    function FindStyle(Element, ClassName, Status: String): Integer;
    function FindChildStyle(ParentElement, ParentClassName, ParentStatus,
        Element, ClassName, Status: String): Integer;
    function StyleNeeded(Idx: Integer): TSkinStyle;
    function IniFileNeeded: TSkinIniFile;
    function IniFileAllocated: Boolean;
    procedure ReleaseIniFile;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(StyleName: String): TSkinStyle;
    procedure Delete(Idx: Integer);
    procedure Clear;
    function IndexOfStyle(StyleName: String): Integer;

    function GetStyle(Element, ClassName, Status: String): TSkinStyle;
    function GetChildStyle(ParentElement, ParentClassName, ParentStatus,
        Element, ClassName, Status: String): TSkinStyle;
    procedure AppendFromFile(const FileName: String);

    property Count: Integer read GetCount;
    property Item[Idx: Integer]: TSkinStyle read GetItem; default;
    property FileName: String read FFileName;
    property IniFile: TSkinIniFile read IniFileNeeded;
    property DefaultStyle: TSkinStyle read FDefaultStyle;
  end;

implementation

uses
  USkinUtils, UBmpMngr;

constructor TSkinStyle.Create(AOwner: TSkinStyles);
begin
  FOwner := AOwner;

  BitmapManager.LoadBitmap(FTopImage, '');
  BitmapManager.LoadBitmap(FBottomImage, '');
  BitmapManager.LoadBitmap(FLeftImage, '');
  BitmapManager.LoadBitmap(FRightImage, '');
  BitmapManager.LoadBitmap(FTopleftImage, '');
  BitmapManager.LoadBitmap(FToprightImage, '');
  BitmapManager.LoadBitmap(FBottomleftImage, '');
  BitmapManager.LoadBitmap(FBottomrightImage, '');
  BitmapManager.LoadBitmap(FBackgroundImage, '');

  FontColor         := clNone;
  FontFamily        := '';
  FontStyle         := [];
  FontSize          := -1;
  TextAlign         := taNone;
  PaddingLeft       := 0;
  PaddingRight      := 0;
  PaddingTop        := 0;
  PaddingBottom     := 0;
  BorderStyle       := sbNone;
  BorderWidth       := 1;
  BorderColor       := clNone;
  BackgroundColor   := clNone;
  TransparentColor  := clNone;
  BackgroundMode    := bmRepeat;
  Left              := -1;
  Top               := -1;
  Right             := -1;
  Bottom            := -1;
  Width             := -1;
  Height            := -1;
  Visibility        := True;
end;

destructor TSkinStyle.Destroy;
begin
  BitmapManager.ReleaseBitmap(FTopImage);
  BitmapManager.ReleaseBitmap(FBottomImage);
  BitmapManager.ReleaseBitmap(FLeftImage);
  BitmapManager.ReleaseBitmap(FRightImage);
  BitmapManager.ReleaseBitmap(FTopleftImage);
  BitmapManager.ReleaseBitmap(FToprightImage);
  BitmapManager.ReleaseBitmap(FBottomleftImage);
  BitmapManager.ReleaseBitmap(FBottomrightImage);
  BitmapManager.ReleaseBitmap(FBackgroundImage);
  inherited;
end;

procedure TSkinStyle.Assign(Source: TSkinStyle);
begin
  BitmapManager.CopyBitmap(Source.TopImage, FTopImage);
  BitmapManager.CopyBitmap(Source.BottomImage, FBottomImage);
  BitmapManager.CopyBitmap(Source.LeftImage, FLeftImage);
  BitmapManager.CopyBitmap(Source.RightImage, FRightImage);
  BitmapManager.CopyBitmap(Source.TopLeftImage, FTopLeftImage);
  BitmapManager.CopyBitmap(Source.TopRightImage, FTopRightImage);
  BitmapManager.CopyBitmap(Source.BottomLeftImage, FBottomLeftImage);
  BitmapManager.CopyBitmap(Source.BottomRightImage, FBottomRightImage);
  BitmapManager.CopyBitmap(Source.BackgroundImage, FBackgroundImage);

  FontColor         := Source.FontColor;
  FontFamily        := Source.FontFamily;
  FontStyle         := Source.FontStyle;
  FontSize          := Source.FontSize;
  TextAlign         := Source.TextAlign;
  PaddingLeft       := Source.PaddingLeft;
  PaddingRight      := Source.PaddingRight;
  PaddingTop        := Source.PaddingTop;
  PaddingBottom     := Source.PaddingBottom;
  BorderStyle       := Source.BorderStyle;
  BorderWidth       := Source.BorderWidth;
  BorderColor       := Source.BorderColor;
  BackgroundColor   := Source.BackgroundColor;
  TransparentColor  := Source.TransparentColor;
  BackgroundMode    := Source.BackgroundMode;
  Left              := Source.Left;
  Top               := Source.Top;
  Right             := Source.Right;
  Bottom            := Source.Bottom;
  Width             := Source.Width;
  Height            := Source.Height;
  Visibility        := Source.Visibility;
end;

procedure TSkinStyle.LoadFromFile(Ini: TSkinIniFile; const Section: String);
var
  BasePath: String;
  Idx: Integer;
  Source: TSkinStyle;
  Clone: String;

  procedure LoadBitmap(var Bitmap: TBitmap; FileName: String);
  begin
    if FileName <> '' then
      BitmapManager.LoadBitmap(Bitmap, BasePath + FileName);
  end;
begin
  BasePath := ExtractFilePath(Ini.FileName);
  // N[
  Clone := Ini.ReadString(Section, 'Clone', '');
  if (Clone <> '') and Assigned(FOwner) then
  begin
    Idx := FOwner.IndexOfStyle(Clone);
    if Idx > -1 then
    begin
      Source := FOwner.StyleNeeded(Idx);
      if Source <> Self then
      begin
        Source.LoadFromFile(Ini, Clone);
        Assign(Source);
      end;
    end;
  end;
  // tHg
  FontColor     := Ini.ReadColor(Section, 'Font-Color', FontColor);
  FontFamily    := Ini.ReadString(Section, 'Font-Family', FontFamily);
  FontStyle     := Ini.ReadFontStyles(Section, 'Font-Style', FontStyle);
  FontSize      := Ini.ReadInteger(Section, 'Font-Size', FontSize);
  // eLXg
  TextAlign     := Ini.ReadTextAlign(Section, 'Text-Align', TextAlign);
  PaddingLeft   := Ini.ReadInteger(Section, 'Padding-Left', PaddingLeft);
  PaddingRight  := Ini.ReadInteger(Section, 'Padding-Right', PaddingRight);
  PaddingTop    := Ini.ReadInteger(Section, 'Padding-Top', PaddingTop);
  PaddingBottom := Ini.ReadInteger(Section, 'Padding-Bottom', PaddingBottom);
  // {bNX
  BorderStyle   := Ini.ReadBorderStyle(Section, 'Border-Style', BorderStyle);
  BorderWidth   := Ini.ReadInteger(Section, 'Border-Width', BorderWidth);
  BorderColor   := Ini.ReadColor(Section, 'Border-Color', BorderColor);
  // g{[_[X^C
  LoadBitmap(FTopImage, Ini.ReadString(Section, 'BorderTop-Image', ''));
  LoadBitmap(FBottomImage, Ini.ReadString(Section, 'BorderBottom-Image', ''));
  LoadBitmap(FLeftImage, Ini.ReadString(Section, 'BorderLeft-Image', ''));
  LoadBitmap(FRightImage, Ini.ReadString(Section, 'BorderRight-Image', ''));
  LoadBitmap(FTopLeftImage, Ini.ReadString(Section, 'BorderTopLeft-Image', ''));
  LoadBitmap(FTopRightImage, Ini.ReadString(Section, 'BorderTopRight-Image', ''));
  LoadBitmap(FBottomLeftImage, Ini.ReadString(Section, 'BorderBottomLeft-Image', ''));
  LoadBitmap(FBottomRightImage, Ini.ReadString(Section, 'BorderBottomRight-Image', ''));
  // wi
  BackgroundColor   := Ini.ReadColor(Section, 'Background-Color', BackgroundColor);
  TransparentColor  := Ini.ReadColor(Section, 'Transparent-Color', TransparentColor);
  LoadBitmap(FBackgroundImage, Ini.ReadString(Section, 'Background-Image', ''));
  BackgroundMode    := Ini.ReadBackgroundMode(Section, 'Background-Mode', BackgroundMode);
  // ̑
  Left          := Ini.ReadInteger(Section, 'Left', Left);
  Top           := Ini.ReadInteger(Section, 'Top', Top);
  Right         := Ini.ReadInteger(Section, 'Right', Right);
  Bottom        := Ini.ReadInteger(Section, 'Bottom', Bottom);
  Height        := Ini.ReadInteger(Section, 'Height', Height);
  Width         := Ini.ReadInteger(Section, 'Width', Width);
  Visibility    := Ini.ReadBool(Section, 'Visibility', Visibility);
end;

procedure TSkinStyle.Draw(ACanvas: TCanvas; R: TRect;
  Transparent: Boolean = False);
begin
  FillStyleBackground(ACanvas, GetBackgroundRect(R), Self);
  DrawStyleBorder(ACanvas, R, Self, Transparent);
end;

procedure TSkinStyle.DrawText(ACanvas: TCanvas; R: TRect; Text: WideString);
begin
  DrawStyleText(ACanvas, R, Self, Text);
end;

procedure TSkinStyle.FillBackGround(ACanvas: TCanvas; R: TRect);
begin
  FillStyleBackground(ACanvas, R, Self);
end;

procedure TSkinStyle.DrawBorder(ACanvas: TCanvas; R: TRect; Transparent: Boolean = False);
begin
  DrawStyleBorder(ACanvas, R, Self, Transparent);
end;

function TSkinStyle.GetBackgroundRect(const R: TRect): TRect;
begin
  if not (BorderStyle in [sbDashed, sbDotted, sbDouble]) then
    Result := GetStyleClientRect(R, Self)
  else
    Result := R;
end;

procedure TSkinStyle.AdjustClientRect(var R: TRect);
begin
  R := GetStyleClientRect(R, Self);
  Inc(R.Left, PaddingLeft);
  Inc(R.Top, PaddingTop);
  Dec(R.Right, PaddingRight);
  Dec(R.Bottom, PaddingBottom);
end;

procedure TSkinStyle.AssignToFont(AFont: TFont);
begin
  AssignStyleToFont(AFont, Self);
end;

procedure TSkinStyle.SetControlSize(AControl: TControl; DefWidth, DefHeight: Integer);
begin
  if Width > 0 then
    AControl.Width := Width
  else if DefWidth > 0 then
    AControl.Width := DefWidth;
  if Height > 0 then
    AControl.Height := Height
  else if DefWidth > 0 then
    AControl.Height := DefHeight;    
end;

procedure TSkinStyle.SetControlBounds(AControl: TControl; DefLeft, DefTop, DefWidth, DefHeight: Integer);
var
  L, T, W, H: Integer;
begin
  if Width >= 0 then
    W := Width
  else
    W := DefWidth;
  if Height >= 0 then
    H := Height
  else
    H := DefHeight;

  if Left >= 0 then
    L := Left
  else if (Right >= 0) and Assigned(AControl.Parent) then
    L := AControl.Parent.ClientWidth - Right - W
  else
    L := DefLeft;
  if Top >= 0 then
    T := Top
  else if (Bottom >= 0) and Assigned(AControl.Parent) then
    T := AControl.Parent.ClientHeight - Bottom - H
  else
    T := DefTop;
  if (Right >= 0) then
    AControl.Anchors := [akTop, akRight]
  else
    AControl.Anchors := [akTop, akLeft];
  AControl.SetBounds(L, T, W, H);
end;

// -----------------------------------------------------------------------------
constructor TSkinStyles.Create;
begin
  FItems := THashedStringList.Create;
  FItems.Sorted := True;
  FItems.CaseSensitive := False;
  FDefaultStyle := TSkinStyle.Create(Self);
end;

destructor TSkinStyles.Destroy;
begin
  Clear;
  FItems.Free;
  FDefaultStyle.Free;
  inherited;
end;

function TSkinStyles.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TSkinStyles.GetItem(Idx: Integer): TSkinStyle;
begin
  if Assigned(FItems.Objects[Idx]) then
    Result := TSkinStyle(FItems.Objects[Idx])
  else
    Result := nil;
end;

function TSkinStyles.Add(StyleName: String): TSkinStyle;
var
  Idx: Integer;
begin
  Idx := FItems.IndexOf(StyleName);
  if Idx > -1 then
  begin
    if FItems.Objects[Idx] = nil then
    begin
      Result := TSkinStyle.Create(Self);
      Result.Assign(FDefaultStyle);
      FItems.Objects[Idx] := Result;
    end else
      Result := TSkinStyle(FItems.Objects[Idx]);
  end else
  begin
    Result := TSkinStyle.Create(Self);
    FItems.AddObject(StyleName, Result);
  end;
end;

procedure TSkinStyles.Delete(Idx: Integer);
begin
  if Assigned(FItems.Objects[Idx]) then
    TSkinStyle(FItems.Objects[Idx]).Free;
  FItems.Delete(Idx);
end;

procedure TSkinStyles.Clear;
begin
  while FItems.Count > 0 do
    Delete(0);
  ReleaseIniFile;
end;

function TSkinStyles.IndexOfStyle(StyleName: String): Integer;
begin
  Result := FItems.IndexOf(StyleName);
end;

function TSkinStyles.FindStyle(Element, ClassName, Status: String): Integer;
var
  Level: Integer;
  StyleName: String;
begin
  Level := 0;
  repeat
    StyleName := TraceBackStyleName(Element, ClassName, Status, Level);
    Result := IndexOfStyle(StyleName);
    Inc(Level);
  until (Result > -1) or (Level > MAXSTYLELEVEL);
end;

function TSkinStyles.FindChildStyle(ParentElement, ParentClassName,
  ParentStatus, Element, ClassName, Status: String): Integer;
var
  Level, ParentLevel: Integer;
  StyleName: String;
begin
  ParentLevel := 0;
  repeat
    Level := 0;
    repeat
      StyleName := TraceBackStyleName(Element, ClassName, Status, Level);
      if ParentLevel <= MAXSTYLELEVEL then
        StyleName := TraceBackStyleName(ParentElement, ParentClassName,
            ParentStatus, ParentLevel) + ' ' + StyleName;
      Result := IndexOfStyle(StyleName);
      Inc(Level);
    until (Result > -1) or (Level > MAXSTYLELEVEL);
    Inc(ParentLevel);
  until (Result > -1) or (ParentLevel > MAXSTYLELEVEL + 1);
end;

function TSkinStyles.StyleNeeded(Idx: Integer): TSkinStyle;
begin
  if not Assigned(GetItem(Idx)) then
  begin
    Result := Add(FItems[Idx]);
    IniFileNeeded;
    if Assigned(FIniFile) then
      Result.LoadFromFile(FIniFile, FItems[Idx]);
  end else
    Result := GetItem(Idx);
end;

function TSkinStyles.GetStyle(Element, ClassName, Status: String): TSkinStyle;
var
  Idx: Integer;
begin
  Idx := FindStyle(Element, ClassName, Status);
  if (Idx > -1) then
  begin
    Result := StyleNeeded(Idx);
  end else
    Result := FDefaultStyle;
end;

function TSkinStyles.GetChildStyle(ParentElement, ParentClassName,
  ParentStatus, Element, ClassName, Status: String): TSkinStyle;
var
  Idx: Integer;
begin
  Idx := FindChildStyle(ParentElement, ParentClassName, ParentStatus,
        Element, ClassName, Status);
  if (Idx > -1) then
  begin
    Result := StyleNeeded(Idx);
  end else
    Result := FDefaultStyle;
end;

function TSkinStyles.IniFileNeeded: TSkinIniFile;
begin
  if Assigned(FIniFile) then
    Result := FIniFile
  else if FileExists(FFileName) then
  begin
    Result := TSkinIniFile.Create(FileName);
    FIniFile := Result;
  end else
    Result := nil;
end;

procedure TSkinStyles.ReleaseIniFile;
begin
  if Assigned(FIniFile) then
    FreeAndNil(FIniFile);
end;

function TSkinStyles.IniFileAllocated: Boolean;
begin
  Result := Assigned(FIniFile);
end;

procedure TSkinStyles.AppendFromFile(const FileName: String);
var
  SL: TStringList;
begin
  FFileName := FileName;
  IniFileNeeded;
  if not IniFileAllocated then Exit;

  SL := TStringList.Create;
  try
    IniFile.ReadSections(SL);
    FItems.AddStrings(SL);
  finally
    SL.Free;
  end;
end;

// -----------------------------------------------------------------------------

function RGBSwap(const Source: Cardinal): Cardinal;
begin
  Result := ((Source and $FF0000) shr 16)
          +  (Source and $FF00)
          + ((Source and $FF) shl 16);
end;

function TSkinIniFile.ReadColor(const Section, Ident: String; Default:
  TColor): TColor;
var
  Str: String;
begin
  Str := ReadString(Section, Ident, '');
  if Str = '' then
    Result := Default
  else if Str[1] = '#' then
  begin
    Str[1] := '$';
    Result := RGBSwap(StrToIntDef(Str, RGBSwap(Default)));
  end else if not IdentToColor(Str, Integer(Result)) then
    Result := TColor(StrToIntDef(Str, Default));
end;

function TSkinIniFile.ReadFontStyles(const Section, Ident: String; Default:
  TFontStyles): TFontStyles;
var
  Str: String;
begin
  Str := ReadString(Section, Ident, '');
  if Str = '' then
    Result := Default
  else
  begin
    Result := [];
    if AnsiContainsText(Str, 'Italic') then
      Result := Result + [fsItalic];
    if AnsiContainsText(Str, 'Bold') then
      Result := Result + [fsBold];
    if AnsiContainsText(Str, 'Underline') then
      Result := Result + [fsUnderline];
    if AnsiContainsText(Str, 'StrikeOut') then
      Result := Result + [fsStrikeOut	];
  end;
end;

function TSkinIniFile.ReadTextAlign(const Section, Ident: String; Default:
  TSkinTextAlign): TSkinTextAlign;
var
  Str: String;
begin
  Str := ReadString(Section, Ident, '');
  if SameText(Str, 'Left') then
    Result := taLeft
  else if SameText(Str, 'Center') then
    Result := taCenter
  else if SameText(Str, 'Right') then
    Result := taRight
  else
    Result := Default;
end;

function TSkinIniFile.ReadBorderStyle(const Section, Ident: String; Default:
  TSkinBorderStyle): TSkinBorderStyle;
var
  Str: String;
begin
  Str := ReadString(Section, Ident, '');
  if SameText(Str, 'none') then
    Result := sbNone
  else if SameText(Str, 'Solid') then
    Result := sbSolid
  else if SameText(Str, 'Dashed') then
    Result := sbDashed
  else if SameText(Str, 'Dotted') then
    Result := sbDotted
  else if SameText(Str, 'Double') then
    Result := sbDouble
  else if SameText(Str, 'Outset') then
    Result := sbOutset
  else if SameText(Str, 'Inset') then
    Result := sbInset
  else if SameText(Str, 'Groove') then
    Result := sbGroove
  else if SameText(Str, 'Ridge') then
    Result := sbRidge
  else if SameText(Str, 'Image') then
    Result := sbImage
  else if SameText(Str, 'System') then
    Result := sbSystem
  else
    Result := Default;
end;

function TSkinIniFile.ReadBackgroundMode(const Section, Ident: String;
  Default: TSkinBackgroundMode): TSkinBackgroundMode;
var
  Str: String;
begin
  Str := ReadString(Section, Ident, '');
  if SameText(Str, 'Repeat') then
    Result := bmRepeat
  else if SameText(Str, 'Repeat-X') then
    Result := bmRepeatX
  else if SameText(Str, 'Repeat-Y') then
    Result := bmRepeatY
  else if SameText(Str, 'No-Repeat') then
    Result := bmNoRepeat
  else if SameText(Str, 'Stretch') then
    Result := bmStretch
  else
    Result := Default;
end;

end.
