unit SimpleTreeView;

interface

uses
  Windows, Messages, Forms, SysUtils, Classes, Controls, Graphics, ExPanel,
  Imglist, UWideGraphics, USkinStyles;

type
  TSimpleTreeNodes = class;
  TSimpleTreeView = class;
  TSimpleTreeNode = class(TObject)
  private
    FOwner: TSimpleTreeNodes;
    FParent: TSimpleTreeNode;
    FItems: TList;
    FText: WideString;
    FBold: Boolean;
    FExpanded: Boolean;
    FImageIndex: Integer;
    FExpandedIndex: Integer;
    FOverlayIndex: Integer;
    FLevel: Integer;
    FData: Pointer;
    procedure SetText(Value: WideString);
    procedure SetBold(Value: Boolean);
    procedure SetExpanded(Value: Boolean);
    procedure SetImageIndex(Value: Integer);
    procedure SetExpandedIndex(Value: Integer);
    procedure SetOverlayIndex(Value: Integer);
    function GetIndex: Integer;
    function GetAbsoluteIndex: Integer;
    function GetVisibleIndex: Integer;
    function GetCount: Integer;
    function GetItem(Index: Integer): TSimpleTreeNode;
    function GetSelected: Boolean;
    procedure SetSelected(Value: Boolean);
    function GetIsVisible: Boolean;
  protected
    procedure SetParent(Value: TSimpleTreeNode);
    procedure SetChild(Child: TSimpleTreeNode);
    procedure UnsetChild(Child: TSimpleTreeNode);
  public
    constructor Create(AOwner: TSimpleTreeNodes);
    destructor Destroy; override;
    procedure DeleteChildren;
    procedure Delete;
    function GetNextSibling: TSimpleTreeNode;
    function GetPrevSibling: TSimpleTreeNode;
    function GetNextVisible: TSimpleTreeNode;
    function GetPrevVisible: TSimpleTreeNode;
    procedure MakeVisible;
    property Owner: TSimpleTreeNodes read FOwner;
    property Text: WideString read FText write SetText;
    property Bold: Boolean read FBold write SetBold;
    property Expanded: Boolean read FExpanded write SetExpanded;
    property IsVisible: Boolean read GetIsVisible;
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property ExpandedIndex: Integer read FExpandedIndex write SetExpandedIndex;
    property OverlayIndex: Integer read FOverlayIndex write SetOverlayIndex;
    property Level: Integer read FLevel;
    property Parent: TSimpleTreeNode read FParent;
    property Data: Pointer read FData write FData;
    property Index: Integer read GetIndex;
    property AbsoluteIndex: Integer read GetAbsoluteIndex;
    property VisibleIndex: Integer read GetVisibleIndex;
    property Item[Index: Integer]: TSimpleTreeNode read GetItem;
    property Count: Integer read GetCount;
    property Selected: Boolean read GetSelected write SetSelected;
  end;

  TSimpleTreeNodes = class(TObject)
  private
    FOwner: TSimpleTreeView;
    FItems: TList;
    FOnChange: TNotifyEvent;
    FUpdating: Integer;
    function GetItem(Index: Integer): TSimpleTreeNode;
    function GetCount: Integer;
    function GetVisibleCount: Integer;
    procedure DoChange;
  public
    constructor Create(AOwner: TSimpleTreeView);
    destructor Destroy; override;

    function Add(Sibling: TSimpleTreeNode; Text: WideString): TSimpleTreeNode;
    function AddChild(Parent: TSimpleTreeNode; Text: WideString): TSimpleTreeNode;
    procedure Delete(Idx: Integer);
    procedure Clear;
    procedure BeginUpdate;
    procedure EndUpdate;
    function IndexOfData(Data: Pointer): Integer;
    function IndexOf(Item: TSimpleTreeNode): Integer;

    property Owner: TSimpleTreeView read FOwner;
    property Item[Index: Integer]: TSimpleTreeNode read GetItem; default;
    property Count: Integer read GetCount;
    property VisibleCount: Integer read GetVisibleCount;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TSTInfoTipEvent = procedure(Sender: TObject; Node: TSimpleTreeNode;
    var InfoTip: String) of object;
  TSTCalcScrollInfoEvent = procedure(Sender: TObject; AVisibleCount,
    ATopVisibleIndex, APageSize: Integer) of object;
  TSTExpandedEvent = procedure(Sender: TObject; Node: TSimpleTreeNode) of Object;
  TSimpleTreeView = class(TCustomControl)
  private
    FItems: TSimpleTreeNodes;
    FSelectedStyle: TSkinStyle;
    FTransparent: Boolean;
    FImages: TImageList;
    FDragImages: TDragImageList;
    FDragSuspended: Boolean;
    FDragMousePos: TPoint;
    FIndent: Integer;
    FSelected: TSimpleTreeNode;
    FTopItem: TSimpleTreeNode;
    FDragMode: TDragMode;
    FOnChange: TNotifyEvent;
    FOnCalcScrollInfo: TSTCalcScrollInfoEvent;
    FOnInfoTip: TSTInfoTipEvent;
    FOnExpanded: TSTExpandedEvent;
    FOnCollapsed: TSTExpandedEvent;

    procedure SetSelectedStyle(Value: TSkinStyle);
    procedure SetTransparent(Value: Boolean);
    procedure SetImages(Value: TImageList);
    procedure SetIndent(Value: Integer);
    procedure SetSelected(Value: TSimpleTreeNode);
    procedure SetTopItem(Value: TSimpleTreeNode);
    function GetTopItem: TSimpleTreeNode;
    procedure SetBottomItem(Value: TSimpleTreeNode);
    function GetBottomItem: TSimpleTreeNode;
    function GetItemHeight: Integer;
    function GetPageSize: Integer;
    procedure EraseBackground;
    function GetNodeRect(Node: TSimpleTreeNode): TRect;
    procedure DrawItems;
    procedure DrawItem(ACanvas: TCanvas; R: TRect; Node: TSimpleTreeNode);
    procedure ItemsChange(Sender: TObject);
    procedure DoChange;
    procedure DoCalcScrollInfo;
    procedure DoInfoTip(Node: TSimpleTreeNode; var InfoTip: String);
    procedure DoExpanded(Node: TSimpleTreeNode);
    procedure DoCollapsed(Node: TSimpleTreeNode);

    procedure ON_WM_SETFOCUS(var msg: TWMSetFocus); message WM_SETFOCUS;
    procedure ON_WM_KILLFOCUS(var msg: TWMKillFocus); message WM_KILLFOCUS;
    procedure ON_WM_GETDLGCODE(var Msg: TMessage); message WM_GETDLGCODE;
    procedure ON_WM_ERASEBKGND(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure ON_CM_HINTSHOW(var Msg: TMessage); message CM_HINTSHOW;
  protected
    procedure Paint; override;
    procedure Resize; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure Click; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    function GetDragImages: TDragImageList; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure MouseWheelHandler(var Message: TMessage); override;
    procedure BeginDrag(Immediate: Boolean; Threshold: Integer = -1);

    procedure ScrollBy(DeltaX: Integer);
    procedure ScrollToItem(Node: TSimpleTreeNode);
    procedure SetTopVisibleIndex(Idx: Integer);
    function GetNodeAt(X, Y: Integer): TSimpleTreeNode;
    property Selected: TSimpleTreeNode read FSelected write SetSelected;
    property TopItem: TSimpleTreeNode read GetTopItem write SetTopItem;
    property BottomItem: TSimpleTreeNode read GetBottomItem write SetBottomItem;
    property PageSize: Integer read GetPageSize;
    property Items: TSimpleTreeNodes read FItems write FItems;
    property SelectedStyle: TSkinStyle read FSelectedStyle write SetSelectedStyle;
  published
    property Transparent: Boolean read FTransparent write SetTransparent;
    property Images: TImageList read FImages write SetImages;
    property Indent: Integer read FIndent write SetIndent;
    property DragMode: TDragMode read FDragMode write FDragMode default dmManual;

    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnCalcScrollInfo: TSTCalcScrollInfoEvent read FOnCalcScrollInfo write FOnCalcScrollInfo;
    property OnInfoTip: TSTInfoTipEvent read FOnInfoTip write FOnInfoTip;
    property OnExpanded: TSTExpandedEvent read FOnExpanded write FOnExpanded;
    property OnCollapsed: TSTExpandedEvent read FOnCollapsed write FOnCollapsed;

    property Align;
    property Anchors;
    property BevelEdges;
    property BevelInner;
    property BevelKind;
    property BevelOuter;
    property BevelWidth;
    property BiDiMode;
    property BorderWidth;
    property Color;
    property Constraints;
    property DragCursor;
    property DragKind;
    property Enabled;
    property Font;

    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TSimpleTreeView]);
end;

// -----------------------------------------------------------------------------

constructor TSimpleTreeNode.Create(AOwner: TSimpleTreeNodes);
begin
  FOwner := AOwner;
  FItems := TList.Create;
  FBold := False;
  FImageIndex := -1;
  FExpandedIndex := -1;
  FOverlayIndex := -1;
  FLevel := 0;
end;

destructor TSimpleTreeNode.Destroy;
begin
  FItems.Free;
  inherited;
end;

procedure TSimpleTreeNode.SetParent(Value: TSimpleTreeNode);
begin
  FParent := Value;
  if Assigned(FParent) then
  begin
    FLevel := FParent.Level + 1;
    FParent.SetChild(Self);
  end else
    FLevel := 0;
end;

procedure TSimpleTreeNode.SetChild(Child: TSimpleTreeNode);
begin
  if FItems.IndexOf(Child) = -1 then
    FItems.Add(Child);
end;

procedure TSimpleTreeNode.UnsetChild(Child: TSimpleTreeNode);
begin
  if FItems.IndexOf(Child) > -1 then
    FItems.Remove(Child);
end;

// qm[hSč폜
procedure TSimpleTreeNode.DeleteChildren;
begin
  FOwner.BeginUpdate;
  try
  while FItems.Count > 0 do
    TSimpleTreeNode(FItems[0]).Delete;
  finally
    FOwner.EndUpdate;
  end;
end;

// gƎqm[hSč폜
procedure TSimpleTreeNode.Delete;
begin
  FOwner.Delete(Self.AbsoluteIndex);
end;

//@̓x̃m[hԂ
function TSimpleTreeNode.GetNextSibling: TSimpleTreeNode;
var
  Idx, I: Integer;
begin
  Result := nil;
  Idx := GetAbsoluteIndex;
  for I := Idx + 1 to FOwner.Count - 1 do
    if FOwner.Item[I].Level = FLevel then
    begin
      Result := FOwner.Item[I];
      Break;
    end;
end;

// 1O̓x̃m[hԂ
function TSimpleTreeNode.GetPrevSibling: TSimpleTreeNode;
var
  Idx, I: Integer;
begin
  Result := nil;
  Idx := GetAbsoluteIndex;
  for I := Idx - 1 downto 0 do
    if FOwner.Item[I].Level = FLevel then
    begin
      Result := FOwner.Item[I];
      Break;
    end;
end;

// ̉m[hԂ
function TSimpleTreeNode.GetNextVisible: TSimpleTreeNode;
var
  Idx, I: Integer;
begin
  Result := nil;
  Idx := GetAbsoluteIndex;
  for I := Idx + 1 to FOwner.Count - 1 do
    if FOwner.Item[I].IsVisible then
    begin
      Result := FOwner.Item[I];
      Break;
    end;
end;

// Ỏm[hԂ
function TSimpleTreeNode.GetPrevVisible: TSimpleTreeNode;
var
  Idx, I: Integer;
begin
  Result := nil;
  Idx := GetAbsoluteIndex;
  for I := Idx - 1 downto 0 do
    if FOwner.Item[I].IsVisible then
    begin
      Result := FOwner.Item[I];
      Break;
    end;
end;

procedure TSimpleTreeNode.SetText(Value: WideString);
begin
  if FText <> Value then
  begin
    FText := Value;
    FOwner.DoChange;
  end;
end;

procedure TSimpleTreeNode.SetBold(Value: Boolean);
begin
  if FBold <> Value then
  begin
    FBold := Value;
    FOwner.DoChange;
  end;
end;

procedure TSimpleTreeNode.SetExpanded(Value: Boolean);
begin
  if FExpanded <> Value then
  begin
    FExpanded := Value;
    FOwner.DoChange;
  end;
end;

procedure TSimpleTreeNode.SetImageIndex(Value: Integer);
begin
  if FImageIndex <> Value then
  begin
    FImageIndex := Value;
    FOwner.DoChange;
  end;
end;

procedure TSimpleTreeNode.SetExpandedIndex(Value: Integer);
begin
  if FExpandedIndex <> Value then
  begin
    FExpandedIndex := Value;
    FOwner.DoChange;
  end;
end;

procedure TSimpleTreeNode.SetOverlayIndex(Value: Integer);
begin
  if FOverlayIndex <> Value then
  begin
    FOverlayIndex := Value;
    FOwner.DoChange;
  end;
end;

// Zm[h̒ł̑Έʒu
function TSimpleTreeNode.GetIndex: Integer;
var
  Sibling: TSimpleTreeNode;
begin
  if Assigned(FParent) then
    Result := FParent.FItems.IndexOf(Self)
  else
  begin
    Result := 0;  
    Sibling := GetPrevSibling;
    while Assigned(Sibling) do
    begin
      Inc(Result);
      Sibling := Sibling.GetPrevSibling;      
    end;
  end;
end;

// c[Ŝł̐Έʒu
function TSimpleTreeNode.GetAbsoluteIndex: Integer;
begin
  Result := FOwner.IndexOf(Self);
end;

// SẲm[h̒ł̑Έʒu
function TSimpleTreeNode.GetVisibleIndex: Integer;
var
  Node: TSimpleTreeNode;
  Idx: Integer;
begin
  Result := -1;
  if not IsVisible or (FOwner.Count = 0) then  Exit;

  Node := FOwner.Item[0];
  Idx := 0;
  while Assigned(Node) do
  begin
    if Node = Self then
    begin
      Result := Idx;
      Break;
    end;
    Inc(Idx);    
    Node := Node.GetNextVisible;
  end;
end;

function TSimpleTreeNode.GetItem(Index: Integer): TSimpleTreeNode;
begin
  Result := TSimpleTreeNode(FItems[Index]);
end;

function TSimpleTreeNode.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TSimpleTreeNode.GetSelected: Boolean;
begin
  Result := (FOwner.Owner.Selected = Self);
end;

procedure TSimpleTreeNode.SetSelected(Value: Boolean);
begin
  FOwner.Owner.Selected := Self;
end;

function TSimpleTreeNode.GetIsVisible: Boolean;
begin
  if not Assigned(FParent) then
    Result := True
  else if Parent.Expanded then
    Result := Parent.IsVisible
  else
    Result := False;
end;

procedure TSimpleTreeNode.MakeVisible;
begin
  if Assigned(FParent) then
  begin
    FParent.Expanded := True;
    FParent.MakeVisible;
  end;
end;

// -----------------------------------------------------------------------------

constructor TSimpleTreeNodes.Create(AOwner: TSimpleTreeView);
begin
  FOwner := AOwner;
  FItems := TList.Create;
  FUpdating := 0;
end;

destructor TSimpleTreeNodes.Destroy;
begin
  FOnChange := nil;
  Clear;
  FItems.Free;
  inherited;
end;

function TSimpleTreeNodes.GetItem(Index: Integer): TSimpleTreeNode;
begin
  Result := TSimpleTreeNode(FItems[Index]);
end;

function TSimpleTreeNodes.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TSimpleTreeNodes.GetVisibleCount: Integer;
var
  I: Integer;
begin
  Result := 0;
  for I := 0 to FItems.Count - 1 do
    if TSimpleTreeNode(FItems[I]).IsVisible then
      Inc(Result);
end;

procedure TSimpleTreeNodes.DoChange;
begin
  if (FUpdating = 0) and Assigned(FOnChange) then
    FOnChange(Self);
end;

// Zm[hǉ
function TSimpleTreeNodes.Add(Sibling: TSimpleTreeNode; Text: WideString): TSimpleTreeNode;
begin
  if Assigned(Sibling) then
    Result := AddChild(Sibling.Parent, Text)
  else
    Result := AddChild(nil, Text);
end;

// qm[hǉ
function TSimpleTreeNodes.AddChild(Parent: TSimpleTreeNode; Text: WideString):
  TSimpleTreeNode;
var
  Sibling: TSimpleTreeNode;
begin
  Result := TSimpleTreeNode.Create(Self);
  Result.Text := Text;
  if Assigned(Parent) and (Parent.Owner = Self) then
  begin
    Sibling := Parent.GetNextSibling;
    if Assigned(Sibling) then
      FItems.Insert(Sibling.AbsoluteIndex, Result)
    else
      FItems.Add(Result);
  end else
    FItems.Add(Result);
  Result.SetParent(Parent);
  DoChange;
end;

// w肳ꂽm[hƑSĂ̎qm[h폜
procedure TSimpleTreeNodes.Delete(Idx: Integer);
var
  Node: TSimpleTreeNode;
begin
  BeginUpdate;
  try
    Node := TSimpleTreeNode(FItems[Idx]);
    Node.DeleteChildren;
    if Assigned(Node.Parent) then
      Node.Parent.UnsetChild(Node);
//    if FOwner.TopItem = Node then
//      FOwner.FTopItem := Node.GetPrevVisible;
    FreeAndNil(Node);
    FItems.Delete(Idx);
  finally
    EndUpdate;
  end;
//  DoChange;
end;

// S
procedure TSimpleTreeNodes.Clear;
begin
  BeginUpdate;
  try
    while FItems.Count > 0 do
      Delete(0);
  finally
    EndUpdate;
  end;
//  DoChange;
end;

procedure TSimpleTreeNodes.BeginUpdate;
begin
  Inc(FUpdating);
end;

procedure TSimpleTreeNodes.EndUpdate;
begin
  Dec(FUpdating);
  DoChange;
end;

function TSimpleTreeNodes.IndexOfData(Data: Pointer): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if TSimpleTreeNode(FItems[I]).Data = Data then
    begin
      Result := I;
      Break;
    end;
end;

function TSimpleTreeNodes.IndexOf(Item: TSimpleTreeNode): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if TSimpleTreeNode(FItems[I]) = Item then
    begin
      Result := I;
      Break;
    end;
end;

// -----------------------------------------------------------------------------


constructor TSimpleTreeView.Create(AOwner: TComponent);
begin
  inherited;
  FItems := TSimpleTreeNodes.Create(Self);
  FItems.OnChange := ItemsChange;
  FDragImages := TDragImageList.Create(Self);
  FSelectedStyle := TSkinStyle.Create(nil);
  FSelectedStyle.BackgroundColor := clHighLight;
  FSelectedStyle.FontColor := clHighlightText;
  FTransparent := False;
  FIndent := 16;
  FDragMode := dmManual;
  ControlStyle := ControlStyle + [csDisplayDragImage];
  DoubleBuffered := True;
  Color := clWindow;
  TabStop := True;
end;

destructor TSimpleTreeView.Destroy;
begin
  FOnChange := nil;
  FOnCalcScrollInfo := nil;
  FOnInfoTip := nil;
  FOnExpanded := nil;
  FOnCollapsed := nil;
  FSelectedStyle.Free;
  FDragImages.Free;
  FItems.Free;
  inherited;
end;

// OnChange CxggK
procedure TSimpleTreeView.DoChange;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

procedure TSimpleTreeView.DoCalcScrollInfo;
var
  Idx: Integer;
  TopNode: TSimpleTreeNode;
begin
  if Assigned(FOnCalcScrollInfo) then
  begin
    TopNode := GetTopItem;
    if Assigned(TopNode) then
      Idx := TopNode.VisibleIndex
    else
      Idx := 0;
    FOnCalcScrollInfo(Self, Fitems.VisibleCount, Idx, GetPageSize);
  end;
end;

procedure TSimpleTreeView.DoInfoTip(Node: TSimpleTreeNode; var InfoTip: String);
begin
  if Assigned(FOnInfoTip) then
    FOnInfoTip(Self, Node, InfoTip);
end;

procedure TSimpleTreeView.DoExpanded(Node: TSimpleTreeNode);
begin
  if Assigned(FOnExpanded) then
    FOnExpanded(Self, Node);
end;

procedure TSimpleTreeView.DoCollapsed(Node: TSimpleTreeNode);
begin
  if Assigned(FOnCollapsed) then
    FOnCollapsed(Self, Node);
end;

function TSimpleTreeView.GetPageSize: Integer;
begin
  Result := ClientHeight div GetItemHeight;
end;

procedure TSimpleTreeView.SetSelectedStyle(Value: TSkinStyle);
begin
  FSelectedStyle.Assign(Value);
  Invalidate;
end;

procedure TSimpleTreeView.SetTransparent(Value: Boolean);
begin
  if FTransparent <> Value then
  begin
    FTransparent := Value;
    if FTransparent then
    begin
      ParentColor := True;
    end;
    Invalidate;
  end;
end;

procedure TSimpleTreeView.SetImages(Value: TImageList);
begin
  if FImages <> Value then
  begin
    FImages := Value;
    Invalidate;
  end;
end;

procedure TSimpleTreeView.SetIndent(Value: Integer);
begin
  if FIndent <> Value then
  begin
    FIndent := Value;
    Invalidate;
  end;
end;

procedure TSimpleTreeView.SetSelected(Value: TSimpleTreeNode);
begin
  if FSelected <> Value then
  begin
    FSelected := Value;
    Invalidate;
    DoChange;
  end;
end;

procedure TSimpleTreeView.SetTopItem(Value: TSimpleTreeNode);
begin
  if FTopItem <> Value then
  begin
    FTopItem := Value;
    Invalidate;
    DoCalcScrollInfo;
  end;
end;

function TSimpleTreeView.GetTopItem: TSimpleTreeNode;
begin
  if FItems.IndexOf(FTopItem) = -1 then
    FTopItem := nil;
  if not Assigned(FTopItem) and (FItems.Count > 0) then
    FTopItem := FItems[0];
  Result := FTopItem;
end;

procedure TSimpleTreeView.SetBottomItem(Value: TSimpleTreeNode);
var
  NextNode: TSimpleTreeNode;
  Cnt: Integer;
begin
  NextNode := GetBottomItem;
  Cnt := 0;
  while Assigned(NextNode) do
  begin
    if (NextNode = Value) then
    begin
      ScrollBy(Cnt);
      Break;
    end;
    Inc(Cnt);
    NextNode := NextNode.GetNextVisible;
  end;
end;

function TSimpleTreeView.GetBottomItem: TSimpleTreeNode;
var
  NextNode: TSimpleTreeNode;
  Cnt, PSize: Integer;
begin
  NextNode := GetTopItem;
  Result := NextNode;
  Cnt := 0;
  PSize := GetPageSize;
  while Assigned(NextNode) do
  begin
    if (Cnt >= PSize - 1) then
    begin
      Result := NextNode;
      Break;
    end;
    Inc(Cnt);
    NextNode := NextNode.GetNextVisible;
  end;
end;

function TSimpleTreeView.GetItemHeight: Integer;
var
  H: Integer;
begin
  Canvas.Font.Assign(Self.Font);
  H := Canvas.TextHeight('H');
  if Assigned(FImages) and (FImages.Height > H) then
    H := FImages.Height;
  Result := H + 2;
end;

procedure TSimpleTreeView.SetTopVisibleIndex(Idx: Integer);
var
  Node: TSimpleTreeNode;
  Cnt, VCnt: Integer;
begin
  if FItems.Count = 0 then Exit;
  VCnt := FItems.VisibleCount;

  if Idx < 0 then
    Idx := 0
  else if Idx > VCnt - GetPageSize then
    Idx := VCnt - GetPageSize;

  Node := FItems[0];
  Cnt := 0;
  while Assigned(Node) do
  begin
    if Cnt >= Idx then
    begin
      SetTopItem(Node);
      Break;
    end;
    Inc(Cnt);
    Node := Node.GetNextVisible;
  end;
end;

// w肳ꂽsΈʒuXN[
procedure TSimpleTreeView.ScrollBy(DeltaX: Integer);
begin
  if FItems.Count > 0 then
    SetTopVisibleIndex(GetTopItem.VisibleIndex + DeltaX);
end;

// wm[h\ʒu܂ŃXN[
procedure TSimpleTreeView.ScrollToItem(Node: TSimpleTreeNode);
var
  TopNode, BottomNode: TSimpleTreeNode;
begin
  if not Assigned(Node) then Exit;
  TopNode := GetTopItem;
  BottomNode := GetBottomItem;

  Node.MakeVisible;
  if Assigned(TopNode) and (TopNode.AbsoluteIndex > Node.AbsoluteIndex) then
    SetTopItem(Node)
  else
  if Assigned(BottomNode) and (BottomNode.AbsoluteIndex < Node.AbsoluteIndex) then
    SetBottomItem(Node);
end;

function TSimpleTreeView.GetNodeAt(X, Y: Integer): TSimpleTreeNode;
var
  T, H: Integer;
  Node: TSimpleTreeNode;
begin
  Result := nil;
  H := GetItemHeight;
  T := 0;

  Node := GetTopItem;
  while Assigned(Node) do
  begin
    if (T <= Y) and (Y < T + H) then
    begin
      Result := Node;
      Break;
    end;
    Node := Node.GetNextVisible;
    Inc(T, H);
    if T > ClientHeight then
      Break;
  end;
end;

procedure TSimpleTreeView.ON_WM_GETDLGCODE(var Msg: TMessage);
begin
  Msg.Result := DLGC_WANTARROWS;
end;

procedure TSimpleTreeView.ON_CM_HINTSHOW(var Msg: TMessage);
var
  Node: TSimpleTreeNode;
begin
  if Dragging or FDragSuspended then Exit;
  with THintInfo(Pointer(Msg.LParam)^) do
  begin
    Node := GetNodeAt(CursorPos.X, CursorPos.Y);
    HintStr := '';
    if Assigned(Node) then
    begin
      DoInfoTip(Node, HintStr);
      CursorRect := GetNodeRect(Node);
      {
      if Assigned(FImages) then
        HintPos.X := FIndent * Node.Level + FImages.Width
      else
        HintPos.X := FIndent * Node.Level;
      HintPos.Y := CursorRect.Top;
      HintPos := ClientToScreen(HintPos);
      }
    end;
  end;
end;

procedure TSimpleTreeView.ON_WM_SETFOCUS(var msg: TWMSetFocus);
begin
  inherited;
  Invalidate;
end;

procedure TSimpleTreeView.ON_WM_KILLFOCUS(var msg: TWMKillFocus);
begin
  inherited;
  FDragSuspended := False;
  Invalidate;
end;

procedure TSimpleTreeView.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  Node: TSimpleTreeNode;
begin
  // m[hI
  Node := GetNodeAt(X, Y);
  SetSelected(Node);
  // OnMouseDown Cxg
  inherited;
  // m[h̓WJ
  if Assigned(Node) then
    if (Button = mbLeft) then
    begin
      Node.Expanded := not Node.Expanded;
      if Node.Expanded then
        DoExpanded(Node)
      else
        DoCollapsed(Node);
      if FDragMode = dmAutomatic then
        BeginDrag(False, -1);
    end;
end;

procedure TSimpleTreeView.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  // hbOJnhbOJn
  if FDragSuspended and
     (Sqrt(Sqr(FDragMousePos.X - X) + Sqr(FDragMousePos.Y - Y)) >
     Mouse.DragThreshold) then
    BeginDrag(True);
end;

procedure TSimpleTreeView.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  FDragSuspended := False;
end;

procedure TSimpleTreeView.BeginDrag(Immediate: Boolean; Threshold: Integer = -1);
begin
  if Immediate then
  begin
    FDragSuspended := False;
    inherited BeginDrag(True);
//    Update;    
  end else
  begin
    FDragSuspended := True;
    GetCursorPos(FDragMousePos);
    FDragMousePos := ScreenToClient(FDragMousePos);
  end;
end;

function TSimpleTreeView.GetDragImages: TDragImageList;
var
  Bmp: TBitmap;
  NodeRect, R: TRect;
  MousePos: TPoint;
begin
  Result := FDragImages;
  Bmp := TBitmap.Create;
  try
    if Assigned(FSelected) then
    begin
      Result.Clear;
      MousePos := FDragMousePos;
      NodeRect := GetNodeRect(FSelected);
      Bmp.Width := NodeRect.Right - NodeRect.Left;
      Bmp.Height := NodeRect.Bottom - NodeRect.Top;
      R := Rect(0, 0, Bmp.Width, Bmp.Height);
      Bmp.Canvas.Brush.Color := clblue;
      Bmp.Canvas.FillRect(R);
      DrawItem(Bmp.Canvas, R, FSelected);
      Result.Width := Bmp.Width;
      Result.Height := Bmp.Height;
      Result.AddMasked(Bmp, clFuchsia);
      Result.SetDragImage(0, MousePos.X - NodeRect.Left, MousePos.Y - NodeRect.Top);
    end;
  finally
    Bmp.Free;
  end;
end;

procedure TSimpleTreeView.Click;
begin
  // tH[JXݒ
  if not Focused and CanFocus then
    SetFocus;
  inherited;
end;

procedure TSimpleTreeView.KeyDown(var Key: Word; Shift: TShiftState);
  procedure SetSelectedIndex(Idx: Integer);
  begin
    if (Idx >= 0) and (Idx < FItems.Count) then
      SetSelected(FItems[Idx]);
  end;
  procedure SetSelectedIgnoreNil(Node: TSimpleTreeNode);
  begin
    if Assigned(Node) then
      SetSelected(Node);
  end;
begin
  case Key of
  VK_DOWN:                                          // Down
  begin
    if Assigned(FSelected) then
      SetSelectedIgnoreNil(FSelected.GetNextVisible)
    else
      SetSelectedIndex(0);
    ScrollToItem(FSelected);
  end;
  VK_UP:                                            // Up
  begin
    if Assigned(FSelected) then
      SetSelectedIgnoreNil(FSelected.GetPrevVisible)
    else
      SetSelectedIndex(0);
    ScrollToItem(FSelected);
  end;
  VK_HOME:                                          // Home
  begin
    SetSelectedIndex(0);
    ScrollToItem(FSelected);
  end;
  VK_END:                                           // End
  begin
    SetSelectedIndex(FItems.Count - 1);
    ScrollToItem(FSelected);
  end;
  VK_NEXT:  ScrollBy(GetPageSize);                  // PageDown
  VK_PRIOR: ScrollBy(-GetPageSize);                 // PageUp
  VK_LEFT:                                          // Left
  begin
    if Assigned(FSelected) and FSelected.Expanded then
      FSelected.Expanded := False
    else if Assigned(FSelected) and Assigned(FSelected.Parent) then
      SetSelected(FSelected.Parent);
    ScrollToItem(FSelected);      
  end;
  VK_RIGHT:                                          // Right
  begin
    if Assigned(FSelected) and not FSelected.Expanded then
      FSelected.Expanded := True
    else if Assigned(FSelected) and FSelected.Expanded and
      (FSelected.Count > 0) then
      SetSelected(FSelected.Item[0]);
    ScrollToItem(FSelected);
  end;
  end;
  inherited;
end;

procedure TSimpleTreeView.MouseWheelHandler(var Message: TMessage);
begin
  if TWMMouseWheel(Message).WheelDelta > 0 then
    ScrollBy(-3)
  else if TWMMouseWheel(Message).WheelDelta < 0 then
    ScrollBy(3);
  inherited;
end;

procedure TSimpleTreeView.Resize;
begin
  if GetTopItem <> nil then
    SetTopVisibleIndex(GetTopItem.VisibleIndex);
  DoCalcScrollInfo;
  inherited;
end;

procedure TSimpleTreeView.ItemsChange(Sender: TObject);
begin
  if Assigned(FSelected) and (FItems.IndexOf(FSelected) = -1) then
    SetSelected(nil);
  DoCalcScrollInfo;
  Invalidate;
end;

procedure TSimpleTreeView.ON_WM_ERASEBKGND(var Msg:TWMEraseBkgnd);
begin
  Msg.Result := 1;
end;

procedure TSimpleTreeView.Paint;
begin
  EraseBackground;
  DrawItems;
end;

procedure TSimpleTreeView.EraseBackground;
var
  R: TRect;
begin
  R := Canvas.ClipRect;
  if FTransparent and (Parent is TExPanel) then
  begin
    TExPanel(Parent).DrawChildBackground(Self, Canvas, R);
  end else
  begin
    Canvas.Brush.Color := Self.Color;
    Canvas.FillRect(R);
  end;
end;

function TSimpleTreeView.GetNodeRect(Node: TSimpleTreeNode): TRect;
var
  T, H: Integer;
  NextNode: TSimpleTreeNode;
begin
  Result := Rect(0, 0, 0, 0);
  H := GetItemHeight;
  T := 0;

  NextNode := GetTopItem;
  while Assigned(NextNode) and (T < ClientHeight) do
  begin
    if NextNode = Node then
    begin
      Result := ClientRect;
      Result.Top := T;
      Result.Bottom := T + H;
      Break;
    end;
    NextNode := NextNode.GetNextVisible;
    Inc(T, H);
  end;
end;

procedure TSimpleTreeView.DrawItems;
var
  T, H: Integer;
  R: TRect;
  Node: TSimpleTreeNode;
begin
  if FItems.Count = 0 then Exit;
  T := 0;
  H := GetItemHeight;

  Node := GetTopItem;
  while Assigned(Node) do
  begin
    R := ClientRect;
    R.Top := T;
    R.Bottom := T + H;
    DrawItem(Canvas, R, Node);
    Node := Node.GetNextVisible;
    Inc(T, H);
    if T > ClientHeight then
      Break;
  end;
end;

procedure TSimpleTreeView.DrawItem(ACanvas: TCanvas; R: TRect; Node: TSimpleTreeNode);
var
  L, ImgIdx: Integer;
begin
  with ACanvas do
  begin
    // wi̓hԂ
    if Node.Selected and Focused then
      FSelectedStyle.Draw(ACanvas, R);
    // ACR`
    L := Node.Level * Indent;
    if Node.Expanded and (Node.ExpandedIndex > -1) then
      ImgIdx := Node.ExpandedIndex
    else
      ImgIdx := Node.ImageIndex;
    if Assigned(FImages) and (ImgIdx > -1) then
      Images.Draw(ACanvas, L, R.Top + ((R.Bottom - R.Top) - Images.Height)
          div 2, ImgIdx, dsTransparent, itImage);
    if Assigned(Images) and (Node.OverlayIndex > -1) then
      Images.Draw(ACanvas, L, R.Top + ((R.Bottom - R.Top) - Images.Height)
        div 2, Node.OverlayIndex, dsTransparent, itImage);

    // eLXg`
    SetBkMode(Handle, Windows.TRANSPARENT);
    Font.Assign(Self.Font);
    if Node.Selected and Focused then
      Font.Color := FSelectedStyle.FontColor;
    if Node.Bold then
      Font.Style := Font.Style + [fsBold];
    if Assigned(Images) then
      L := L + Images.Width + 3
    else
      L := L + 3;
    R.Left := L;
    WideDrawText(ACanvas, Node.Text, R, DT_LEFT or DT_SINGLELINE or
      DT_VCENTER or DT_END_ELLIPSIS or DT_NOPREFIX);
  end;
end;

end.
