unit ExTabSet;
(* g^uZbg *)

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, ComCtrls, ExPanel,
  Forms, UWideGraphics, USkinStyles;

type
  TExTabItems = class;
  TExTabSet = class;

  TExTabItem = class
  private
    FOwner: TExTabItems;
    FCaption: WideString;
    FEmphasis: Boolean;
    FData: Pointer;
    function GetShortCaption: WideString;
    procedure SetCaption(Value: WideString);
    procedure SetEmphasis(Value: Boolean);
    function GetIndex: Integer;
  public
    constructor Create(AOwner: TExTabItems);
    property Owner: TExTabItems read FOwner;
    property Caption: WideString read FCaption write SetCaption;
    property ShortCaption: WideString read GetShortCaption write SetCaption;
    property Emphasis: Boolean read FEmphasis write SetEmphasis;
    property Data: Pointer read FData write FData;
    property Index: Integer read GetIndex;
  end;

  TExTabItems = class
  private
    FOwner: TExTabSet;
    FItems: TList;
    FOnChange: TNotifyEvent;
    FUpdating: Integer;
    function GetItem(Index: Integer): TExTabItem;
    function GetCount: Integer;
    procedure DoChange;
  public
    constructor Create(AOwner: TExTabSet);
    destructor Destroy; override;

    function Add(Caption: WideString; Data: TObject): TExTabItem;
    procedure Delete(Idx: Integer);
    procedure Clear;
    procedure BeginUpdate;
    procedure EndUpdate;
    function IndexOfData(Data: Pointer): Integer;
    function IndexOf(Item: TExTabItem): Integer;
    procedure Move(CurIndex, NewIndex: Integer);

    property Owner: TExTabSet read FOwner;
    property Items[Index: Integer]: TExTabItem read GetItem; default;
    property Count: Integer read GetCount;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TXTInfoTipEvent = procedure(Sender: TObject; Tab: TExTabItem;
    var InfoTip: String) of object;
  TExTabSet = class(TCustomControl)
  private
    FStyle: TSkinStyle;
    FActiveItemStyle: TSkinStyle;
    FInactiveItemStyle: TSkinStyle;
    FItems: TExTabItems;
    FTabIndex: Integer;
    FFirstIndex: Integer;
    FTabHeight: Integer;
    FUpDown: TUpDown;
    FTabPosition: TTabPosition;
    FMaxCaptionLength: Integer;
    FOnChange: TNotifyEvent;
    FOnInfoTip: TXTInfoTipEvent;

    procedure SetTabHeight(Value: Integer);
    procedure SetTabIndex(Value: Integer);
    procedure SetFirstIndex(Value: Integer);
    procedure SetStyle(Value: TSkinStyle);
    procedure SetActiveItemStyle(Value: TSkinStyle);
    procedure SetInactiveItemStyle(Value: TSkinStyle);
    procedure SetTabPosition(Value: TTabPosition);
    procedure ItemsChange(Sender: TObject);
    procedure UpDownChangingEx(Sender: TObject; var AllowChange: Boolean; NewValue: SmallInt; Direction: TUpDownDirection);
    procedure DoChange;
    procedure DoInfoTip(Tab: TExTabItem; var InfoTip: String);

    function GetTabRect(Idx: Integer): TRect;
    procedure SetMaxCaptionLength(Value: Integer);
    procedure DrawTab(ACanvas: TCanvas; ARect: TRect; Tab: TExTabItem; Selected: Boolean);
    procedure ON_CM_HINTSHOW(var Msg: TMessage); message CM_HINTSHOW;
  protected
    procedure Paint; override;
    procedure Resize; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ItemAtPos(Pos: TPoint): Integer;
    function GetTabAt(X, Y: Integer): TExTabItem;
    property FirstIndex: Integer read FFirstIndex write SetFirstIndex;
    property Items: TExTabItems read FItems;
  published
    property TabHeight: Integer read FTabHeight write SetTabHeight default 16;
    property TabIndex: Integer read FTabIndex write SetTabIndex default -1;
    property Style: TSkinStyle read FStyle write SetStyle;
    property ActiveItemStyle: TSkinStyle read FActiveItemStyle write SetActiveItemStyle;
    property InactiveItemStyle: TSkinStyle read FInactiveItemStyle write SetInactiveItemStyle;
    property TabPosition: TTabPosition read FTabPosition write SetTabPosition default tpTop;
    property MaxCaptionLength: Integer read FMaxCaptionLength write SetMaxCaptionLength;

    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnInfoTip: TXTInfoTipEvent read FOnInfoTip write FOnInfoTip;

    property Align;
    property Hint;
    property DragKind;
    property DragCursor;
    property DragMode;
    property Font;
    property ParentFont;
    property ShowHint;
    property ParentShowHint;
    property Visible;    

    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnStartDrag;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheel;
    property OnMouseWheelUp;
    property OnMouseWheelDown;    
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExTabSet]);
end;

constructor TExTabItem.Create(AOwner: TExTabItems);
begin
  FOwner := AOwner;
end;

function TExTabItem.GetShortCaption: WideString;
begin
  if Length(FCaption) > FOwner.Owner.MaxCaptionLength then
  begin
    Result := FCaption;
    SetLength(Result, FOwner.Owner.MaxCaptionLength);
    Result := Result + '...';
  end else
    Result := FCaption;
end;

procedure TExTabItem.SetCaption(Value: WideString);
begin
  if FCaption <> Value then
  begin
    FCaption := Value;
    FOwner.DoChange;
  end;
end;

procedure TExTabItem.SetEmphasis(Value: Boolean);
begin
  if FEmphasis <> Value then
  begin
    FEmphasis := Value;
    FOwner.DoChange;
  end;
end;

function TExTabItem.GetIndex: Integer;
begin
  Result := FOwner.IndexOf(Self);
end;

// -----------------------------------------------------------------------------

constructor TExTabItems.Create(AOwner: TExTabSet);
begin
  FOwner := AOwner;
  FItems := TList.Create;
  FUpdating := 0;
end;

destructor TExTabItems.Destroy;
begin
  Clear;
  FItems.Free;
end;

function TExTabItems.GetItem(Index: Integer): TExTabItem;
begin
  Result := TExTabItem(FItems[Index]);
end;

function TExTabItems.GetCount: Integer;
begin
  Result := FItems.Count;
end;

procedure TExTabItems.DoChange;
begin
  if (FUpdating = 0) and (@FOnChange <> nil) then
    FOnChange(Self);
end;

// ǉ
function TExTabItems.Add(Caption: WideString; Data: TObject): TExTabItem;
begin
  Result := TExTabItem.Create(Self);
  FItems.Add(Result);
  Result.Caption := Caption;
  Result.Data := Data;
  DoChange;
end;

// 폜
procedure TExTabItems.Delete(Idx: Integer);
begin
  TExTabItem(FItems[Idx]).Free;
  FItems.Delete(Idx);
  DoChange;
end;

// S
procedure TExTabItems.Clear;
begin
  while FItems.Count > 0 do
  begin
    TExTabItem(FItems[0]).Free;
    FItems.Delete(0);
  end;
  DoChange;
end;

procedure TExTabItems.BeginUpdate;
begin
  Inc(FUpdating);
end;

procedure TExTabItems.EndUpdate;
begin
  Dec(FUpdating);
  DoChange;
end;

function TExTabItems.IndexOfData(Data: Pointer): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if TExTabItem(FItems[I]).Data = Data then
    begin
      Result := I;
      Break;
    end;
end;

function TExTabItems.IndexOf(Item: TExTabItem): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if TExTabItem(FItems[I]) = Item then
    begin
      Result := I;
      Break;
    end;
end;

procedure TExTabItems.Move(CurIndex, NewIndex: Integer);
begin
  FItems.Move(CurIndex, NewIndex);
  DoChange;
end;

// -----------------------------------------------------------------------------

constructor TExTabSet.Create(AOwner: TComponent);
begin
  inherited;
  FStyle := TSkinStyle.Create(nil);
  FActiveItemStyle := TSkinStyle.Create(nil);
  FInactiveItemStyle := TSkinStyle.Create(nil);

  with FActiveItemStyle do
  begin
    FontStyle := [fsBold];
    BorderStyle := sbInset;
    BackgroundColor := clBtnFace;
  end;
  with FInactiveItemStyle do
  begin
    BackgroundColor := clBtnFace;
  end;
  with FStyle do
  begin
    BackgroundColor := clBtnFace;
  end;

  FUpDown := TUpDown.Create(Self);
  FUpDown.Hide;
  FUpDown.OnChangingEx := UpDownChangingEx;
  FUpDown.Parent := Self;
  FUpDown.Orientation := udHorizontal;

  FTabIndex := -1;
  FTabHeight := 16;
  FTabPosition := tpTop;
  FMaxCaptionLength := 10;  
  DoubleBuffered := True;

  FItems := TExTabItems.Create(Self);
  FItems.OnChange := ItemsChange;
end;

destructor TExTabSet.Destroy;
begin
  FStyle.Free;
  FActiveItemStyle.Free;
  FInactiveItemStyle.Free;
  FItems.Free;
  FUpDown.Free;
  inherited;
end;

function TExTabSet.ItemAtPos(Pos: TPoint): Integer;
var
  I: Integer;
  R: TRect;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
  begin
    R := GetTabRect(I);
    if (Pos.X > R.Left) and (Pos.X < R.Right) and (Pos.Y > R.Top) and (Pos.Y < R.Bottom) then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TExTabSet.GetTabAt(X, Y: Integer): TExTabItem;
var
  Idx: Integer;
begin
  Idx := ItemAtPos(Point(X, Y));
  if Idx > -1 then
    Result := FItems[Idx]
  else
    Result := nil;
end;

procedure TExTabSet.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  Sel: Integer;
begin
  Sel := ItemAtPos(Point(X, Y));
  if (Sel > -1) then
  begin
    if (Sel <> FTabIndex) then
      SetTabIndex(Sel);
  end;
  inherited;  
end;

procedure TExTabSet.Resize;
begin
  case FTabPosition of
    tpTop:    FUpDown.Top := 0;
    tpBottom: FUpDown.Top := 2;
    tpLeft:   FUpDown.Left := 0;
    tpRight:  FUpDown.Left := 2;
  end;

  case FTabPosition of
    tpTop, tpBottom:
    begin
      FUpDown.Orientation := udHorizontal;
      FUpDown.Height := ClientHeight - 2;
      FUpDown.Width := 25;
      FUpDown.Left := ClientWidth - FUpDown.Width;
    end;
    tpLeft, tpRight:
    begin
      FUpDown.Orientation := udVertical;
      FUpDown.Height := 25;
      FUpDown.Width := ClientWidth - 2;
      FUpDown.Top := ClientHeight - FUpDown.Height;
    end;
  end;
end;

procedure TExTabSet.Paint;
var
  I: Integer;
  R: TRect;
  BtnVisible: Boolean;
begin
  Canvas.Font.Assign(Self.Font);
  BtnVisible := False;

  with Canvas do
  begin
    if (FStyle.BackgroundColor = clNone) and FStyle.BackgroundImage.Empty and
      (Parent is TExPanel) then
    begin
      TExPanel(Parent).DrawChildBackground(Self, Canvas, ClientRect);
      FStyle.DrawBorder(Canvas, ClientRect, True);
    end else
      FStyle.Draw(Canvas, ClientRect, True);
    for I := FItems.Count - 1 downto FFirstIndex do
    begin
      R := GetTabRect(I);
      if I <> FTabIndex then
        DrawTab(Canvas, R, FItems[I], False);
      case FTabPosition of
        tpTop, tpBottom: if (R.Right > ClientWidth) then BtnVisible := True;
        tpLeft, tpRight: if (R.Bottom > ClientHeight) then BtnVisible := True;
      end;
    end;
    if (FTabIndex > -1) and (FTabIndex < FItems.Count) and
       (FTabIndex >= FFirstIndex) then
    begin
      R := GetTabRect(FTabIndex);
      DrawTab(Canvas, R, FItems[FTabIndex], True);
    end;
  end;
  FUpDown.Visible := BtnVisible or (FFirstIndex > 0);
end;

function TExTabSet.GetTabRect(Idx: Integer): TRect;
var
  I, W, CaptionWidth: Integer;
begin
  Canvas.Font.Style := Canvas.Font.Style + [fsBold];
  W := 10;
  for I := FFirstIndex to Idx - 1 do
    W := W + WideTextWidth(Canvas, FItems[I].ShortCaption) + 10;

  CaptionWidth := WideTextWidth(Canvas, FItems[Idx].ShortCaption);
  if Idx >= FFirstIndex then
    case FTabPosition of
      tpTop:
      begin
        Result.TopLeft := Point(W, ClientHeight - FTabHeight);
        Result.BottomRight := Point(W + CaptionWidth + 10, ClientHeight);
      end;
      tpBottom:
      begin
        Result.TopLeft := Point(W, 0);
        Result.BottomRight := Point(W + CaptionWidth + 10, FTabHeight);
      end;
      tpLeft:
      begin
        Result.TopLeft := Point(ClientWidth - FTabHeight, W);
        Result.BottomRight := Point(ClientWidth, W + CaptionWidth + 10);
      end;
      tpRight:
      begin
        Result.TopLeft := Point(0, W);
        Result.BottomRight := Point(FTabHeight, W + CaptionWidth + 10);
      end;
    end
  else
  begin
    Result.TopLeft := Point(0, 0);
    Result.BottomRight := Point(0, 0);
  end;
  Canvas.Font.Style := Self.Font.Style;
end;

procedure TExTabSet.DrawTab(ACanvas: TCanvas; ARect: TRect; Tab: TExTabItem; Selected: Boolean);
var
  lf: TLogFont;
  tf: TFont;
  ACaption: WideString;
  Extent: TSize;
begin
  with ACanvas do
  begin
    if Selected then
    begin
      FActiveItemStyle.Draw(ACanvas, ARect, True);
      FActiveItemStyle.AssignToFont(ACanvas.Font);
    end else
    begin
      FInactiveItemStyle.Draw(ACanvas, ARect, True);
      FInactiveItemStyle.AssignToFont(ACanvas.Font);
    end;
    if Tab.Emphasis then
      Font.Color := clBlue;
  end;

  InflateRect(ARect, -1, -1);
  ACaption := Tab.ShortCaption;
  Extent := WideTextExtent(ACanvas, ACaption);
  SetBkMode(ACanvas.Handle, Windows.TRANSPARENT);

  if (FTabPosition = tpLeft) or (FTabPosition = tpRight) then
  begin
    tf := TFont.Create;
    tf.Assign(ACanvas.Font);
    GetObject(tf.Handle, sizeof(lf), @lf);
    lf.lfEscapement := 2700;
    lf.lfOrientation := 0;
    tf.Handle := CreateFontIndirect(lf);
    ACanvas.Font.Assign(tf);
    tf.Free;
    WideTextOut(ACanvas,
      ARect.Right - ((ARect.Right - ARect.Left) - Extent.cy) div 2,
      ARect.Top + ((ARect.Bottom - ARect.Top) - Extent.cx) div 2,
      ACaption);
  end else
    WideTextOut(ACanvas,
      ARect.Left + ((ARect.Right - ARect.Left) - Extent.cx) div 2,
      ARect.Top + ((ARect.Bottom - ARect.Top) - Extent.cy) div 2,
      ACaption);
end;

procedure TExTabSet.ON_CM_HINTSHOW(var Msg: TMessage);
var
  Tab: TExTabItem;
begin
  with THintInfo(Pointer(Msg.LParam)^) do
  begin
    Tab := GetTabAt(CursorPos.X, CursorPos.Y);
    HintStr := '';
    if Assigned(Tab) then
    begin
      DoInfoTip(Tab, HintStr);
      CursorRect := GetTabRect(Tab.Index);
    end;
  end;
end;

procedure TExTabSet.DoChange;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

procedure TExTabSet.DoInfoTip(Tab: TExTabItem; var InfoTip: String);
begin
  if Assigned(FOnInfoTip) then
    FOnInfoTip(Self, Tab, InfoTip);
end;

procedure TExTabSet.SetTabHeight(Value: Integer);
begin
  if FTabHeight <> Value then
  begin
    FTabHeight := Value;
    Invalidate;
  end;
end;

procedure TExTabSet.ItemsChange(Sender: TObject);
begin
  if FTabIndex >= FItems.Count then
    SetTabIndex(-1);

  case FTabPosition of
    tpLeft, tpRight:
    begin
      FUpDown.Min := FItems.Count - 1;
      FUpDown.Max := 0;
    end;
    tpTop, tpBottom:
    begin
      FUpDown.Min := 0;
      FUpDown.Max := FItems.Count - 1;
    end;
  end;
  FUpDown.Position := FFirstIndex;
  Invalidate;
end;

procedure TExTabSet.SetTabIndex(Value: Integer);
begin
  if FTabIndex <> Value then
  begin
    FTabIndex := Value;
    Invalidate;
    DoChange;
  end;
end;

procedure TExTabSet.SetFirstIndex(Value: Integer);
begin
  if FFirstIndex <> Value then
  begin
    FFirstIndex := Value;
    FUpDown.Position := FFirstIndex;
    Invalidate;
  end;
end;

procedure TExTabSet.SetTabPosition(Value: TTabPosition);
begin
  if FTabPosition <> Value then
  begin
    FTabPosition := Value;
    Resize;
    ItemsChange(Self);
    Invalidate;
  end;
end;

// v90x]i蔲ɂx...j
procedure RotateBitmap(Source: TBitmap; Count: Integer = 1);
var
  X, Y, W, H: Integer;
  Dest: TBitmap;
begin
  W := Source.Width;
  H := Source.Height;
  Dest := TBitmap.Create;
  try
    Dest.Assign(Source);
    if Count mod 2 = 0 then
    begin
      Dest.Width  := W;
      Dest.Height := H;
    end else
    begin
      Dest.Width  := H;
      Dest.Height := W;
    end;
    for Y := 0 to H - 1 do
      for X := 0 to W - 1 do
        case Count of
        1: Dest.Canvas.Pixels[H - Y - 1, X] :=  Source.Canvas.Pixels[X, Y];
        2: Dest.Canvas.Pixels[W - X - 1, H - Y - 1] :=  Source.Canvas.Pixels[X, Y];
        3: Dest.Canvas.Pixels[Y, W - X - 1] :=  Source.Canvas.Pixels[X, Y];
        end;
    Source.Assign(Dest);
  finally
    Dest.Free;
  end;
end;
{
procedure RotateStyle(Source: TSkinStyle; Count: Integer = 1);
var
  Bmp: TBitmap;
begin

  RotateBitmap(Source.TopImage, Count);
  RotateBitmap(Source.BottomImage, Count);
  RotateBitmap(Source.LeftImage, Count);
  RotateBitmap(Source.RightImage, Count);
  RotateBitmap(Source.TopLeftImage, Count);
  RotateBitmap(Source.TopRightImage, Count);
  RotateBitmap(Source.BottomLeftImage, Count);
  RotateBitmap(Source.BottomRightImage, Count);

  Bmp := Source.TopImage;
  Source.TopImage := Source.LeftImage;
  Source.LeftImage := Source.BottomImage;
  Source.BottomImage := Source.RightImage;
  Source.RightImage := Bmp;
  Bmp := Source.TopLeftImage;
  Source.TopLeftImage := Source.BottomLeftImage;
  Source.BottomLeftImage := Source.BottomRightImage;
  Source.BottomRightImage := Source.TopRightImage;
  Source.TopRightImage := Bmp;
end;
}
procedure TExTabSet.SetStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FStyle.Assign(Value);
    {
    case FTabPosition of
    tpRight   : RotateStyle(FStyle);
    tpBottom  : RotateStyle(FStyle, 2);
    tpLeft    : RotateStyle(FStyle, 3);
    end;
    }
    Invalidate;
  end;
end;

procedure TExTabSet.SetActiveItemStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FActiveItemStyle.Assign(Value);
    {
    case FTabPosition of
    tpRight   : RotateStyle(FActiveItemStyle);
    tpBottom  : RotateStyle(FActiveItemStyle, 2);
    tpLeft    : RotateStyle(FActiveItemStyle, 3);
    end;
    }
    Invalidate;
  end;
end;

procedure TExTabSet.SetInactiveItemStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FInactiveItemStyle.Assign(Value);
    {
    case FTabPosition of
    tpRight   : RotateStyle(FInactiveItemStyle);
    tpBottom  : RotateStyle(FInactiveItemStyle, 2);
    tpLeft    : RotateStyle(FInactiveItemStyle, 3);
    end;
    }
    Invalidate;
  end;
end;

procedure TExTabSet.UpDownChangingEx(Sender: TObject; var AllowChange: Boolean; NewValue: SmallInt; Direction: TUpDownDirection);
begin
  if (NewValue >= 0) and (NewValue < FItems.Count) then
  begin
    FFirstIndex := NewValue;
    Invalidate;
  end;
end;

procedure TExTabSet.SetMaxCaptionLength(Value: Integer);
begin
  if FMaxCaptionLength <> Value then
  begin
    FMaxCaptionLength := Value;
    Invalidate;
  end;
end;

end.
