unit ExSpeedButton;
(* gXs[h{^ *)

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, ExtCtrls,
  USkinStyles;

type
  TButtonState = (bsUp, bsDown, bsHot);
  TExSpeedButton = class(TGraphicControl)
  private
    FUpStyle: TSkinStyle;
    FDownStyle: TSkinStyle;
    FHotStyle: TSkinStyle;
    FButtonState: TButtonState;
    FOldButtonState: TButtonState;
    FMouseDowned: Boolean;
    FMouseTimer: TTimer;
    procedure MouseTimerProc(Sender: TObject);
    procedure SetUpStyle(Value: TSkinStyle);
    procedure SetDownStyle(Value: TSkinStyle);
    procedure SetHotStyle(Value: TSkinStyle);
    procedure SetColor(Value: TColor);
    function GetColor: TColor;
    procedure SetFont(Value: TFont);
    function GetFont: TFont;
  protected
    procedure Paint; override;
    procedure Resize; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseEnter;
    procedure MouseLeave;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property UpStyle: TSkinStyle read FUpStyle write SetUpStyle;
    property DownStyle: TSkinStyle read FDownStyle write SetDownStyle;
    property HotStyle: TSkinStyle read FHotStyle write SetHotStyle;
  published
    property Color: TColor read GetColor write SetColor;
    property Font: TFont read GetFont write SetFont;
    property ParentFont;
    property ParentColor;
    property Align;
    property Anchors;
    property Action;
    property Caption;
    property Hint;
    property ShowHint;
    property ParentShowHint;

    property OnClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExSpeedButton]);
end;

constructor TExSpeedButton.Create(AOwner: TComponent);
begin
  inherited;
  FUpStyle := TSkinStyle.Create(nil);
  FDownStyle := TSkinStyle.Create(nil);
  FHotStyle := TSkinStyle.Create(nil);

  with FUpStyle do
  begin
    BackgroundColor := inherited Color;
    FontFamily := inherited Font.Name;
    FontStyle := inherited Font.Style;
    FontColor := inherited Font.Color;
    FontSize := inherited Font.Size;
  end;
  FDownStyle.Assign(FUpStyle);
  FHotStyle.Assign(FUpStyle);

  FMouseTimer := TTimer.Create(Self);
  FMouseTimer.Enabled := False;
  FMouseTimer.Interval := 100;
  FMouseTimer.OnTimer := MouseTimerProc;
  FButtonState := bsUp;
  FOldButtonState := bsUp;
  FMouseDowned := False;
end;

destructor TExSpeedButton.Destroy;
begin
  FUpStyle.Free;
  FDownStyle.Free;
  FHotStyle.Free;
  FMouseTimer.Free;
  inherited;
end;

procedure TExSpeedButton.SetColor(Value: TColor);
begin
  inherited Color := Value;
  FUpStyle.BackgroundColor := Value;
  FDownStyle.BackgroundColor := Value;
  FHotStyle.BackgroundColor := Value;
end;

function TExSpeedButton.GetColor: TColor;
begin
  Result := inherited Color;
end;

procedure TExSpeedButton.SetFont(Value: TFont);
  procedure FontToStyle(AStyle: TSkinStyle);
  begin
    AStyle.FontFamily := Value.Name;
    AStyle.FontStyle := Value.Style;
    AStyle.FontSize := Value.Size;
    AStyle.FontColor := Value.Color;
  end;
begin
  inherited Font := Value;
  FontToStyle(FUpStyle);
  FontToStyle(FDownStyle);
  FontToStyle(FHotStyle);
end;

function TExSpeedButton.GetFont: TFont;
begin
  Result := inherited Font;
end;

procedure TExSpeedButton.SetUpStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FUpStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExSpeedButton.SetDownStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FDownStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExSpeedButton.SetHotStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FHotStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExSpeedButton.Paint;
var
  R: TRect;
  Style: TSkinStyle;
begin
  case FButtonState of
  bsUp    : Style := FUpStyle;
  bsDown  : Style := FDownStyle;
  bsHot   : Style := FHotStyle;
  else
    Style := FUpStyle;
  end;

  Style.Draw(Canvas, ClientRect);
  if Style.BackgroundImage.Empty then
  begin
    R := ClientRect;
    if FButtonState = bsDown then
      OffsetRect(R, 1, 1);
    Canvas.Font.Assign(Self.Font);
    Style.DrawText(Canvas, R, Caption);
  end;
end;

procedure TExSpeedButton.Resize;
begin
  inherited;
  Invalidate;
end;

procedure TExSpeedButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  if Button = mbLeft then
  begin
    FButtonState := bsDown;
    FOldButtonState := bsDown;
    FMouseDowned := True;
    FMouseTimer.Enabled := False;
    Invalidate;
  end;
end;

procedure TExSpeedButton.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  Rect: TRect;
  MouseIn: Boolean;
begin
  inherited;
  Rect := ClientRect;
  MouseIn :=  (X>=Rect.Left) and (X<Rect.Right) and (Y>=Rect.Top) and (Y<Rect.Bottom);
  if(FMouseDowned = True) and not MouseIn then
  begin
    FButtonState := bsUp;
    if(FOldButtonState <> FButtonState) then
      Invalidate;
  end else
  if(FMouseDowned = True) then
  begin
    FButtonState := bsDown;
    if(FOldButtonState <> FButtonState) then
      Invalidate;
  end else
  if (FButtonState <> bsHot) and MouseIn then
    MouseEnter;
  FOldButtonState := FButtonState;
end;

procedure TExSpeedButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  if Button = mbLeft then
  begin
    FButtonState := bsUp;
    FOldButtonState := bsUp;
    FMouseDowned := False;
    Invalidate;
  end;
end;

procedure TExSpeedButton.MouseTimerProc(Sender: TObject);
var
  MPos: TPoint;
begin
  GetCursorPos(MPos);
  MPos := ScreenToClient(MPos);
  if (ClientRect.Left > MPos.X) or (ClientRect.Right < MPos.X) or
     (ClientRect.Top > MPos.Y) or (ClientRect.Bottom < MPos.Y) then
     MouseLeave;
end;

procedure TExSpeedButton.MouseEnter;
begin
  if Enabled and (FButtonState = bsUp) and (FButtonState <> bsHot) then
  begin
    FButtonState := bsHot;
    FMouseTimer.Enabled := True;
    Invalidate;
  end;
end;

procedure TExSpeedButton.MouseLeave;
begin
  if Enabled and (FButtonState = bsHot) then
  begin
    FButtonState := bsUp;
    FMouseTimer.Enabled := False;
    Invalidate;
  end;
end;

end.
