/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model {

	import flash.events.Event;
	import flash.events.MouseEvent;
	
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.manager.ComponentManager;
	
	import mx.containers.ControlBar;
	import mx.containers.TitleWindow;
	import mx.containers.VBox;
	import mx.controls.Button;
	import mx.controls.Text;
	import mx.controls.TextInput;
	import mx.core.Container;

	/**
	 * <p>[概 要]</p>
	 * ユーザとのインタラクティブな会話を行う為の機能モデルです.
	 * 
	 * <p>[詳 細]</p>
	 * Reffiが形成するシステムロジックフローを中断してユーザとの対話を行う画面を表示します。<br />
	 * Flexでは「表示された画面スレッド」と「画面を表示したスレッド」は常にパラレル動作しますが、<br />
	 * この機能モデルを使用することで、両スレッドが同期的に実行されます。
	 * <br /> 
	 * 表示される画面の構成は各種プロパティを外部から設定することで変更出来ます。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @example 
	 * HTTP送信前にユーザに送信是非を確認する
	 * 
	 * <listing version="3.0">
         public class SendAction extends BaseAction {
            
            override protected function prepare(parameterMapping:ParameterMapping):Boolean {
                var ret:Boolean = true;
                
                // 画面入力情報を取得します
                var postal:String = TextInput(getComponentByName("form.postal1")).text + "-" + 
                                    TextInput(getComponentByName("form.postal2")).text;
                var address:String = TextInput(getComponentByName("form.address")).text;
                var username:String = TextInput(getComponentByName("form.username")).text;
                var tel:String = TextInput(getComponentByName("form.tel1")).text + 
                      TextInput(getComponentByName("form.tel2")).text + 
                      TextInput(getComponentByName("form.tel3")).text;
                var fax:String = TextInput(getComponentByName("form.fax1")).text + 
                      TextInput(getComponentByName("form.fax2")).text + 
                      TextInput(getComponentByName("form.fax3")).text;
                var birth:String = TextInput(getComponentByName("form.birth")).text;
                var notes:String = TextArea(getComponentByName("form.notes")).text;
                
                // prepareメソッド外で利用する為、parameterMappingに設定しておきます
                parameterMapping.setParameter("postal", postal);
                parameterMapping.setParameter("address", address);
                parameterMapping.setParameter("username", username);
                parameterMapping.setParameter("tel", tel);
                parameterMapping.setParameter("fax", fax);
                parameterMapping.setParameter("birth", birth);
                parameterMapping.setParameter("notes", notes);
	                
                return ret;
            }
    
            // ① HTTP送信モデルの前にUserInteractiveCoreを予約します
            override protected function reserveModels(models:Array):void {
                models.push(Class(UserInteractiveCore));
                models.push(Class(HTTPServiceCore));
            }

            // ② UserInteractiveCoreで表示するメッセージを形成します
            //    HTTPServiceCoreのターン(index 1)では前回ターン(index 0)のUserInteractiveCoreの結果を判定し、
            //    「いいえ」が選択されていれば送信を取りやめます    
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch(index) {
                    case 0:
                        var message:String = "以下の内容で送信しますか？\n\n";
                        message += "郵便番号：" + parameterMapping.getParameter("postal") + "\n";
                        message += "住所：" + parameterMapping.getParameter("address") + "\n";
                        message += "氏名：" + parameterMapping.getParameter("username") + "\n";
                        message += "電話番号：" + parameterMapping.getParameter("tel") + "\n";
                        message += "FAX：" + parameterMapping.getParameter("fax") + "\n";
                        message += "誕生日：" + parameterMapping.getParameter("birth") + "\n";
                        message += "備考：" + parameterMapping.getParameter("notes") + "\n";
    
                        UserInteractiveCore(next).message = message;
                        break;
                    case 1:
                        if(Button(MouseEvent(prev.cause).currentTarget).label == "いいえ") {
                            return false;
                        }
                        HTTPServiceCore(next).url ="webController";
                        HTTPServiceCore(next).resultFormat = RPCCore.E4X;
                        HTTPServiceCore(next).addUrlParameter("model.fqcn", "demo.rpc.model.DummyRestResponseModel");
                        HTTPServiceCore(next).addUrlParameter("forward.page", "DummyRestResponse.xml");
                        HTTPServiceCore(next).addUrlParameter("postal", parameterMapping.getParameter("postal") as String);
                        HTTPServiceCore(next).addUrlParameter("address", parameterMapping.getParameter("address") as String);
                        HTTPServiceCore(next).addUrlParameter("username", parameterMapping.getParameter("username") as String);
                        HTTPServiceCore(next).addUrlParameter("tel", parameterMapping.getParameter("tel") as String);
                        HTTPServiceCore(next).addUrlParameter("fax", parameterMapping.getParameter("fax") as String);
                        HTTPServiceCore(next).addUrlParameter("birth", parameterMapping.getParameter("birth") as String);
                        HTTPServiceCore(next).addUrlParameter("notes", parameterMapping.getParameter("notes") as String);
                        break;
                    case 2:
                        Alert.show("正常に送信されました");
                        break;
                        
                }
                return true;
            }
        }
	 *  
	 * </listing>
	 * 
	 * 様々な対話ウィンドウを連続で表示する
	 * <listing version="3.0">
        override protected function reserveModels(models:Array):void{
            models.push(Class(UserInteractiveCore));
            models.push(Class(UserInteractiveCore));
            models.push(Class(UserInteractiveCore));
            models.push(Class(UserInteractiveCore));
        }
        
        override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
            switch(index){
                case 0:
                    // 次モデル設定
                    // 通常使用パターン
                    UserInteractiveCore(next).title = "確認";
                    UserInteractiveCore(next).message = "続行しますか？";
                    break;
                case 1:
                    // 前モデル結果評価
                    if(Button(MouseEvent(prev.cause).currentTarget).label == "いいえ") {
                        return false;
                    }
                    
                    // 次モデル設定
                    // 独自ボタン使用パターン
                    UserInteractiveCore(next).title = "質問１";
                    UserInteractiveCore(next).message = "性別を選択して下さい";
                    var male:Button = new Button();
                    male.label = "男";
                    var female:Button = new Button();
                    female.label = "女";
                    var buttons:Array = [male, female];
                    UserInteractiveCore(next).buttons = buttons;
                    break;
                case 2:
                    // 前モデル結果評価
                    var gender:Label = new Label();
                    gender.text = "性別：" + Button(MouseEvent(prev.cause).currentTarget).label;
                    DisplayObjectContainer(getComponentByName("userInteractive.vbox")).addChild(gender);
                
                    // 次モデル設定
                    // INPUTモード、ボタンオプション使用パターン
                    UserInteractiveCore(next).title = "質問２";
                    UserInteractiveCore(next).message = "氏名を入力して下さい";
                    UserInteractiveCore(next).mode = UserInteractiveCore.INPUT_MODE;
                    UserInteractiveCore(next).buttonOption = UserInteractiveCore.YES_OPTION;
                    break;
                case 3:
                    // 前モデル結果評価
                    var name:Label = new Label();
                    name.text = "氏名：" + UserInteractiveCore(prev.target).getInputText(); 
                    DisplayObjectContainer(getComponentByName("userInteractive.vbox")).addChild(name);
                    
                    // 次モデル設定
                    // 独自コンテントコンテナ使用パターン
                    var label:Label = new Label();
                    label.text = "血液型を選択して下さい";
                    var aType:RadioButton = new RadioButton();
                    aType.label = "A型";
                    aType.selected = true;
                    aType.name = "aType";
                    var bType:RadioButton = new RadioButton();
                    bType.label = "B型";
                    var abType:RadioButton = new RadioButton();
                    abType.label = "AB型";
                    var oType:RadioButton = new RadioButton();
                    oType.label = "O型";
                    
                    var bloodType:HBox = new HBox();
                    bloodType.addChild(aType);
                    bloodType.addChild(bType);
                    bloodType.addChild(abType);
                    bloodType.addChild(oType);

                    var content:VBox = new VBox();
                    content.addChild(label);
                    content.addChild(bloodType);
                                                            
                    UserInteractiveCore(next).title = "質問３";
                    UserInteractiveCore(next).content = content;
                    UserInteractiveCore(next).buttonOption = UserInteractiveCore.YES_OPTION;
                    break;
                case 4:
                    // 前モデル結果評価 
                    var blood:Label = new Label();
                    aType = RadioButton(getComponentByNameFrom(UserInteractiveCore(prev.target).content, "aType"));
                    blood.text = "血液型：" + aType.group.selection.label;  
                    DisplayObjectContainer(getComponentByName("userInteractive.vbox")).addChild(blood);

                    var send:Button = new Button();
                    send.label = "送信";
                    send.name = "userInteractive.send";
                    DisplayObjectContainer(getComponentByName("userInteractive.vbox")).addChild(send);
                    controller.eventBinder.addEventBindingImmediately("userInteractive.send", MouseEvent.CLICK, Class(SendAction));

            }
            return true;
        }
	 * </listing>
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class UserInteractiveCore extends BaseModel {
		
		/** メッセージ表示のみの画面モードです. */
		public static const MESSAGE_MODE:int = 1;
		
		/** メッセージ＋テキスト入力の画面モードです. */
		public static const INPUT_MODE:int = 2; 

		/** 「はい」ボタンオプションです. */
		public static const YES_OPTION:int = 1;
		
		/** 「はい」「いいえ」ボタンオプションです. */
		public static const YES_NO_OPTION:int = 3;
		
		/** 「はい」「いいえ」「キャンセル」ボタンオプションです. */
		public static const YES_NO_CANCEL_OPTION:int = 7;

		// 画面表示モード
		private var _mode:int = MESSAGE_MODE;

		// ボタン表示モード
		private var _buttonOption:int = YES_NO_OPTION;				
		
		// 表示画面インスタンス
		private var _interactiveScreen:Container;

		// メイン表示内容
		private var _content:Container;
		
		// ボタン配置領域
		private var _controlBar:ControlBar;

		// 表示メッセージ
		private var _message:String;

		// 表示ボタン
		private var _buttons:Array;
		
		// INPUT_MODE時のTextInput
		private var _textInput:TextInput;
		
		// 表示画面タイトル
		private var _title:String;

		// 表示画面の幅
		private var _screenWidth:int;
		
		// 表示画面の高さ
		private var _screenHeight:int;
		
		// メイン表示内容の左パディング
		private var _contentPaddingLeft:int = 20;
		
		// メイン表示内容の右パディング
		private var _contentPaddingRight:int = 20;
		
		// メイン表示内容の上パディング
		private var _contentPaddingTop:int = 10;
		
		// メイン表示内容の下パディング
		private var _contentPaddingBottom:int = 10;
		
		// 「はい」ボタンのラベル
		private var _yesLabel:String = "はい";

		// 「いいえ」ボタンのラベル
		private var _noLabel:String = "いいえ";
		
		// 「キャンセル」ボタンのラベル
		private var _cancelLabel:String = "キャンセル";


		/**
		 * <p>[概 要]</p>
		 * 画面表示モードです.
		 * 
		 * <p>[詳 細]</p>
		 * MESSAGE_MODE、INPUT_MODEを指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default MESSAGE_MODE
		 */
		public function get mode():int {
			return this._mode;
		}
		public function set mode(mode:int):void {
			this._mode = mode;
		}
		
		/**
		 * <p>[概 要]</p>
		 * ボタン表示モードです.
		 * 
		 * <p>[詳 細]</p>
		 * YES_OPTION、YES_NO_OPTION、YES_NO_CANCEL_OPTIONを指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default YES_NO_OPTION 
		 */
		public function get buttonOption():int {
			return this._buttonOption;
		}
		public function set buttonOption(buttonOption:int):void {
			this._buttonOption = buttonOption;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 表示画面インスタンスです.
		 * 
		 * <p>[詳 細]</p>
		 * 表示される画面全体です。<br />
		 * createScreenメソッドの戻り値が適用されます。<br />
		 * このプロパティを外部から設定するか、createScreenメソッドを
		 * オーバーライドすることで、表示する画面を変更できます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default createScreeメソッドの戻り値
		 * @see createScreen()
		 */
		public function get interactiveScreen():Container {
			return this._interactiveScreen;
		}
		public function set interactiveScreen(interactiveScreen:Container):void {
			this._interactiveScreen = interactiveScreen;
		}

		/**
		 * <p>[概 要]</p>
		 * メイン表示内容です.
		 * 
		 * <p>[詳 細]</p>
		 * interactiveScreenの中に配置されるオブジェクトです。
		 * createContentメソッドの戻り値が適用されます。<br />
		 * このプロパティを外部から設定するか、createContentメソッドを
		 * オーバーライドすることで、表示内容を変更できます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get content():Container {
			return this._content;
		}
		public function set content(content:Container):void {
			this._content = content;
		}

		/**
		 * <p>[概 要]</p>
		 * ボタン配置領域です.
		 * 
		 * <p>[詳 細]</p>
		 * interactiveScreenの下部に配置されるオブジェクトです。
		 * createControlBarメソッドの戻り値が適用されます。<br />
		 * このプロパティを外部から設定するか、createControlBarメソッドを
		 * オーバーライドすることで、ボタン配置部分を変更できます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get controlBar():ControlBar {
			return this._controlBar;
		}
		public function set controlBar(controlBar:ControlBar):void {
			this._controlBar = controlBar;
		}

		/**
		 * <p>[概 要]</p>
		 * 表示するメッセージです.
		 * 
		 * <p>[詳 細]</p>
		 * メイン表示部分に表示されるメッセージです。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get message():String {
			return this._message;
		}
		public function set message(message:String):void {
			this._message = message;
		}

		/**
		 * <p>[概 要]</p>
		 * 表示するボタンです.
		 * 
		 * <p>[詳 細]</p>
		 * このプロパティが設定されている場合、buttonOptionプロパティの
		 * 設定より優先して配置されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get buttons():Array {
			return this._buttons;
		}
		public function set buttons(buttons:Array):void {
			this._buttons = buttons;
		}

		/**
		 * <p>[概 要]</p>
		 * INPUT_MODE時のTextInputです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @see getInputText()
		 */
		public function get textInput():TextInput {
			return this._textInput;
		}
		public function set textInput(textInput:TextInput):void {
			this._textInput = textInput;
		}

		/**
		 * <p>[概 要]</p>
		 * 表示画面タイトルです.
		 * 
		 * <p>[詳 細]</p>
		 * 表示する画面のタイトル文言を指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get title():String {
			return this._title;
		}
		public function set title(title:String):void {
			this._title = title;
		}

		/**
		 * <p>[概 要]</p>
		 * 表示画面の幅です.
		 * 
		 * <p>[詳 細]</p>
		 * 設定されていない場合は内容物でパックされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get screenWidth():int {
			return this._screenWidth;
		}
		public function set screenWidth(screenWidth:int):void {
			this._screenWidth = screenWidth;
		}

		/**
		 * <p>[概 要]</p>
		 * 表示画面の高さです.
		 * 
		 * <p>[詳 細]</p>
		 * 設定されていない場合は内容物でパックされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get screenHeight():int {
			return this._screenHeight;
		}
		public function set screenHeight(screenHeight:int):void {
			this._screenHeight = screenHeight;
		}
		
		/**
		 * <p>[概 要]</p>
		 * メイン表示領域の左パディングです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default 20
		 */
		public function get contentPaddingLeft():int {
			return this._contentPaddingLeft;
		}
		public function set contentPaddingLeft(contentPaddingLeft:int):void {
			this._contentPaddingLeft = contentPaddingLeft;
		}

		/**
		 * <p>[概 要]</p>
		 * メイン表示領域の右パディングです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default 20
		 */
		public function get contentPaddingRight():int {
			return this._contentPaddingRight;
		}
		public function set contentPaddingRight(contentPaddingRight:int):void {
			this._contentPaddingRight = contentPaddingRight;
		}
		
		/**
		 * <p>[概 要]</p>
		 * メイン表示領域の上パディングです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default 10
		 */
		public function get contentPaddingTop():int {
			return this._contentPaddingTop;
		}
		public function set contentPaddingTop(contentPaddingTop:int):void {
			this._contentPaddingTop = contentPaddingTop;
		}

		/**
		 * <p>[概 要]</p>
		 * メイン表示領域の下パディングです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default 10
		 */
		public function get contentPaddingBottom():int {
			return this._contentPaddingBottom;
		}
		public function set contentPaddingBottom(contentPaddingBottom:int):void {
			this._contentPaddingBottom = contentPaddingBottom;
		}

		/**
		 * <p>[概 要]</p>
		 * 「はい」ボタンのラベルです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get yesLabel():String {
			return this._yesLabel;
		}
		public function set yesLabel(yesLabel:String):void {
			this._yesLabel = yesLabel;
		}

		/**
		 * <p>[概 要]</p>
		 * 「いいえ」ボタンのラベルです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get noLabel():String {
			return this._noLabel;
		}
		public function set noLabel(noLabel:String):void {
			this._noLabel = noLabel;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 「キャンセルボタン」のラベルです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get cancelLabel():String {
			return this._cancelLabel;
		}
		public function set cancelLabel(cancelLabel:String):void {
			this._cancelLabel = cancelLabel;
		}


		/**
		 * <p>[概 要]</p>
		 * INPUT_MODEで表示されているTextInputの値を取得します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return INPUT_MODEテキスト入力値
		 */
		 public function getInputText():String {
			var ret:String = "";
			
			if(textInput != null) {
				ret = textInput.text;
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要]</p>
		 * ユーザと対話する画面の初期化を行います.
		 *  
		 * <p>[詳 細]</p>
		 * interactiveScreenプロパティが設定されている（既に独自の画面が設定されている）
		 * 場合は処理を行いません。
		 * <br />
		 * interactiveScreenプロパティがnullの場合は、
		 * <ol>
		 *   <li>createScreenメソッド呼び出し</li>
		 *   <li>（contentプロパティがnullの場合）createContentメソッド呼び出し</li>
		 *   <li>（controllBarプロパティがnullの場合）createControllBarメソッド呼び出し</li>
		 * </ol>
		 * を行い、画面を生成します。<br />
		 * contentプロパティ、controlBarプロパティが既に設定されている場合は、それが
		 * 画面の内容物として優先的に採用されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return true
		 */
		override protected function preProc():Boolean {
			// フィールドがnullで無い場合（アクションから設定されている場合）
			// は設定済みインスタンスを優先
						
			// 表示する画面インスタンスを生成
			if(interactiveScreen == null) {
				interactiveScreen = createScreen();

				// コンテント部分インスタンスを生成			
				if(content == null) {
					content = createContent();
				}
				// 画面下部コントロールバーインスタンスを生成
				if(controlBar == null) {
					controlBar = createControlBar();
				}

				interactiveScreen.addChild(content);
				interactiveScreen.addChild(controlBar);
			}

			return true;
		}

		/**
		 * <p>[概 要]</p>
		 * ユーザと対話を行う画面インスタンスを生成します.
		 *  
		 * <p>[詳 細]</p>
		 * interactiveScreenプロパティにTitleWindowインスタンスを設定します。
		 * titleプロパティが設定されている場合はTitleWindowのタイトルとして採用されます。
		 * <br />
		 * 同様にscreenWidthプロパティ、screenHeightプロパティが設定されている場合、
		 * TitleWindowの幅、高さとして設定されます。<br />
		 * screenWidthプロパティ、screenHeightプロパティが設定されていない場合、
		 * TitleWindowのサイズは表示する内容物のサイズでパックされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		protected function createScreen():Container {
			var interactiveScreen:TitleWindow = new TitleWindow();
			interactiveScreen.title = title;
			
			// サイズ指定がされていない場合は内包コンポーネントサイズにパックされる
			if(screenWidth != 0 && screenHeight != 0) {
				interactiveScreen.width = screenWidth;
				interactiveScreen.height = screenHeight;
			}
			
			return interactiveScreen;
		}

		/**
		 * <p>[概 要]</p>
		 * ユーザとの対話内容を格納するコンテナインスタンスを生成します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		protected function createContent():Container {
			var content:VBox = new VBox();
			content.setStyle("paddingLeft", contentPaddingLeft);
			content.setStyle("paddingRight", contentPaddingRight);
			content.setStyle("paddingTop", contentPaddingTop);
			content.setStyle("paddingBottom", contentPaddingBottom);
			
			var text:Text = new Text();
			text.htmlText = message;
			content.addChild(text);

			if(mode == INPUT_MODE) {
				textInput = new TextInput();
				content.addChild(textInput);
			}
			
			return content;			
		}
		
		/**
		 * <p>[概 要]</p>
		 * 画面下部のボタン格納コントロールバーインスタンスを生成します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		protected function createControlBar():ControlBar {
			var controlBar:ControlBar = new ControlBar();

			// 画面下部コントロールバーに配置するボタンインスタンスを生成
			if(buttons == null) {
				buttons = new Array();
				if((buttonOption & YES_OPTION) == YES_OPTION) {
					var yes:Button = new Button();
					yes.label = yesLabel;
					buttons.push(yes);
				}
				if((buttonOption & YES_NO_OPTION) == YES_NO_OPTION) {
					var no:Button = new Button();
					no.label = noLabel;						
					buttons.push(no);
				}
				if((buttonOption & YES_NO_CANCEL_OPTION) == YES_NO_CANCEL_OPTION) {
					var cancel:Button = new Button();
					cancel.label = cancelLabel;						
					buttons.push(cancel);
				}
			}
			
			// 生成したボタンにイベントリスナを付与してコントロールバーに追加 
			for each(var button:Button in buttons) {
				button.addEventListener(MouseEvent.CLICK, onResult);
	    		controlBar.addChild(button);
			}

			return controlBar;						
		}
		
		/**
		 * <p>[概 要]</p>
		 * preProcで生成された画面インスタンスを表示します.
		 * 
		 * <p>[詳 細]</p>
		 * "interactive"という名前でWindowManagerに表示処理委譲します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		override protected function mainProc():void {
			ComponentManager.getInstance().addPopUp("interactive", interactiveScreen, null, true, true);
		}
		
		/**
		 * <p>[概 要]</p>
		 * ControllBarに配置されたボタンの押下イベントハンドラです.
		 * 
		 * <p>[詳 細]</p>
		 * モデル終了イベント、モデル完了イベントを発行します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event ボタン押下イベント
		 */
		public function onResult(event:Event):void {
			ComponentManager.getInstance().removePopUp("interactive");
			
			// モデル終了イベントを生成			
			var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
			// モデル終了の原因となったResultEventをModelFinishEventに格納
			modelSuccessEvent.cause = event;
			// モデル終了イベントをディスパッチしてコントローラにキャッチさせる		
			dispatchModelSuccess(modelSuccessEvent);

			// モデル完了イベントを生成			
			var modelFinishedEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
			// モデル完了の原因となったResultEventをModelFinishEventに格納
			modelFinishedEvent.cause = event;
			// モデル完了イベントをディスパッチしてコントローラにキャッチさせる		
			dispatchModelFinished(modelFinishedEvent);
		}
	}
}
