/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.android.model;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;

import jp.co.fujitsu.reffi.client.android.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.android.manager.SQLiteOpenHelperManager;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;
import android.database.sqlite.SQLiteDatabase.CursorFactory;

/**
 * <p>[概 要]</p>
 * デフォルトモデルクラスです.
 * 
 * <p>[詳 細]</p>
 * コントローラに１つもモデルを実行委譲しなかった場合
 * （アクションでモデルをリザーブしなかった場合）、コントローラによって代替採用される
 * モデルクラスです。<br>
 * イベントハンドル時の処理シーケンスを均一化する為に使用されます。
 * 
 * <p>[備 考]</p>
 *
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * @author Project Reffi
 */
public class SQLiteCore extends BaseModel {

	private SQLiteOpenHelperManager manager;
	
	private SQLiteDatabase database;
	
	private String[] onCreateSql;
	
	private String[] onUpdateSql;

	private String databaseName;
	
	private CursorFactory cursorFactory;
	
	private int databaseVersion = 1;
	
	private List<SQLInfo> sqls = new ArrayList<SQLInfo>();
	
	private boolean autoClose;
	
	private int sqlIndex;
	
	private boolean transaction;
	
	
	public SQLiteOpenHelperManager getManager() {
		return manager;
	}

	public void setManager(SQLiteOpenHelperManager manager) {
		this.manager = manager;
	}

	public SQLiteDatabase getDatabase() {
		return database;
	}

	public void setDatabase(SQLiteDatabase database) {
		this.database = database;
	}

	public String[] getOnCreateSql() {
		return onCreateSql;
	}

	public void setOnCreateSql(String[] onCreateSql) {
		this.onCreateSql = onCreateSql;
	}

	public String[] getOnUpdateSql() {
		return onUpdateSql;
	}

	public void setOnUpdateSql(String[] onUpdateSql) {
		this.onUpdateSql = onUpdateSql;
	}

	public String getDatabaseName() {
		if(databaseName == null) {
			databaseName = getController().getClientConfig().getDefaultDatabaseName();
		}
		return databaseName;
	}

	public void setDatabaseName(String databaseName) {
		this.databaseName = databaseName;
	}
	
	public CursorFactory getCursorFactory() {
		return cursorFactory;
	}

	public void setCursorFactory(CursorFactory cursorFactory) {
		this.cursorFactory = cursorFactory;
	}

	public int getDatabaseVersion() {
		if(databaseVersion == 0) {
			databaseVersion = getController().getClientConfig().getDefaultDatabaseVersion();
		}
		return databaseVersion;
	}

	public void setDatabaseVersion(int databaseVersion) {
		this.databaseVersion = databaseVersion;
	}

	public List<SQLInfo> getSqls() {
		return sqls;
	}

	public void setSqls(List<SQLInfo> sqls) {
		this.sqls = sqls;
	}

	public boolean isAutoClose() {
		return autoClose;
	}

	public void setAutoClose(boolean autoClose) {
		this.autoClose = autoClose;
	}
	
	public int getSqlIndex() {
		return sqlIndex;
	}

	public void setSqlIndex(int sqlIndex) {
		this.sqlIndex = sqlIndex;
	}
	
	public boolean isTransaction() {
		return transaction;
	}

	public void setTransaction(boolean transaction) {
		this.transaction = transaction;
	}

	public void addSql(String sql, Object[] args) {
		SQLInfo sqlInfo = new SQLInfo(sql, args);
		getSqls().add(sqlInfo);
	}
	
	
	
	@Override
	protected boolean preproc() throws Exception {
		
		this.manager = SQLiteOpenHelperManager.getInstance();
		manager.open(this);

		// DMLが登録されていなければ以降の処理は中止
		if(getSqls().size() == 0) {
			return false;
		}else{
			registSqls();
			return true;
		}
	}
	
	@Override
	protected void mainproc() throws Exception {
		SQLiteOpenHelper helper = getManager().getHelper(getDatabaseName());
		setDatabase(helper.getWritableDatabase());

		if(isTransaction()){
			getDatabase().beginTransaction();
		}

		for(int i=0; i<getSqls().size(); i++) {
			SQLInfo sqlInfo = getSqls().get(i);
			
			preExecute(getSqlIndex(), sqlInfo);

			Object ret = executeSql(sqlInfo);
			setResult(ret);
			
			postExecute(getSqlIndex(), sqlInfo, ret);
			
			ModelProcessEvent success = new ModelProcessEvent(this);
			success.setResult(getResult());
			fireModelSuccess(success);
			
			setSqlIndex(getSqlIndex()+1);
		}
		
		if(isTransaction()) {
			getDatabase().setTransactionSuccessful();
		}

		fireModelFinished(new ModelProcessEvent(this));
	}
	
	@Override
	protected void finalproc() {
		if(isTransaction()) {
			getDatabase().endTransaction();
		}
	}
	
	@Override
	public void done() {
		if(isAutoClose()) {
			if(getResult() instanceof Cursor && 
				((Cursor)getResult()).isClosed() == false) 
			{
				((Cursor)getResult()).close();
			}
			
			getManager().close(this);
		}
	}

	protected void bindArgs(SQLiteStatement statement, Object[] args) {
		for(int i=0; i<args.length; i++) {
            DatabaseUtils.bindObjectToProgram(statement, i+1, args[i]);
		}
	}

	protected void registSqls() {
	}
	
	protected void preExecute(int sqlIndex, SQLInfo sqlInfo) {

	}

	public Object executeSql(SQLInfo sqlInfo) {
		Object ret = null;
		
		String sql = sqlInfo.getSql().trim();
		
		if(sql.toLowerCase().startsWith("select")) {
			String[] selectionArgs = null;
			if(sqlInfo.getArgs() != null) {
				Object[] args = sqlInfo.getArgs();
				selectionArgs = (String[])Array.newInstance(String.class, args.length);
				System.arraycopy(args, 0, selectionArgs, 0, args.length);
			}
			Cursor cursor = getDatabase().rawQuery(sql, selectionArgs);
			ret = cursor;
		}else if(sql.toLowerCase().startsWith("update") || 
				  sql.toLowerCase().startsWith("delete"))
		{
			SQLiteStatement statement = getDatabase().compileStatement(sql);
			bindArgs(statement, sqlInfo.getArgs());
			int affectedRowCount = statement.executeUpdateDelete();
			ret = affectedRowCount;
		}else if(sql.toLowerCase().startsWith("insert")) {
			SQLiteStatement statement = getDatabase().compileStatement(sql);
			bindArgs(statement, sqlInfo.getArgs());
			long rowId = statement.executeInsert();
			ret = rowId;
		}else{
			getDatabase().execSQL(sql, sqlInfo.getArgs());
		}

		return ret;
	}

	protected  void postExecute(int sqlIndex, SQLInfo sqlInfo, Object result) {
	}
	
	public class SQLInfo {
		
		private String sql;
		
		private Object[] args;
		
		
		public String getSql() {
			return sql;
		}

		public void setSql(String sql) {
			this.sql = sql;
		}

		public Object[] getArgs() {
			return args;
		}

		public void setArgs(Object[] args) {
			this.args = args;
		}

		public SQLInfo(String sql, Object[] args) {
			this.sql = sql;
			this.args = args;
		}
	}
}
