package jp.co.fujitsu.reffi.client.android.manager;

import java.util.HashMap;
import java.util.Map;

import jp.co.fujitsu.reffi.client.android.model.SQLiteCore;
import jp.co.fujitsu.reffi.client.android.model.TimerProcessCore;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

/**
 * <p>[概 要] </p>
 * 
 * <p>[詳 細] </p>
 * 
 * 
 * <p>[備 考] </p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class SQLiteOpenHelperManager {
	
	public static final String ON_MEMORY = "ON_MEMORY";
	
	/** このマネージャのシングルトンインスタンスです。 */
	private static SQLiteOpenHelperManager instance;
	
	/** TimerProcessCoreを管理するマップオブジェクトです。 */
	private Map<String, SQLiteOpenHelperImpl> helpers = new HashMap<String, SQLiteOpenHelperImpl>();
	
	
	/**
	 * <p>[概 要] </p>
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return TimerProcessCoreを管理するマップオブジェクト
	 */
	public Map<String, SQLiteOpenHelperImpl> getHelpers() {
		return helpers;
	}

	/**
	 * <p>[概 要] </p>
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timers TimerProcessCoreを管理するマップオブジェクト
	 */
	public void setHelpers(Map<String, SQLiteOpenHelperImpl> helpers) {
		this.helpers = helpers;
	}

	/**
	 * <p>[概 要] </p>
	 * プライベートコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * インスタンス生成には{@link #getInstance()}を使用します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	private SQLiteOpenHelperManager() {
	}
	
	/**
	 * <p>[概 要] </p>
	 * シングルトンインスタンス取得メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * TimerProcessCoreManagerのインスタンスはJVM内でユニークです。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return マネージャのシングルトンインスタンス
	 */
	public static SQLiteOpenHelperManager getInstance() {
		if(instance == null) {
			instance = new SQLiteOpenHelperManager();
		}
		
		return instance;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数timerIdを持つTimerProcessCoreが既に管理されているか調べます。
	 * 
	 * <p>[詳 細] </p>
	 * timers管理マップに引数timerIdがキーとして登録されているかどうか
	 * 調べて返却します。<br>
	 * trueの場合、タイマーをstartしているTimerProcessCoreが既に存在します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerId タイマーの識別子
	 * @return true : 既登録、false : 未登録
	 */
	public boolean isRegist(String dbName) {
		if(dbName == null) {
			dbName = ON_MEMORY;
		}
		return getHelpers().containsKey(dbName);
	}
	
	public SQLiteOpenHelper getHelper(String dbName) {
		if(dbName == null) {
			dbName = ON_MEMORY;
		}
		return getHelpers().get(dbName);
	}
	
	/**
	 * <p>[概 要] </p>
	 * タイマー処理を開始します。
	 * 
	 * <p>[詳 細] </p>
	 * 既に同名のtimerIdを持つtimerProcessCoreが存在する場合はfalseを返却します。<br>
	 * 
	 * ScheduledThreadPoolExecuterに定期処理を委譲します。<br>
	 * Executerの実行にはTimerProcessCoreに設定されている情報が使用されます。<br>
	 * 
	 * <ol>
	 *   <li>初期遅延：{@link TimerProcessCore#getInitialDelay()}</li>
	 *   <li>実行間隔：{@link TimerProcessCore#getPeriod()}</li>
	 *   <li>終了時間：{@link TimerProcessCore#getStopLater()}（0の場合は自動停止無し）</li>
	 *   <li>実行処理：{@link TimerProcessCore#getIntervalAction()} （nullの場合は成功イベント発行のみ）</li>
	 * </ol>
	 *
	 * タイマー処理を開始したTimerProcessCoreは、timerIdをキーにtimersフィールドで管理開始されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerProcessCore 開始するタイマー情報を持った機能モデルインスタンス
	 * @return true : タイマー開始成功、 false : 失敗
	 */
	public SQLiteOpenHelper open(SQLiteCore sqliteCore) {
		SQLiteOpenHelperImpl ret = null;
		String dbName = sqliteCore.getDatabaseName();
		
		if(isRegist(sqliteCore.getDatabaseName())) {
			ret = getHelpers().get(dbName);
		}else{
			ret = new SQLiteOpenHelperImpl(sqliteCore);
			if(dbName == null) {
				dbName = ON_MEMORY;
			}
			getHelpers().put(dbName, ret);
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerId タイマーの識別子
	 * @return true : 停止成功、false : 停止失敗
	 */
	public boolean close(SQLiteCore sqliteCore) {
		
		String dbName = sqliteCore.getDatabaseName();

		boolean ret = close(dbName);

		return ret;
	}
	
	public boolean close(String dbName) {
		if(!isRegist(dbName)) {
			return false;
		}
		
		SQLiteOpenHelperImpl helper = (SQLiteOpenHelperImpl)getHelper(dbName);
		
		SQLiteDatabase readableDatabase = helper.getReadDB();
		if(readableDatabase != null && readableDatabase.isOpen()) {
			readableDatabase.close();
		}

		SQLiteDatabase writableDatabase = helper.getWriteDB();
		if(writableDatabase != null && writableDatabase.isOpen()) {
			writableDatabase.close();
		}

		getHelpers().remove(dbName);

		return true;
	}
	
	public class SQLiteOpenHelperImpl extends SQLiteOpenHelper {
		
		private SQLiteCore sqliteCore;

		private SQLiteDatabase readDB;
		
		private SQLiteDatabase writeDB;
		

		protected SQLiteDatabase getReadDB() {
			return readDB;
		}

		protected SQLiteDatabase getWriteDB() {
			return writeDB;
		}
		
		public SQLiteOpenHelperImpl(SQLiteCore sqliteCore) {			
			super(
				sqliteCore.getParameterMapping().getContext(), 
				sqliteCore.getDatabaseName(), 
				sqliteCore.getCursorFactory(), 
				sqliteCore.getDatabaseVersion()
			);
			
			this.sqliteCore = sqliteCore;
		}

		@Override
		public SQLiteDatabase getReadableDatabase() {
			if(this.readDB == null) {
				SQLiteDatabase readableDatabase = super.getReadableDatabase();
				this.readDB = readableDatabase;
			}
			
			return this.readDB;
		}

		@Override
		public SQLiteDatabase getWritableDatabase() {
			if(this.writeDB == null) {
				SQLiteDatabase writableDatabase = super.getWritableDatabase();
				this.writeDB = writableDatabase;
			}
			
			return this.writeDB;
		}

		@Override
		public void onCreate(SQLiteDatabase db) {
			db.beginTransaction();
			if(this.sqliteCore.getOnCreateSql() != null) {
				for(String ddl : this.sqliteCore.getOnCreateSql()) {
					db.execSQL(ddl);
				}
			}
			db.setTransactionSuccessful();
			db.endTransaction();
		}

		@Override
		public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
			db.beginTransaction();
			if(this.sqliteCore.getOnUpdateSql() != null) {
				for(String ddl : this.sqliteCore.getOnUpdateSql()) {
					db.execSQL(ddl);
				}
			}
			db.setTransactionSuccessful();
			db.endTransaction();
		}
	}

}
