/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.android.controller;

//import java.awt.Component;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.fujitsu.reffi.client.android.action.AbstractAction;
import android.content.Context;
//import jp.co.fujitsu.reffi.client.swing.util.ComponentSearchUtil;

/**
 * <p>[概 要] </p>
 * イベント紐付け情報保持クラスです。
 * 
 * <p>[詳 細] </p>
 * {@link BaseController#bind(EventBinder)}によるコンポーネント、イベントタイプ、アクションクラス
 * の紐付け情報を、bindInfoフィールドMapに保持します。<br>
 * bindInfoオブジェクトは以下のような構造を持ちます。
 * <pre>
 * [
 * コンポーネント名1 = 
 *     イベントリスナクラス型 = 
 *         [イベントタイプ1(イベントハンドラメソッド名) = アクションクラス, 
 *          イベントタイプ2(イベントハンドラメソッド名) = アクションクラス],
 * コンポーネント名2 =
 *     イベントリスナクラス型 =
 *         [イベントタイプ1(イベントハンドラメソッド名) = アクションクラス] 
 * ]
 * </pre>
 * 
 * <pre class="samplecode">
 *		eventBinder.addEventBinding("chatFrame.jbEnterChannel", ActionListener.class, "actionPerformed", EnterChannelAction.class);
 *		eventBinder.addEventBinding("chatFrame.jbRemarkSend", ActionListener.class, "actionPerformed", RemarkSendAction.class);
 *		eventBinder.addEventBinding("chatFrame", WindowListener.class, "windowClosing", ChatFrameCloseAction.class);
 * </pre>
 * 
 * 上記の場合、以下のようになります。
 * <pre>
 * [
 * "chatFrame.jbEnterChannel" = 
 *     ActionListener.class =
 *         ["actionPerformed" = EnterChannelAction.class]
 * "chatFrame.jbRemarkSend" =
 *     ActionListener.class =
 *         ["actionPerformed" = RemarkSendAction.class] 
 * ]
 * 
 * </pre>
 * 
 * <p>[備 考] </p>
 * アプリケーション動作中に紐付け情報を追加する場合、以下のようにEventBinderオブジェクト
 * を取得してaddEventBinding、又はaddEventBindingImmediatelyをコールします。
 * 
 * <pre class="samplecode">
 * 	getController().getEventBinder().addEventBinding("inputFormFrame.jtfUserId", FocusListener.class, "focusLost", UserIdFocusLostAction.class);
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class EventBinder {

	/** コンポーネント名、リスナタイプ、eventタイプ、Actionクラスの紐付け情報を保持するオブジェクトです。 */
	private Map<String, Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>>> bindInfo;

	/** コントローラオブジェクトです。 */
	private AbstractController controller;
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報保持オブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return イベント紐付け情報
	 */
	public Map<String, Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>>> getBindInfo() {
		return bindInfo;
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報保持オブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドを、引数bindInfoで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param bindInfo イベント紐付け情報
	 */
	public void setBindInfo(Map<String, Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>>> bindInfo) {
		this.bindInfo = bindInfo;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コントローラオブジェクト
	 */
	public AbstractController getController() {
		return controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを引数controllerで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラオブジェクト
	 */
	public void setController(AbstractController controller) {
		this.controller = controller;
	}

	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドインスタンスを生成します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public EventBinder(AbstractController controller) {
		this.controller = controller;
		this.bindInfo = new HashMap<String, Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>>>();
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数viewIdが既に登録されている場合、viewId用のMapを取り出し、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * viewIdが未登録の場合、viewId用のMapを新規に生成して、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * 登録された情報はイベント紐付け情報保持オブジェクト（bindInfoフィールド）に保持され、
	 * コントローラのコンポーネント挿抜ハンドラによって参照されます。
	 * 
	 * <p>[備 考] </p>
	 * 登録された紐付け情報が反映されるのは、viewIdをname属性値として持つ
	 * コンポーネントが画面追加されたタイミングです。
	 * 
	 * @param viewId
	 * @param listenerFqcn
	 * @param eventType
	 * @param actionFqcn
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	public void addEventBinding(String name, String listenerFqcn, String eventType, String actionFqcn) throws Exception {
		Class<? extends Object> listenerType = (Class<? extends Object>)Class.forName(listenerFqcn);
		
		addEventBinding(name, listenerType, eventType, actionFqcn);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数viewIdが既に登録されている場合、viewId用のMapを取り出し、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * viewIdが未登録の場合、viewId用のMapを新規に生成して、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * 登録された情報はイベント紐付け情報保持オブジェクト（bindInfoフィールド）に保持され、
	 * コントローラのコンポーネント挿抜ハンドラによって参照されます。
	 * 
	 * <p>[備 考] </p>
	 * 登録された紐付け情報が反映されるのは、viewIdをname属性値として持つ
	 * コンポーネントが画面追加されたタイミングです。
	 * 
	 * @param viewId
	 * @param listenerType
	 * @param eventType
	 * @param actionFqcn
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	public void addEventBinding(String name, Class<? extends Object> listenerType, String eventType, String actionFqcn) throws Exception {
		Class<? extends AbstractAction> actionClass = (Class<? extends AbstractAction>)Class.forName(actionFqcn);
		
		addEventBinding(name, listenerType, eventType, actionClass);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数viewIdが既に登録されている場合、viewId用のMapを取り出し、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * viewIdが未登録の場合、viewId用のMapを新規に生成して、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * 登録された情報はイベント紐付け情報保持オブジェクト（bindInfoフィールド）に保持され、
	 * コントローラのコンポーネント挿抜ハンドラによって参照されます。
	 * 
	 * <p>[備 考] </p>
	 * 登録された紐付け情報が反映されるのは、viewIdをname属性値として持つ
	 * コンポーネントが画面追加されたタイミングです。
	 * 
	 * @param viewId
	 * @param listenerType
	 * @param eventType
	 * @param actionClass
	 */
	public void addEventBinding(String name, Class<? extends Object> listenerType, String eventType, Class<? extends AbstractAction> actionClass) {

		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap = null; 

		if(this.bindInfo.containsKey(name)) {
			listenerTypeMap = this.bindInfo.get(name);
		} else {
			listenerTypeMap = new HashMap<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>>(); 
			this.bindInfo.put(name, listenerTypeMap);
		}
		
		Map<String, Class<? extends AbstractAction>> eventTypeMap = null;
		
		if(listenerTypeMap.containsKey(listenerType)) {
			eventTypeMap = listenerTypeMap.get(listenerType);
		} else {
			eventTypeMap = new HashMap<String, Class<? extends AbstractAction>>();
			listenerTypeMap.put(listenerType, eventTypeMap);
		}

		eventTypeMap.put(eventType, actionClass);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数viewIdに紐付いているイベントリスナタイプマップを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * 下記構造のMapを返却します。
	 * <pre>
	 * [
	 *   イベントリスナ型 = [イベントハンドラメソッド名 = アクションクラス],
	 *   イベントリスナ型 = [イベントハンドラメソッド名 = アクションクラス],
	 *   イベントリスナ型 = [イベントハンドラメソッド名 = アクションクラス],
	 * ]
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param viewId イベントリスナタイプマップを取得するコンポーネント名
	 * @return イベントリスナタイプマップ
	 */
	public Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> getListenerTypeMap(String name) {
		return this.bindInfo.get(name);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数viewIdに紐付いているイベントリスナのリストを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * viewIdに対して以下のリスナタイプマップが紐付けられている場合、
	 * <pre>
	 * [
	 *   イベントリスナ型1 = [イベントハンドラメソッド名 = アクションクラス],
	 *   イベントリスナ型2 = [イベントハンドラメソッド名 = アクションクラス],
	 *   イベントリスナ型3 = [イベントハンドラメソッド名 = アクションクラス],
	 * ]
	 * </pre>
	 * 
	 * 以下のリストを返却します。
	 * <pre>
	 * [イベントリスナ型1、イベントリスナ型2、イベントリスナ型3]
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 *  
	 * @param viewId イベントリスナリストを取得するコンポーネント名
	 * @return イベントリスナリスト
	 */
	public List<Class<? extends Object>> getListenerTypes(String name) {
		List<Class<? extends Object>> ret = new ArrayList<Class<? extends Object>>();
		
		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap = 
			this.bindInfo.get(name);
		
		if(listenerTypeMap != null) {
			Set<Class<? extends Object>> keys = listenerTypeMap.keySet();
			for(Iterator<Class<? extends Object>> it = keys.iterator(); it.hasNext();) {
				Class<? extends Object> listener = it.next();
				ret.add(listener);
			}
		}
		
		return ret;	
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数viewId、listenerTypeに紐付いているイベントタイプリストを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * viewIdに対して以下のリスナタイプマップが紐付けられている場合、
	 * <pre>
	 * [
	 *   イベントリスナ型1 = [イベントハンドラメソッド名1 = アクションクラス,
	 *                       イベントハンドラメソッド名2 = アクションクラス],
	 *   イベントリスナ型2 = [イベントハンドラメソッド名2 = アクションクラス],
	 *   イベントリスナ型3 = [イベントハンドラメソッド名3 = アクションクラス],
	 * ]
	 * </pre>
	 * 
	 * イベントリスナ型1に対して以下のリストを返却します。
	 * <pre>
	 * [イベントハンドラメソッド名1、イベントハンドラメソッド名2]
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 *  
	 * @param viewId イベントタイプリストを取得するコンポーネント名
	 * @param listenerType イベントタイプリストを取得するイベントリスナタイプ
	 * @return イベントタイプリスト
	 */
	public List<String> getEventTypes(String name, Class<? extends Object> listenerType) {
		List<String> ret = new ArrayList<String>();
		
		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap = 
			this.bindInfo.get(name);
		
		if(listenerTypeMap != null) {
			Map<String, Class<? extends AbstractAction>> eventTypeMap = listenerTypeMap.get(listenerType);
			if(eventTypeMap != null) {
				Set<String> keys = eventTypeMap.keySet();
				for(Iterator<String> it = keys.iterator(); it.hasNext();) {
					String eventType = it.next();
					ret.add(eventType);
				}
			}
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数viewId、listenerType、eventTypeに紐付いているイベントタイプリストを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * viewIdに対して以下のリスナタイプマップが紐付けられている場合、
	 * <pre>
	 * [
	 *   イベントリスナ型1 = [イベントハンドラメソッド名1 = アクションクラス1,
	 *                       イベントハンドラメソッド名2 = アクションクラス2],
	 *   イベントリスナ型2 = [イベントハンドラメソッド名2 = アクションクラス3],
	 *   イベントリスナ型3 = [イベントハンドラメソッド名3 = アクションクラス4],
	 * ]
	 * </pre>
	 * 
	 * イベントリスナ型1、イベントハンドラメソッド名1に対してアクションクラス1を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param viewId アクションクラスを取得するコンポーネント名
	 * @param listenerType アクションクラスを取得するイベントリスナタイプ
	 * @param eventType アクションクラスを取得するイベントタイプ
	 * @return アクションクラス
	 */
	public Class<? extends AbstractAction> getActionClass(String name, 
															Class<? extends Object> listenerType, 
															String eventType)
	{
		Class<? extends AbstractAction> ret = null;
		
		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap = 
			this.bindInfo.get(name);
		
		if(listenerTypeMap != null) {
			Map<String, Class<? extends AbstractAction>> eventTypeMap = listenerTypeMap.get(listenerType);
			if(eventTypeMap != null) {
				ret = eventTypeMap.get(eventType);
			}
		}		
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数viewIdに対してイベント紐付けが行われているかどうか調べます。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドにviewIdがキーとして含まれているかどうか調べます。
	 * 
	 * <p>[備 考] </p>
	 *  
	 * @param viewId イベント紐付けが行われているかどうか調べるコンポーネント名
	 * @return true : 紐付け有り、false : 紐付け無し
	 */
	public boolean isEventBinding(String name) {
		return this.bindInfo.containsKey(name);
	}
	
//	/**
//	 * <p>[概 要] </p>
//	 * イベント紐付け情報追加メソッドです。
//	 * 
//	 * <p>[詳 細] </p>
//	 * {@link #addEventBindingImmediately(Component, Class, String, Class)}メソッドに
//	 * 処理委譲します。
//	 * 引数viewIdを名前として持つコンポーネントがメモリ上に存在しない場合、
//	 * 処理は行われません。
//	 * 
//	 * <p>[備 考] </p>
//	 * 
//	 * @param viewId
//	 * @param listenerType
//	 * @param eventType
//	 * @param actionClass
//	 */
//	public void addEventBindingImmediately(String name, Class<? extends Object> listenerType, 
//											String eventType,
//											Class<? extends AbstractAction> actionClass)
//	{
////		Component c = ComponentSearchUtil.searchComponentByNameFromAllWindow(viewId);
////		if(c == null) {
////			return;
////		}
////		
////		addEventBindingImmediately(c, listenerType, eventType, actionClass);
//	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * {@link #addEventBinding(String, Class, String, Class)}オーバーロードメソッドを
	 * 呼び出して、イベント紐付け情報保持オブジェクト（bindInfo）に追加します。<br>
	 * 追加後、ui Documentを取得して、追加された紐付け情報を即座に反映します。
	 * 
	 * <p>[備 考] </p>
	 * {@link #addEventBinding(String, Class, String, Class)}と違い、
	 * 登録された情報が画面コンポーネントにイベントリスナ追加されるのはメソッド呼び出し直後です。
	 * 
	 * @param view
	 * @param listenerType
	 * @param eventType
	 * @param actionClass
	 */
	public void addEventBindingImmediately(Context context,
											Object viewOrActivity,
											String name, 
											Class<? extends Object> listenerType, 
											String eventType,
											Class<? extends AbstractAction> actionClass) throws Exception
	{
		addEventBinding(name, listenerType, eventType, actionClass);
		
		List<String> eventTypes = new ArrayList<String>();
		eventTypes.add(eventType);
		this.controller.addListener(context, viewOrActivity, listenerType, eventTypes);
	}
	
	/**
	 * <p>[概 要]</p>
	 * イベント紐付け情報削除メソッドです.
	 *  
	 * <p>[詳 細]</p>
	 * 保持されているコンポーネン紐付け情報から、引数viewId
	 * に紐付いている全リスナを削除します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param viewId
	 */
	public void removeEventBinding(int viewId) {
		this.bindInfo.remove(viewId);
	}

	/**
	 * <p>[概 要]</p>
	 * イベント紐付け情報削除メソッドです.
	 *  
	 * <p>[詳 細]</p>
	 * 保持されているコンポーネン紐付け情報から、引数viewId
	 * に紐付いている引数listenerTypeを削除します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param viewId
	 * @param listenerType
	 */
	public void removeEventBinding(int viewId, Class<? extends Object> listenerType) {
		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap =
			this.bindInfo.get(viewId);

		listenerTypeMap.remove(listenerType);
	}

	/**
	 * <p>[概 要]</p>
	 * イベント紐付け情報削除メソッドです.
	 *  
	 * <p>[詳 細]</p>
	 * 保持されているコンポーネン紐付け情報から、引数viewIdの
	 * 引数listenerTypeに紐付いているeventTypeを削除します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param viewId
	 * @param listenerType
	 * @param eventType
	 */
	public void removeEventBinding(int viewId, 
									Class<? extends Object> listenerType, 
									String eventType)
	{
		Map<Class<? extends Object>, Map<String, Class<? extends AbstractAction>>> listenerTypeMap =
			this.bindInfo.get(viewId);
		
		Map<String, Class<? extends AbstractAction>> eventTypeMap = listenerTypeMap.get(listenerType);
		eventTypeMap.remove(eventType);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報削除メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * {@link #removeEventBinding(String, Class, String)}メソッドを
	 * 呼び出して、イベント紐付け情報保持オブジェクト（bindInfo）から
	 * 引数情報を元に削除を行います。<br>
	 * 削除後、引数viewから引数listenerを即座に削除反映します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param view
	 * @param listenerType
	 * @param eventType
	 * @param listener
	 */
//	public void removeEventBindingImmediately(Component view, 
//												Class listenerType, 
//												String eventType,
//												EventListener listener) 
//		throws SecurityException, NoSuchFieldException, IllegalArgumentException, IllegalAccessException
//	{
//		int viewId = view.getName();
//		removeEventBinding(viewId, listenerType, eventType);
//		
//		Field listenerListField = JComponent.class.getDeclaredField("listenerList");
//		listenerListField.setAccessible(true);
//
//		EventListenerList listenerList = (EventListenerList)listenerListField.get(view);
//		
//		listenerList.remove(listenerType, listener);
//	}
}
