/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.android.controller;

//import java.awt.AWTEvent;
//import java.awt.Component;
//import java.awt.Container;
//import java.awt.Window;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.EventListener;
import java.util.EventObject;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.co.fujitsu.reffi.client.android.action.AbstractAction;
import jp.co.fujitsu.reffi.client.android.controller.attachment.ListenerAttachment;
import jp.co.fujitsu.reffi.client.android.controller.attachment.ListenerAttachmentFactory;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.util.Log;
import android.view.View;

//import javax.swing.SwingUtilities;

//import jp.co.fujitsu.reffi.client.android.action.AbstractAction;
//import jp.co.fujitsu.reffi.client.swing.listener.ContainerListenerImpl;
//import jp.co.fujitsu.reffi.client.swing.listener.WindowOpenShutListener;
//import jp.co.fujitsu.reffi.client.swing.manager.WindowManager;

/**
 * <p>[概 要] </p>
 * アプリケーションを制御する抽象コントローラです。
 * 
 * <p>[詳 細] </p>
 * JVM上のウィンドウレベルコンポーネントの開閉を監視して、
 * <ul>
 *   <li>コンポーネントに対するイベントリスナ追加</li>
 *   <li>イベント発生時のハンドラ集約</li>
 * </ul>
 * を行います。
 * <p>
 * 
 * <b>コンポーネントに対するイベントリスナ追加</b>
 * <p>
 * 
 * EventBinderフィールドがコンポーネントとアクションをイベントタイプ毎に紐付け管理します。<br>
 * この情報は具象コントローラを作成し、{@link #bind(EventBinder)}メソッドをオーバーライド、
 * 実装することで蓄積されます。<br>
 * ウィンドウコンポーネントの画面表示イベントを監視して、包含するコンポーネントを走査、
 * EventBinderに登録されているコンポーネント名であれば
 * 「{@link #handlerFacade(EventObject, Class, String, EventListener)}
 * をコールするイベントリスナプロキシ」をそのコンポーネントに対して設定します。
 * <p>
 * 
 * <b>イベント発生時のハンドラ集約</b>
 * <p>
 *
 * 上記イベントリスナの自動追加により、Reffi使用アプリケーションのイベントハンドラは全て、
 * このクラスのhandlerFacadeメソッドになります。<br>
 * Swingコーディングにおけるイベントハンドラ追加の常套手段である、
 * <pre>
 * 	jbutton.addActionListener(new ActionListener(){
 * 		public void actionPerformed(ActionEvent evt) {
 * 			....
 *		}
 * 	});
 * </pre>
 * や、
 * 
 * <pre>
 * 	JButton button = new JButton(new LoginAction());
 * </pre>
 * といった画面クラスで行うイベント追加と、イベントリスナ処理の実装は不要です。
 * <p>
 *
 * <p>[備 考] </p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public abstract class AbstractController extends Application implements Controller {
	
	/** イベント紐付けオブジェクトです。 */
	private EventBinder eventBinder;
	
	/** アプリ起動～終了まで存在するデータ保存領域です。 */
	private Map<Object, Object> permanent;

	/** フレームワーク挙動情報を保持するオブジェクトです。 */
	private ClientConfig clientConfig;

	private Activity frontActivity;
	
	public static AbstractController instance;

	
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付けオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * eventBinderフィールドオブジェクトを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return イベント紐付けオブジェクト
	 */
	public EventBinder getEventBinder() {
		return eventBinder;
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付けオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * eventBinderフィールドオブジェクトを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventBinder イベント紐付けオブジェクト
	 */
	public void setEventBinder(EventBinder eventBinder) {
		this.eventBinder = eventBinder;
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * permanentフィールドオブジェトを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return アプリ起動～終了まで存在するデータ保存領域
	 */
	public Map<Object, Object> getPermanent() {
		return this.permanent;
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * permanentフィールドオブジェクトを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param permanent アプリ起動～終了まで存在するデータ保存領域
	 */
	public void setPermanent(Map<Object, Object> Permanent) {
		this.permanent = Permanent;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワーク挙動情報を保持するオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * clientConfigフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return フレームワーク挙動情報を保持するオブジェクト
	 */
	public ClientConfig getClientConfig() {
		return clientConfig;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワーク挙動情報を保持するオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * clientConfigフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientConfig フレームワーク挙動情報を保持するオブジェクト
	 */
	public void setClientConfig(ClientConfig clientConfig) {
		this.clientConfig = clientConfig;
	}
	
	

	public Activity getFrontActivity() {
		return frontActivity;
	}

	public void setFrontActivity(Activity frontActivity) {
		this.frontActivity = frontActivity;
	}

	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラの初期化を行います。
	 * <p>
	 * 	<UL>
	 * 		<LI>イベント紐付け情報オブジェクトの生成</LI>
	 * 		<LI>アプリ起動～終了まで存在するデータ保存領域の生成</LI>
	 * 		<LI>reffi-client設定情報保持領域の生成</LI>
	 * 		<LI>エレメントのエラー前クローン保存領域</LI>
	 * 	</UL>
	 * <p>
	 * を行った後、イベント紐付け登録読込みの為、{@link #bind(EventBinder)}メソッドを
	 * テンプレートコールします。
	 * 
	 * <p>
	 * イベント紐付け情報が正常に読み込まれた後、アプリケーション初期化の為のinitialize
	 * がコールされます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public AbstractController() {
		setEventBinder(new EventBinder(this));
		setClientConfig(new ClientConfig());
		setPermanent(new HashMap<Object, Object>());
		
		bind(getEventBinder());
		
		initialize(getClientConfig());
		postInitialize(getClientConfig());
		
		Runtime.getRuntime().addShutdownHook(new Thread(){
			public void run() {
				shutdown();
			}
		});
		
		instance = this;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 初期化処理が記述可能なメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * Document、ClientSessionが生成されるタイミングでテンプレートコールされます。
	 * デフォルトの処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * 業務固有の初期化処理が必要な場合は、具象コントローラ内でこのメソッドを
	 * オーバーライドして下さい。<br>
	 * 又、Reffiの挙動設定をこのタイミングで設定することが出来ます。<p>
	 * 
	 * <b>使用例）</b><br>
	 * <pre class="samplecode">
	 *	protected void initialize(ClientConfig config){
	 *		// デフォルトJMS接続環境を設定
	 *		Hashtable<String, String> jmsEnvironment = new Hashtable<String, String>();
	 *		jmsEnvironment.put(Context.INITIAL_CONTEXT_FACTORY, "org.jnp.interfaces.NamingContextFactory");
	 *		jmsEnvironment.put(Context.PROVIDER_URL, "localhost:1099");
	 *		jmsEnvironment.put("java.naming.rmi.security.manager", "yes");
	 *		jmsEnvironment.put(Context.URL_PKG_PREFIXES, "org.jboss.naming");
	 *		config.setDefaultJmsEnvironment(jmsEnvironment);
	 *		
	 *		// デフォルトEJB接続環境を設定
	 *		Hashtable<String, String> ejbEnvironment = new Hashtable<String, String>();
	 *		ejbEnvironment.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.naming.NamingContextFactory");
	 *		ejbEnvironment.put(Context.PROVIDER_URL, "jnp://localhost:1099");
	 *		config.setDefaultEjbEnvironment(ejbEnvironment);
	 *		
	 *		// デフォルトDatabase接続環境を設定
	 *		Hashtable<String, String> dbEnvironment = new Hashtable<String, String>();
	 *		dbEnvironment.put(DatabaseCore.DB_DRIVER_FQCN, "org.postgresql.Driver");
	 *		dbEnvironment.put(DatabaseCore.DB_URL, "jdbc:postgresql://localhost:5432/RFD");
	 *		dbEnvironment.put(DatabaseCore.DB_USER, "nakanishi");
	 *		dbEnvironment.put(DatabaseCore.DB_PASSWORD, "shingon");
	 *		dbEnvironment.put(DatabaseCore.DB_AUTO_COMMIT, "true");
	 *		config.setDefaultDatabaseEnvironment(dbEnvironment);
	 *		
	 *		Hashtable<String, String> httpEnvironment = new Hashtable<String, String>();
	 *		httpEnvironment.put(HTTPRequestCore.HTTP_URL_PREFIX, "http://localhost:8080/RFDforSwingWeb/");
	 *		config.setDefaultHttpEnvironment(httpEnvironment);
	 *	}
	 * </pre>
	 *  
	 * @param config フレームワーク挙動情報を保持するオブジェクト
	 */
	protected void initialize(ClientConfig config) {
	}
	
	/**
	 * <p>[概 要] </p>
	 * ClientConfigの値に依存するクライアント初期化設定を行います。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 *  
	 * @param config フレームワーク挙動情報を保持するオブジェクト
	 */
	protected void postInitialize(ClientConfig config){
	}
	
	/**
	 * <p>[概 要] </p>
	 * 追加されたコンポーネントに対して予約されているリスナ追加をaddListenerメソッドに委譲します。
	 * 
	 * <p>[詳 細] </p>
	 * ContainerListenerImplが検知したコンポーネント追加イベント時にコールされます。<br>
	 * 追加されたコンポーネントの名前がEventBinderに紐付け予約されているものである場合、<br>
	 * 追加すべきイベントリスナ分addListenerメソッドを呼び出します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param addedComponent 画面に追加されたコンポーネント
	 */
	public void bindEvents(Context context, Object viewOrActivity) {
	
		String name = NamingRepository.get(context, viewOrActivity);
		
		if(name != null && !"".equals(name)) {
			List<Class<? extends Object>> listenerTypes = 
				this.eventBinder.getListenerTypes(name);
			
			for(Class<? extends Object> listenerType : listenerTypes){
				List<String> eventTypes = this.eventBinder.getEventTypes(name, listenerType);
				try {
					addListener(context, viewOrActivity, listenerType, eventTypes);
				} catch (Exception e) {
					// TODO 自動生成された catch ブロック
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 追加されたコンポーネントに対してイベントリスナを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数listenerTypeのプロキシオブジェクトを生成して、
	 * 引数componentにイベントリスナとして追加します。
	 * <p>
	 * 
	 * Swingコンポーネントはイベントタイプによってイベント追加メソッド名が異なります。<br>
	 * 生成されたプロキシはJavaの命名規則「"add" + listenerTypeクラス名」に沿って
	 * 追加メソッド名を以下のように動的判別します。
	 * 
	 * <pre>
	 * 		String addMethodName = "add" + listenerType.getSimpleName();
	 *		Method addMethod = component.getClass().getMethod(addMethodName, listenerType);
	 *		addMethod.invoke(component, eventListener);
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * プロキシ内のハンドラでは発生したイベントタイプを引数eventTypesと比較、
	 * eventTypesに含まれるハンドラメソッドが呼ばれる場合は
	 * {@link #handlerFacade(EventObject, Class, String, EventListener)}
	 * をコールします。
	 * 
	 * @param component イベントリスナプロキシを追加するコンポーネント
	 * @param listenerType プロキシの原型
	 * @param eventTypes イベントハンドラメソッド名リスト
	 */
	public void addListener(Context context,
			Object viewOrActivity,
			Class<? extends Object> listenerType, 
			List<String> eventTypes) throws Exception 
	{

		InvocationHandler handler = new EventInvocationHandler(context, viewOrActivity, this, listenerType, eventTypes);
		
		Object eventListener = 
			Proxy.newProxyInstance(
									this.getClass().getClassLoader(),
									new Class[] { listenerType }, 
									handler);

		ListenerAttachment listenerAttachment = 
			ListenerAttachmentFactory.getAttachment(viewOrActivity.getClass());
		listenerAttachment.attachListener(viewOrActivity, listenerType, eventListener);
		Log.d("bbbb = ", viewOrActivity.getClass().getSimpleName());

//		String addMethodName = "set" + listenerType.getSimpleName();
//		Method addMethod = viewOrActivity.getClass().getMethod(addMethodName, listenerType);
//		addMethod.invoke(viewOrActivity, listenerType.cast(eventListener));
	}
	
	/**
	 * <p>[概 要] </p>
	 * 全ユーザ定義イベントをハンドルする入り口になるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 発生したイベントタイプ、イベントを起こしたコンポーネント名を元に、<br>
	 * イベント紐付けオブジェクト（EventBinder）から対応するアクションクラス型を取得します。<br>
	 * 取得したアクションクラス型を引数にして{@link #invoke(Class, ParameterMapping)}に処理委譲します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e
	 * @param listenerType
	 * @param eventType
	 * @param eventListenerProxy
	 */
	public void handlerFacade(Context context,
								Object eventSource,
								Object[] eventArgs, 
								Class<? extends Object> listenerType,
								String eventType, 
								Object eventListenerProxy)
	{
		Log.d("BaseController", "handlerFacade called");
		
		
		String name = NamingRepository.get(context, eventSource);
		final Class<? extends AbstractAction> actionClass = 
			getEventBinder().getActionClass(name, listenerType, eventType);

		final ParameterMapping parameterMapping = 
			createParameterMapping(eventSource, name, eventArgs, listenerType, eventType, eventListenerProxy);

		invoke(actionClass, parameterMapping);
	}
	
	/**
	 * <p>[概 要] </p>
	 * MVC各レイヤを巡回するParameteraMappingオブジェクトを生成、初期化します。
	 * 
	 * <p>[詳 細] </p>
	 * イベント発生の際、Controller、Action、Modelを流れるデータマップを作成します。<br>
	 * このメソッドによって、以下の情報がParameterMappingオブジェクトに設定されます。<p>
	 * <ul>
	 * 		<li>発生したイベントオブジェクト</li>
	 * 		<li>イベントソースコンポーネント</li>
	 * 		<li>イベントリスナクラス型</li>
	 * 		<li>イベントリスナインスタンス</li>
	 * 		<li>イベントタイプ（イベントハンドラメソッド名）</li>
	 * </ul>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e イベントオブジェクト
	 * @param listenerType イベントリスナクラス型
	 * @param eventType イベントタイプ（イベントハンドラメソッド名）
	 * @param eventListenerProxy イベントリスナインスタンス
	 * @return MVC各レイヤを巡回するパラメータオブジェクト
	 */
	protected ParameterMapping createParameterMapping(Object eventSource,
														String eventSourceName,
														Object[] eventArgs, 
														Class<? extends Object> listenerType,
														String eventType,
														Object eventListenerProxy) 
	{
		ParameterMapping ret = new ParameterMapping();
		
		ret.setEventSource(eventSource);
		ret.setEventSourceName(eventSourceName);
		ret.setEventArgs(eventArgs);
		if(eventSource instanceof View) {
			ret.setContext(((View)eventSource).getContext());
		}else{
			ret.setContext((Context)eventSource);
		}
		
		ret.setEventListenerType(listenerType);
		ret.setEventListener(eventListenerProxy);
		ret.setEventType(eventType);
		
		return ret;
	}
	
	/**
	 * <p>[概 要]</p>
	 * UIコンポーネントとアクションを紐付ける、EventBinderオブジェクトを作成するメソッドです。
	 * 
	 * <p>[詳 細]</p>
	 * コントローラ実装クラスでこのメソッドをオーバーライドして、<br>
	 * eventBinder.addEventBinding("コンポーネント名", "イベントリスナ型", "イベントハンドラメソッド名", BaseAction継承クラス);<br>
	 * のように紐付け処理を列挙して下さい。<br>
	 * 
	 * ex.)eventBinder.addEventBinding("loginFrame.jbLogin", ActionListener.class, "actionPerformed", LoginAction.class);
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param eventBinder イベント紐付けオブジェクト
	 */
	protected void bind(EventBinder eventBinder) {
	}
	
	/**
	 * <p>[概 要] </p>
	 * JVMのシャットダウン時にコールされるフックハンドラです。
	 * 
	 * <p>[詳 細] </p>
	 * 具象コントローラでこのメソッドをオーバーライドして、
	 * アプリケーションの終末処理を記述します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void shutdown() {
	}
}
