/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.validator;

import java.awt.Component;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.co.fujitsu.reffi.client.swing.parser.ComponentValue;
import jp.co.fujitsu.reffi.client.swing.parser.ComponentValues;
import jp.co.fujitsu.reffi.common.util.ResourceUtil;

/**
 * <p>[概 要] </p>
 * 正規表現バリデータクラスです。
 * 
 * <p>[詳 細] </p>
 * コンポーネントの文字列値をpatternフィールドに設定された
 * 正規表現パターンでバリデーションします。
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b><br>
 * BaseAction継承クラス#validatorsメソッドで以下のように使用します。
 * <pre class="samplecode">
 *	&#064;Override
 *	protected void validators(List<Validator> validators) {
 *		
 *		// バリデーション対象コンポーネントを引数にしてインスタンス生成
 *		Component jtfEmail = getComponent("inputFormFrame.jtfEmail");
 *		RegExpValidator regexp = new RegExpValidator(jtfEmail);
 *
 *		// Eメールパターンをバリデーションパターンに設定 
 *		regexp.setPattern("[\\w\\.\\-]+@(?:[\\w\\-]+\\.)+[\\w\\-]+");
 *
 *		// 引数validatorsに追加
 *		validators.add(regexp);
 *	}
 * </pre>
 *
 * @author Project Reffi
 */
public class RegExpValidator extends Validator {

	/** エラーメッセージ取得ソースです。 */
	private final String MESSAGE_RESOURCE = "jp.co.fujitsu.reffi.common.exception.corelogic_message";

	/** バリデーションに使用する正規表現パターンです。 */
	private String pattern;

	
	/**
	 * <p>[概 要] </p>
	 * バリデーションに使用する正規表現パターンを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * patternフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return バリデーションに使用する正規表現パターン
	 */
	public String getPattern() {
		return pattern;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションに使用する正規表現パターンを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * patternフィールドに引数patternを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param pattern バリデーションに使用する正規表現パターン
	 */
	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 *   
	 * <p>[詳 細] </p>
	 * 引数component付きでsuper()を呼び出します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param component
	 */
	public RegExpValidator(Component component) {
		super(component);
	}

	/**
	 * <p>[概 要] </p>
	 * エラー文言登録メソッドです。
	 *  
	 * <p>[詳 細] </p>
	 * jp.co.fujitsu.reffi.shared.exception.corelogic_message.properties
	 * から対応するエラーメッセージを取得して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return エラー時メッセージ
	 */
	@Override
	protected String registerErrorMessage() {
		Properties messages = ResourceUtil.instance.asProperties(MESSAGE_RESOURCE);
		return messages.getProperty("EFC2004");
	}

	/**
	 * <p>[概 要] </p>
	 * 正規表現バリデーションを行います。
	 *  
	 * <p>[詳 細] </p>
	 * コンポーネントの文字列値をpatternフィールド値で正規表現マッチングします。
	 * 入力値がpatternにマッチしなかった場合、falseが返却されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param value コンポーネント値
	 * @return エラー有：false、エラー無：true
	 */
	@Override
	protected boolean validate(ComponentValues values) {
		boolean ret = true;
		
		for (int i = 0; i < values.size(); i++) {
			ComponentValue componentValue = values.getComponentValue(i);
			String val = componentValue.getValue().toString();

			Pattern ptn = Pattern.compile(getPattern());
			Matcher mc = ptn.matcher(val);
			if(mc.matches()) {
				ret = ret & true;
			} else{
				ret = ret & false;
			}
		}
		
		return ret;
	}
}
