/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.model;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import jp.co.fujitsu.reffi.client.nexaweb.controller.ParameterMapping;
import jp.co.fujitsu.reffi.client.nexaweb.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.common.exception.CoreLogicException;
import jp.co.fujitsu.reffi.common.nexaweb.constant.GlobalConstants;

import com.nexaweb.client.ClientSession;
import com.nexaweb.client.netservice.HttpRequest;
import com.nexaweb.client.netservice.HttpResponse;
import com.nexaweb.client.netservice.NetServiceException;
import com.nexaweb.client.netservice.NetServiceListener;
import com.nexaweb.client.netservice.RequestService;
import com.nexaweb.util.UrlUtils;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.ParserException;

/**
 * <p>[概 要] </p>
 * 同期、非同期HTTPリクエストを送信、レスポンス受信する為のモデルクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b>
 * 
 * ・同期通信でリモートサーバからxalファイルを受信、画面にレンダリングする
 * TODO javadoc修正
 * <pre class="samplecode">
 * 	public class OpenChatAction extends BaseAction {
 *
 *		&#064;Override
 *		protected void reserveModels(List<Class<? extends Model>> models) {
 *			// HTTP通信モデルを実行予約
 *			models.add(HTTPRequestCore.class);
 *		} 
 *
 *		&#064;Override
 *		public void nextModel(int index, Model prev, Model next) throws Exception{
 *			switch(index){
 *				case 0:
 *					// サーバロジックの指定方法は任意（strutsでも自作F/Wでも可）
 *					((HTTPRequestCore)next).setRequestUrl("webcontroller");
 *					((HTTPRequestCore)next).addUrlParameters("forward.page", "/pages/chat.xal");
 *					break;
 *			}
 *		}
 *	}
 * </pre>
 * 
 * ・非同期通信でリモートサーバからシリアライズオブジェクトを受信、テーブルに受信データを埋め込む
 * <pre class="samplecode">
 * 	public class InitializeTableWindowAction extends BaseAction {
 *	
 *		&#064;Override
 *		protected void reserveModels(List<Class<? extends Model>> models) {
 *			// HTTP通信モデルを実行予約
 *			models.add(HTTPRequestCore.class);
 *		} 
 *
 *		&#064;Override
 *		public void nextModel(int index, Model prev, Model next) throws Exception {
 *			switch(index) {
 *				case 0:
 *					// サーバロジックの指定方法は任意（strutsでも自作F/Wでも可）
 *					((HTTPRequestCore)next).setRequestUrl("webcontroller");
 *					((HTTPRequestCore)next).addUrlParameters("model.fqcn", "demo.server.model.FetchTableDataModel");
 *					// 受信データを自動レンダリングさせない
 *					((HTTPRequestCore)next).setRenderResponse(false);
 *					// 通信は非同期を指定
 *					((HTTPRequestCore)next).setAsync(true);
 *					break;
 *			}
 *		}
 *
 *		&#064;Override
 *		public void successForward(int index, Model model, Object result) throws Exception {
 *			// モデル結果オブジェクトをHttpResponse型にキャスト
 *			HttpResponse response = (HttpResponse)result;
 *			// HttpResponseからシリアライズされたTableDataリストを取得
 *			List<TableData> data = (List<TableData>)getSerializedHttpResponseContent(response);
 *
 *			// 設定対象データソースオブジェクトを取得
 *			ObjectDataSource objectDataSource = getObjectDataSource("tableDataSource");
 *			// データソースにサーバモデル結果を設定
 *			objectDataSource.setSource(data);
 *		}
 *	}
 *
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class HTTPRequestCore extends XalReturnPossibilityModel implements NetServiceListener {

	/** HTTPリクエストするURLです。 */
	private String requestUrl;

	/** HTTPリクエストメソッドです。(デフォルト："POST") */
	private String requestMethod = "POST";

	/** 「key=value」形式URLパラメータがマッピングされたオブジェクトです。 */
	private Map<String, String> urlParameters;

	/** HTTPレスポンスをUIにレンダリングするかどうかのフラグです。（デフォルト：true） */
	private boolean renderResponse = true;

	/** HTTPリクエストを非同期で送るかどうかのフラグです。（デフォルト：true） */
	private boolean async = true;

	/** HTTPリクエストを非同期で送る際にURLを動的にするかどうかのフラグです。（デフォルト：false） */
	private boolean dynamicUrl = false;

	/**
	 * <p>[概 要] </p>
	 * HTTPリクエストするURLを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return リクエストURL
	 */
	public String getRequestUrl() {
		return this.requestUrl;
	}

	/**
	 * <p>[概 要] </p>
	 * HTTPリクエストするURLを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param requestUrl リクエストURL
	 */
	public void setRequestUrl(String requestUrl) {
		this.requestUrl = requestUrl;
	}

	/**
	 * <p>[概 要] </p>
	 * HTTPリクエストメソッドを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return リクエストメソッド
	 */
	public String getRequestMethod() {
		return this.requestMethod;
	}

	/**
	 * <p>[概 要] </p>
	 * HTTPリクエストメソッドを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param requestMethod リクエストメソッド
	 */
	public void setRequestMethod(String requestMethod) {
		this.requestMethod = requestMethod;
	}

	/**
	 * <p>[概 要] </p>
	 * 「key=value」形式URLパラメータマップを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return URLパラメータマップ
	 */
	public Map<String, String> getUrlParameters() {
		return this.urlParameters;
	}

	/**
	 * <p>[概 要] </p>
	 * 「key=value」形式URLパラメータマップを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param urlParameters URLパラメータマップ
	 */
	public void setUrlParameters(Map<String, String> urlParameters) {
		this.urlParameters = urlParameters;
	}

	/**
	 * <p>[概 要] </p>
	 * 「key=value」形式URLパラメータマップに要素を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param key パラメータキー
	 * @param value パラメータ値
	 */
	public void addUrlParameters(String key, String value) {
		getUrlParameters().put(key, value);
	}

	/**
	 * <p>[概 要] </p>
	 * 「key=value」形式URLパラメータマップから要素を削除します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param key パラメータキー
	 */
	public void removeUrlParameters(String key) {
		getUrlParameters().remove(key);
	}

	/**
	 * <p>[概 要] </p>
	 * HTTPレスポンスをUIにレンダリングするかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return レンダリングする場合はtrue、それ以外はfalse
	 */
	public boolean isRenderResponse() {
		return this.renderResponse;
	}

	/**
	 * <p>[概 要] </p>
	 * HTTPレスポンスをUIにレンダリングするかどうかのフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param renderResponse レンダリングする場合はtrue、それ以外はfalse
	 */
	public void setRenderResponse(boolean renderResponse) {
		this.renderResponse = renderResponse;
	}

	/**
	 * <p>[概 要] </p>
	 * 非同期リクエスト送信フラグ取得
	 * 
	 * <p>[詳 細] </p>
	 * サーバへの通信を非同期で実行するフラグを取得します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 非同期実行ならtrue、それ以外はfalse
	 */
	public boolean isAsync() {
		return this.async;
	}

	/**
	 * <p>[概 要] </p>
	 * 非同期リクエスト送信フラグ設定
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param async 非同期実行ならtrue、それ以外はfalse
	 */
	public void setAsync(boolean async) {
		this.async = async;
	}

	/**
	 * <p>[概 要] </p>
	 * 動的URLフラグ取得
	 * 
	 * <p>[詳 細] </p>
	 * URLを動的に生成してサーバへ送信を行うフラグを取得します。
	 * 
	 * <p>[備 考] </p>
	 * 本フラグがtrueの場合、あらかじめサーバ側のweb.xmlでDynamicUrlFilterが
	 * 定義されている必要があります。
	 * 
	 * @return 動的URLを生成する場合はtrue、それ以外はfalse
	 */
	public boolean isDynamicUrl() {
		return dynamicUrl;
	}

	/**
	 * <p>[概 要] </p>
	 * 動的URLフラグ設定
	 * 
	 * <p>[詳 細] </p>
	 * URLを動的に生成してサーバへ送信を行うフラグを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 本フラグがtrueの場合、あらかじめサーバ側のweb.xmlでDynamicUrlFilterが
	 * 定義されている必要があります。
	 * 
	 * @param dynamicUrl 動的URLを生成する場合はtrue、それ以外はfalse
	 */
	public void setDynamicUrl(boolean dynamicUrl) {
		this.dynamicUrl = dynamicUrl;
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタ
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public HTTPRequestCore() {
		setUrlParameters(new HashMap<String, String>());
	}

	@Override
	protected void mainproc() throws Exception {
		super.mainproc();

		// HTTPリクエスト情報オブジェクトを生成
		HttpRequest httpRequest = createFormRequest(getParameterMapping());
		if (httpRequest == null) {
			throw new CoreLogicException("EFC1001");
		}

		// HTTPリクエスト送信サービスオブジェクトを取得
		ClientSession clientSession = getParameterMapping().getClientSession();
		RequestService requestService = clientSession.getRequestService();

		if (isAsync()) {
			// 非同期リクエスト
			asyncRequest(httpRequest, requestService);
		} else {
			// 同期リクエスト
			syncRequest(httpRequest, requestService);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 同期HTTPリクエストを送信します。
	 * 
	 * <p>[詳 細] </p>
	 * isRenderingResponse()の返却値を判定して、以下の処理を行います。
	 * 【trueの場合】
	 * response = requestService.retrieveAndProcess(httpRequest);
	 * 【falseの場合】
	 * response = requestService.retrieve(httpRequest);
	 *
	 * trueの場合はレスポンス返却時点で画面に反映されます。
	 * この場合、HTTPレスポンスのコンテントはXALデータである必要が有ります。
	 * 
	 * isRenderingResponse()の結果如何に関わらず、受信したデータはrequestCompleted
	 * メソッドに譲渡されます。
	 * 
	 * <p>[備 考] </p>
	 * isRenderingResponse() == true時、レスポンスがXALデータで無かった場合は
	 * 例外が発生します。
	 * シリアライズオブジェクトをレスポンスデータとして受信する場合は
	 * setRenderingResponse(false)を設定しておく必要があります。
	 * 
	 * @param httpRequest HTTPリクエスト
	 * @param requestService リクエストサービス
	 * @throws Exception リクエスト送信例外
	 */
	protected void syncRequest(HttpRequest httpRequest, RequestService requestService) throws Exception {
		HttpResponse response = null;
		try {
			response = requestService.retrieve(httpRequest);
			requestCompleted(httpRequest.getUri(), response);
		} catch (NetServiceException e) {
			throw new CoreLogicException("EFC1002", e);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 非同期HTTPリクエストを送信します。
	 * 
	 * <p>[詳 細] </p>
	 * isRenderingResponse()の返却値を判定して、以下の処理を行います。
	 * 【trueの場合】
	 * requestService.retrieveAndProcessAsynchronously(httpRequest,this);
	 * 【falseの場合】
	 * requestService.retrieveAsynchronously(httpRequest, this);
	 *
	 * trueの場合はレスポンス返却時点で画面に反映されます。
	 * この場合、HTTPレスポンスのコンテントはXALデータである必要が有ります。
	 * 
	 * レスポンス受信成功はthis.requestCompletedが、失敗時はthis.requestFailed
	 * が、NetServiceListenerによってコールバックされます。
	 * 
	 * <p>[備 考] </p>
	 * isRenderingResponse() == true時、レスポンスがXALデータで無かった場合は
	 * 例外が発生します。
	 * シリアライズオブジェクトをレスポンスデータとして受信する場合は
	 * setRenderingResponse(false)を設定しておく必要があります。
	 * 
	 * @param httpRequest HTTPリクエスト
	 * @param requestService リクエストサービス
	 * @throws Exception リクエスト送信例外
	 */
	protected void asyncRequest(HttpRequest httpRequest, RequestService requestService) throws Exception {
		try {
			requestService.retrieveAsynchronously(httpRequest, this);
		} catch (Exception e) {
			throw new CoreLogicException("EFC1003", e);
		}
	}

	/**
	 * <p>[概 要]</p>
	 * リクエスト情報生成
	 * 
	 * <p>[詳 細]</p>
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param リクエストするURL（パラメータ無し）
	 * @return 生成されたリクエスト情報
	 */
	private HttpRequest createFormRequest(ParameterMapping parameterMapping)
			throws Exception {

		String url = getRequestUrl();
		if (isAsync() && isDynamicUrl()) {
			url = createDynamicUrl(url);
		}
		
		String requestMethod = getRequestMethod();
		Map<String, String> urlParameters = getUrlParameters();

		// HTTPリクエストオブジェクト生成
		ClientSession clientsession = parameterMapping.getClientSession();
		HttpRequest httprequest = clientsession.getRequestService()
				.createHttpRequest(url);
		httprequest.setRequestMethod(requestMethod);

		// 最終的なURLパラメータ文字列バッファ
		StringBuilder urlParameterBuilder = new StringBuilder();
		if (urlParameters != null) {
			Set<String> set = urlParameters.keySet();
			Iterator<String> it = set.iterator();
			for (int i = 0; it.hasNext(); i++) {
				String name = it.next();
				String value = urlParameters.get(name);

				if (i > 0) {
					urlParameterBuilder.append('&');
				}
				if (name == null) {
					throw new CoreLogicException("EFC0004");
				}
				urlParameterBuilder.append(UrlUtils.encode(name));
				urlParameterBuilder.append('=');
				urlParameterBuilder.append(UrlUtils.encode(value));
			}

			// 作成したURLパラメータを基にリクエストメソッド毎に処理
			if (urlParameterBuilder.length() > 0) {
				String urlParameter = urlParameterBuilder.toString();
				try {
					if ("POST".equalsIgnoreCase(httprequest.getRequestMethod())) {
						httprequest.setHeader("Content-Type",
								"application/x-www-form-urlencoded");
						httprequest.setContent(urlParameter.getBytes("UTF8"));
					} else {
						String uri = httprequest.getUri();
						int index = uri.indexOf('?');
						if (index != -1) {
							if (!uri.endsWith("?") && !uri.endsWith("&")) {
								uri = uri + "&";
							}
							uri = uri + urlParameter;
						} else {
							uri = uri + "?" + urlParameter;
						}
						httprequest.setUri(uri);
					}
				} catch (UnsupportedEncodingException e) {
					throw new CoreLogicException("EFC1004", e);
				}
			}
		}
		return httprequest;
	}

	/**
	 * <p>[概 要] </p>
	 * リクエスト完了処理
	 * 
	 * <p>[詳 細] </p>
	 * リクエストが完了しレスポンスを受信した際の処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public final void requestCompleted(String uri, HttpResponse response) {
		// 汎用モデル処理結果オブジェクトに設定
		setResult(response);

		// HTTPリクエスト結果受信用メソッドをテンプレートコール
		postRequest(uri, response);

		// ModelProcessEvent#getResult用
		Object result = null;
		// HttpResponseのコンテント判別開始
		try {
			// HttpResponseコンテントのDOM化試行
			Document document = response.toDocument();
			// パース成功
			// 受信したのがxalの場合、windowやdialogに識別子（parentId, communicateId）を付ける
			addIdentifierToWindows(document);
			// XMLパースに成功した場合、documentが返却値
			result = document;
			if (isRenderResponse()) {
				getController().getSession().processXml(document);
			}
		} catch (ParserException e) {
			// XMLパースに失敗した場合
			ObjectInputStream objectInputStream = null;
			try {
				// シリアライズ復元に成功した場合、シリアライズオブジェクトが返却値
				objectInputStream = new ObjectInputStream(
						new ByteArrayInputStream(response.getContent()));
				Object serializedObj = objectInputStream.readObject();
				result = serializedObj;
			} catch (IOException ee) {
				// シリアライズに失敗した場合、生コンテントが返却値
				result = response.getContent();
			} catch (ClassNotFoundException ee) {
				// シリアライズに失敗した場合、生コンテントが返却値
				result = response.getContent();
			} catch (NullPointerException ee) {
				// HttpResponseにコンテントが無かった場合、nullが返却値
				result = null;
			}finally{
				if(objectInputStream != null){
					try {
						objectInputStream.close();
						objectInputStream = null;
					} catch (IOException ee) {
					}
				}
			}
		} catch (NullPointerException e) {
			// HttpResponseにコンテントが無かった場合、nullが返却値
			result = null;
		}

		ModelProcessEvent evt = new ModelProcessEvent(this);
		evt.setResult(result);
		fireModelSuccess(evt);
		
		ModelProcessEvent finishedEvent = new ModelProcessEvent(this);
		finishedEvent.setResult(result);
		fireModelFinished(finishedEvent);
	}

	/**
	 * <p>[概 要] 非同期リクエスト連続実行時の動的URLを生成します。</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @param url リクエストURL
	 * @return 生成した動的URL
	 */
	private String createDynamicUrl(String url) {

		Random random = new Random();
		StringBuilder sb = new StringBuilder();
		sb.append(random.nextInt(10));
		sb.append(random.nextInt(10));
		sb.append(random.nextInt(10));
		sb.append(random.nextInt(10));
		sb.append(random.nextInt(10));
		
		return GlobalConstants.DYNAMIC_URL_PATH + "/" + sb.toString() + url;
	}

	/**
	 * <p>[概 要] </p>
	 * リクエスト失敗処理
	 * 
	 * <p>[詳 細] </p>
	 * リクエスト送信が失敗した場合の処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public final void requestFailed(String uri, NetServiceException e) {
		// HTTPリクエスト失敗ハンドリングメソッドをテンプレートコール
		postRequestFailed(uri, e);
		Exception ex = trap(e);
		// trapの戻り値がnullでなければこのタイミングで例外イベント発火
		if (ex != null) {
			ModelProcessEvent evt = new ModelProcessEvent(this);
			evt.setException(new CoreLogicException("EFC1003", ex));
			fireModelFailure(evt);

			ModelProcessEvent finishedEvent = new ModelProcessEvent(this);
			finishedEvent.setException(new CoreLogicException("EFC1003", ex));
			fireModelFinished(finishedEvent);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * リクエスト送信後処理
	 * 
	 * <p>[詳 細] </p>
	 * リクエスト送信後の処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * 本メソッドは、本クラスを継承するモデルが送信後に何かしらの処理を
	 * 行いたい場合のみ実装されます。
	 * 
	 * @param uri リクエストURI
	 * @param response HTTPレスポンス
	 */
	public void postRequest(String uri, HttpResponse response) {
	}

	/**
	 * <p>[概 要] </p>
	 * リクエスト送信失敗後処理
	 * 
	 * <p>[詳 細] </p>
	 * リクエスト送信失敗の後処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * 本メソッドは、本クラスを継承するモデルが送信失敗後に何かしらの処理を
	 * 行いたい場合のみ実装されます。
	 * 
	 * @param uri リクエストURI
	 * @param e リクエスト送信例外
	 */
	public void postRequestFailed(String uri, NetServiceException e) {
	}
}
