/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.action {

	import flash.events.Event;
	
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	/**
	 * <p>[概 要] </p>
	 * モデルの処理結果（成功、失敗）を任意の実装メソッドに振り分けるアクションクラスです.
	 * 
	 * <p>[詳 細] </p>
	 * BaseAction実装アクションでは、登録されたモデルの処理結果をsuccessForward、
	 * failureForwardメソッドの二つの固定メソッドコールバックで取得しますが、<br />
	 * DispatchAction実装アクションでは、任意のメソッドをコールバックメソッドに
	 * 指定します。
	 * 
	 * BaseAction実装とはモデルの予約方法が異なります。<br />
	 * DispatchActionを実装したアクションでは、reserveModelsの代わりに
	 * reserveModelsAndCallbackMethod(CallbackMapping)を実装します。
	 * <listing version="3.0">
         protected function reserveModelsAndCallbackMethod(callbackMapping:CallbackMapping):void {
             // HTTPServiceCore機能モデルが成功した場合successFetchDataメソッドが、失敗した場合filureFetchDataメソッドが呼ばれる
             callbackMapping.add(Class(HTTPServicetCore), "successFetchData", "failureFetchData");
         }
	 * </listing>
	 *
	 * reserveModelsAndCallbackMethodで予約したコールバックメソッドを、以下のシグネチャで実装します。
	 * <listing version="3.0">
         // モデル処理正常終了時にコールバックされる
         public function successFetchData(model:Model, resultEvent:Event):void {
            trace("データ取得成功");
         }
     
         // モデル処理異常終了時にコールバックされる
         public function failureFetchData(model:Model, faultEvent:Event):Event {
             trace("データ取得失敗");
             return e;
         }
	 * </listing>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class DispatchAction extends BaseAction {

		// 実行モデルクラス、成功、失敗コールバックメソッド名をマッピングするオブジェクト
		private var _callbackMapping:CallbackMapping;
		
		/**
		 * <p>[概 要]</p>
		 * 実行モデルクラス、成功、失敗コールバックメソッド名をマッピングするプロパティです.
		 * 
		 * <p>[詳 細]</p>
		 * DispatchアクションがsuccessForward、failureFowardをハンドリングすると、
		 * このプロパティから対応するメソッド名を取得、コールします。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @see jp.co.fujitsu.reffi.client.flex.action.CallbackMapping
		 */
		public function get callbackMapping():CallbackMapping {
			return this._callbackMapping;
		}		
		public function set callbackMapping(callbackMapping:CallbackMapping):void {
			this._callbackMapping = callbackMapping;
		}

		/**
		 * <p>[概 要]</p>
		 * コントローラにコールされるアクションの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * BaseAction#runを実行した後、reserveModelsAndCallbackMethodメソッドの
		 * テンプレートコールを行います。<br />
		 * オーバーライドされたreserveModelsAndCallbackMethodで登録された
		 * 機能モデルはparameterMapping#modelsとして登録されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @return 継承Actionから収集した、コントローラに返却される制御情報
		 */
        override public function run(parameterMapping:ParameterMapping):ParameterMapping{
        	super.run(parameterMapping);
        	
        	try{
				callbackMapping = new CallbackMapping();
				reserveModelsAndCallbackMethod(callbackMapping);
				
				var modelClasses:Array = new Array();
				for each(var map:Object in callbackMapping.list) {
					modelClasses.push(map[CallbackMapping.CLASS]);
				}
				
				parameterMapping.models = modelClasses;
        	}catch(e:Error) {
        		e = trap(e);
        		if(e) {
        			throw e;
        		}
        	}
        	
			return parameterMapping;
        }

		/**
		 * <p>[概 要]</p>
		 * 機能モデル、成功イベントハンドラ名、失敗イベントハンドラ名を予約します.
		 * 
		 * <p>[詳 細]</p>
		 * 引数callbackMappingに対してコントローラに実行委譲する機能モデルクラス型、
		 * その機能モデルが成功した時のイベントハンドラ名、失敗した時のイベントハンドラ名を
		 * 追加します。
		 * 
		 * <p>[備 考]</p>
		 */
		protected function reserveModelsAndCallbackMethod(callbackMapping:CallbackMapping):void {
		}
		
		/**
		 * <p>[概 要]</p>
		 * オーバーライド不可です.
		 * 
		 * <p>[詳 細]</p>
		 * reserveModelsAndCallbackMethodメソッドをオーバーライドして下さい。
		 * 
		 * <p>[備 考]</p>
		 */
		override protected final function reserveModels(models:Array):void {
		} 
		
		/**
		 * <p>[概 要]</p>
		 * オーバーライド不可です.
		 * 
		 * <p>[詳 細]</p>
		 * callbackMappingプロパティから引数index番目の情報を取得して
		 * 対応する成功イベントハンドリングメソッド名を取得、
		 * 同名のメソッドを以下のシグネチャでコールします。
		 * 
		 * 成功イベントハンドルメソッド名(model:BaseModel, resultEvent:Event);
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param index 結果を返却したモデルの実行インデックス
		 * @param model 結果を返却したモデルインスタンス
		 * @param resultEvent モデル処理結果オブジェクト
		 */
		override public final function successForward(index:int, model:BaseModel, resultEvent:Event) : void {
			var map:Object = callbackMapping.getMapping(index);
			
			this[map[CallbackMapping.SUCCESS]].call(null, model, resultEvent);
		}
		
		/**
		 * <p>[概 要]</p>
		 * オーバーライド不可です.
		 * 
		 * <p>[詳 細]</p>
		 * callbackMappingプロパティから引数index番目の情報を取得して
		 * 対応する失敗イベントハンドリングメソッド名を取得、
		 * 同名のメソッドを以下のシグネチャでコールします。
		 * 
		 * faultEvent = 失敗イベントハンドルメソッド名(model:BaseModel, faultEvent:Event);
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param index 結果を返却したモデルの実行インデックス
		 * @param model 結果を返却したモデルインスタンス
		 * @param faultEvent モデル異常終了時のイベント
		 * @return 引数faultEvent、若しくはオーバーライドメソッドで変換されたイベント
		 */
		override public final function failureForward(index:int, model:BaseModel, faultEvent:Event):Event{
			var map:Object = callbackMapping.getMapping(index);
			
			faultEvent = this[map[CallbackMapping.FAILURE]].call(null, model, faultEvent);

			return faultEvent;
		}
	}
}
	


