/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.manager {
	import flash.utils.Dictionary;
	
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.messaging.ConsumerCore;
	
	import mx.messaging.ChannelSet;
	import mx.messaging.Consumer;
	import mx.messaging.events.MessageEvent;
	import mx.messaging.events.MessageFaultEvent;
	
	/**
	 * <p>[概 要]</p>
	 * コンシューマを内包するConsumerCoreを管理するマネージャクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * destination名をキーにしたマッププロパティ「destinations」に、
	 * 購読を行うConsumerインスタンスと、<br>
	 * イベントハンドリングを行うConsumerCoreのインスタンスを保持します。<p>
	 * 
	 * destinationsプロパティの構造は以下のようになります。
	 * <pre>
	 * destinations:Dictionary
	 *     "destination名1" = consumerMap:Dictionary
	 *                            "consumer" = Consumer
	 *                            "coreList" = consumerCoreList:Array
	 *                                             [0] = ConsumerCore
	 *                                             [1] = ConsumerCore
	 *                                             [2] = ConsumerCore
	 *     "destination名2" = consumerMap:Dictionary
	 *                            "consumer" = Consumer
	 *                            "coreList" = consumerCoreList:Array
	 *                                             [0] = ConsumerCore
	 *     "destination名3" = consumerMap:Dictionary
	 *         :
	 *         :
	 * </pre>
	 * 1 destination購読につき、1個のConsumerと、n個のConsumerCoreが保持されます。<br>
	 * 既に購読中のdestinationをsubscribeした場合、登録済みのConsumerに対して
	 * 新規ConsumerCoreのresultHandlerメソッドとfaultHandlerメソッドがaddEventListenerされます。<br>
	 * 
	 * <p>[備 考]</p>
	 * ConsumerCoreをsubscribeモードfalseで実行させることでsubscribeの停止が行えますが、<br>
	 * ConsumerManager#unsubscribeを使用すると、任意のタイミングでのsubscribe停止が行えます。
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 * @see jp.co.fujitsu.reffi.client.flex.model.messaging.ConsumerCore
	 */
	public class ConsumerCoreManager {
		
		// destinationとConsumer、ConsumerCoreのマッピング
		private var _destinations:Dictionary = new Dictionary();

		/**
		 * <p>[概 要]</p>
		 * destinationとConsumer、ConsumerCoreのマッピング情報を管理するオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * 現在購読中のdestinationと、実際に購読処理を行うConsumer、
		 * Consumerの購読結果を受信する為のイベントハンドラを持つConsumerCore
		 * をマッピング管理します。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get destinations():Dictionary {
			return this._destinations;
		}
		public function set destinations(destinations:Dictionary) :void {
			this._destinations = destinations;
		}
		

		/**
		 * <p>[概 要]</p>
		 * コンストラクタです.
		 * 
		 * <p>[詳 細]</p>
		 * 本クラスはシングルトンクラスのため外部からの呼び出しは行えません。
		 * 
		 * <p>[備 考]</p>
		 * ConsumerManagerのインスタンスを取得する場合は、getInstance()を使用してください。
		 * 
		 * <p>
		 * @param blocker シングルトン生成用ブロックインスタンス
		 * @see #getInstance()
		 */
		public function ConsumerCoreManager(blocker:ConsumerBlocker) {
			if (blocker == null) {
				throw Error("This class is singleton.");
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * コンシューママネージャインスタンスを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * シングルトンコンシューママネージャインスタンスを取得して返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return コンシューママネージャインスタンス
		 */
		public static function getInstance():ConsumerCoreManager {
			return ConsumerBlocker.instance;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数destinationを現在購読しているか確認します.
		 * 
		 * <p>[詳 細]</p>
		 * Consumer管理オブジェクト「destinations」から、引数destinationが
		 * 登録されているかどうか調べます。<br>
		 * destinationを購読しているConsumerCoreが一つでも有った場合はtrue
		 * を返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param destination 購読されているかどうか調べるdestination名
		 * @return 購読されている場合はtrue、それ以外はfalse
		 */
		public function isSubscribing(destination:String):Boolean {
			return this._destinations.hasOwnProperty(destination);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 現在購読中のdestination名の配列を返却します.
		 * 
		 * <p>[詳 細]</p>
		 * Consumer管理プロパティ「destinations」内のキー名配列を返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return 現在購読中のdestination名配列
		 */
		public function getSubscribingDestinationNames():Array {
			var destinationNames:Array = new Array();
			for(var destination:String in this.destinations) {
				destinationNames.push(destination);
			}
			
			return destinationNames; 
		}
		
		/**
		 * <p>[概 要]</p>
		 * destinationの購読を開始します.
		 * 
		 * <p>[詳 細]</p>
		 * 指定されたdestinationが既に購読状態であった場合、購読中のConsumerに
		 * consumerCoreのresultHandlerとfaultHandlerをイベントリスナ追加します。<br>
		 * 未読状態であった場合は新規にConsumerインスタンスを生成して同様にconsumerCore
		 * のresultHandlerとfaultHandlerをイベントリスナに追加します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param destination 購読を開始するdestination
		 * @param consumerCore このマネージャに管理されるConsumerCoreインスタンス
		 * @param channelSet 任意設定可能なチャンネル設定
		 */
		public function subscribe(destination:String, 
									consumerCore:ConsumerCore, 
									channelSet:ChannelSet = null):void 
		{
			// 単一destinationに対応するConsumer、ConsumerCore配列マッピング
			var consumerMap:Dictionary = null;
			// destinationをsubscribeするConsumerオブジェクト
			var consumer:Consumer = null;
			// ConsumerのsubscribeをハンドリングするConsumerCoreの配列
			var consumerCoreList:Array = null;
			
			// 既に引数destinationに対するConsumerがマッピングされている場合
			if(this.destinations.hasOwnProperty(destination)) {
				consumerMap = this.destinations[destination];
				consumer = consumerMap["consumer"];
				consumerCoreList = consumerMap["coreList"];
			}else{
				// 第一引数destinationの購読用Consumerインスタンス生成
				consumer = new Consumer();
				// 第三引数にチャンネル情報が指定されていればそれを使用
				if (channelSet != null) {
					consumer.channelSet = channelSet;
				}
				// destination設定、購読開始
				consumer.destination = destination;
				consumer.subscribe();

				// 購読開始したConsumer、ConsumerCore配列をdestinationsにマッピング
				consumerMap = new Dictionary();
				consumerCoreList = new Array();
				consumerMap["consumer"] = consumer;
				consumerMap["coreList"] = consumerCoreList;
				this.destinations[destination] = consumerMap;
			}
			
			// イベントハンドラ（ConsumerCore#resultHandler、faultHandler）追加。
			consumer.addEventListener(MessageEvent.MESSAGE, consumerCore.resultHandler);
			consumer.addEventListener(MessageFaultEvent.FAULT, consumerCore.faultHandler);
			// consumerCore配列に引数consumerCoreを追加管理
			consumerCoreList.push(consumerCore);
		}
		
		/**
		 * <p>[概 要]</p>
		 * destinationの購読を中止します.
		 * 
		 * <p>[詳 細]</p>
		 * 引数destinationを購読中のConsumerを停止します。<br>
		 * 引数identifierが指定されている場合、購読イベントをハンドリング中のConsumerCore
		 * の中から、同じidentifierをもつものだけハンドリングを停止します。<br>
		 * destination購読イベントをハンドリング中のConsumerCoreが全て無くなった場合、
		 * Consumerはunsubscribeを行います。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param destination 購読を中止するdestination
		 * @param identifier destinationを購読している複数ConsumerCoreを識別する為の文字列
		 */
		public function unsubscribe(destination:String, identifier:String = null):void {
			
			// 現在購読中ではないdestinationの場合は処理しない
			if(!this.destinations.hasOwnProperty(destination)) {
				return;
			}
			
			// 第一引数destinationを購読しているConsumerCoreの配列を取得
			var consumerMap:Dictionary = this.destinations[destination];
			var consumer:Consumer = consumerMap["consumer"];
			var consumerCoreList:Array = consumerMap["coreList"];
			var consumerCore:ConsumerCore = null;
			
			// 第二引数identifierがnullの場合はConsumerCore配列の全てを購読中止
			if(identifier == null) {
				// 無条件に購読中止
				consumer.unsubscribe();
				for each(consumerCore in consumerCoreList) {
					// consumerに関連付いているイベントハンドラを解放
					consumer.removeEventListener(MessageEvent.MESSAGE, consumerCore.resultHandler);
					consumer.removeEventListener(MessageFaultEvent.FAULT, consumerCore.faultHandler);
					// ConsumerCoreにModel完全終了イベントを送信
					var modelFinishedEvent:ModelProcessEvent = 
						new ModelProcessEvent(ModelProcessEvent.FINISHED);
					consumerCore.dispatchModelFinished(modelFinishedEvent);
				}
				// ConsumerCore管理オブジェクトからdestinationキーを削除
				delete this.destinations[destination];
			}else {
				for (var index:int = consumerCoreList.length -1; index >= 0; index--) {
					consumerCore = consumerCoreList[index];
					// 第二引数identifierと等しいidentifierを持つConsumerCoreの場合
					if(consumerCore.identifier == identifier) {
						// consumerに関連付いているイベントハンドラを解放
						consumer.removeEventListener(MessageEvent.MESSAGE, consumerCore.resultHandler);
						consumer.removeEventListener(MessageFaultEvent.FAULT, consumerCore.faultHandler);
						// ConsumerCoreにModel完全終了イベントを送信
						modelFinishedEvent = 
							new ModelProcessEvent(ModelProcessEvent.FINISHED);
						consumerCore.dispatchModelFinished(modelFinishedEvent);
						// ConsumerCore配列から削除
						consumerCoreList.splice(index, 1);
					}
				}
				// ConsumerCore配列が空になっている場合はsubscribeを停止してdestination登録を削除
				if(consumerCoreList.length == 0) {
					consumer.unsubscribe();
					delete this.destinations[destination];
				}
			}
		}
	}
}

import jp.co.fujitsu.reffi.client.flex.manager.ConsumerCoreManager;

/**
 * シングルトンブロック用クラス
 */
class ConsumerBlocker {
	public static var instance:ConsumerCoreManager = new ConsumerCoreManager(new ConsumerBlocker());
	
	public function ConsumerBlocker() {
		/** */
	}
}
