/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.util;

import java.awt.Component;
import java.awt.Container;
import java.awt.Window;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import jp.co.fujitsu.reffi.client.swing.manager.WindowManager;

/**
 * <p>[概 要] </p>
 * Swingコンポーネントを検出するユーティリティクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * <p>[備 考] </p>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ComponentSearchUtil {

	/**
	 * <p>[概 要] </p>
	 * 引数Componentから引数nameを持つコンポーネントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * componentがコンテナコンポーネントの場合、再帰的に子孫コンポーネントを走査します。
	 * nameを持つコンポーネントが見つかった時点で再帰を中断、コンポーネントを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param component コンポーネントを探すルート
	 * @param name 探すコンポーネントの名前
	 * @return nameを名前として持つコンポーネント。見つからなかった場合はnull。
	 */
	public static Component searchComponentByName(Component component, String name) {
		Component ret = null;

		if(name.equals(component.getName())) {
			return component;
		}
		
		if(component instanceof Container) {
			Component[] cs = ((Container)component).getComponents();
			for(Component c : cs) {
				ret = searchComponentByName(c, name);
				if(ret != null) {
					return ret;
				}
			}
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * メモリ上に存在する全ウィンドウコンポーネントから引数nameを持つコンポーネントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * WindowManagerが管理している画面コンポーネントリストの中から
	 * nameを名前として持つコンポーネントを探して返却します。
	 * 
	 * <p>[備 考] </p>
	 *  
	 * @param name 探すコンポーネントの名前
	 * @return nameを名前として持つコンポーネント。見つからなかった場合はnull。
	 */
	public static Component searchComponentByNameFromAllWindow(String name) {
		Component ret = null;
		
		for(Window window : WindowManager.getInstance().getWindowList()) {
			ret = searchComponentByName(window, name);
			if(ret != null) {
				break;
			}
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数Componentから引数nameを持つ全てのコンポーネントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * componentがコンテナコンポーネントの場合、再帰的に子孫コンポーネントを走査します。
	 * nameを持つコンポーネントが見つかった場合でも全ての子孫を走査し終わるまで再帰を続けます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param list 検出されたコンポーネントが格納されるリスト
	 * @param component コンポーネントを探すルート
	 * @param name 探すコンポーネントの名前
	 */
	public static void searchComponentsByName(List<Component> list, Component component, String name) {
		
		if(name.equals("*") || name.equals(component.getName())) {
			list.add(component);
		}
		
		if(component instanceof Container) {
			Component[] cs = ((Container)component).getComponents();
			for(Component c : cs) {
				searchComponentsByName(list, c, name);
			}
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数Componentが所属するウィンドウコンポーネントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * componentのparentを辿り、Windowクラス継承オブジェクトを探します。<br>
	 * nullを返すparentまで辿り着いた場合、nullを返却したコンポーネントを返却します。
	 * 
	 * <p>[備 考] </p>
	 * componentがJMenuItemのようなJPopupMenuコンポーネントの要素の場合、
	 * 戻り値はJPopupMenuを呼び出したコンポーネントが所属するウィンドウコンポーネント
	 * になります。
	 * 
	 * @param component ウィンドウを探す子コンポーネント
	 * @return componentが属するウィンドウコンポーネント
	 */
	public static Window searchWindowLevelObject(Component component) {
		while(!(component instanceof Window) &&
				component.getParent() != null) 
		{
			component = component.getParent();
		}
		
		// parent==nullを返却したのがJPopupMenuの場合、ポップアップ発生元ウィンドウを返却
		if(component instanceof JPopupMenu) {
			component = ((JPopupMenu)component).getInvoker();
			component = searchWindowLevelObject(component);
		}
		return (Window)component;
	}

	/**
	 * <p>[概 要] </p>
	 * 引数Componentが所属するウィンドウコンポーネントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * {@link #searchWindowLevelObject(Component)}に処理委譲します。
	 * <p/>
	 * 
	 * Componentの中には、getParent()で返却される親コンポーネントの参照が
	 * イベントディスパッチスレッド間で維持されないものが有ります。<br/>
	 * (TableCellEditorで生成されたセル内ボタン等)<br/>
	 * これらのコンポーネントの親を確実に取る為に第二引数parentが用意されています。<br/>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param component ウィンドウを探す子コンポーネント
	 * @param parent ウィンドウを探す子コンポーネントの親コンポーネント
	 * @return componentが属するウィンドウコンポーネント
	 */
	public static Window searchWindowLevelObject(Component component, Component parent) {
		if(component.getParent() == null && parent != null) {
			if(component instanceof Window) {
				return (Window)component;
			}
			component = parent;
		}
		
		return searchWindowLevelObject(component);
	}
}
