/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.controller;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.EventListener;
import java.util.EventObject;
import java.util.List;

/**
 * <p>[概 要] </p>
 * 汎用イベントハンドラクラスです。
 * 
 * <p>[詳 細] </p>
 * EventBinder#addEventBindingメソッドで登録されたリスナタイプの動的Proxyオブジェクトです。
 * 
 * 
 * <p>[備 考] </p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class EventInvocationHandler implements InvocationHandler {

	/** コントローラオブジェクトです。 */
	private AbstractController controller;
	
	/** このプロキシがハンドリングするイベントリスナの型です。 */
	private Class<? extends EventListener> listenerType;
	
	/** AbstractController#handlerFacadeに処理委譲するイベントタイプのリストです。 */
	private List<String> eventTypes;
	
	private static Method hashCodeMethod;
	private static Method equalsMethod;
	private static Method toStringMethod;

	static {
		try {
			hashCodeMethod = Object.class.getMethod("hashCode", null);
			equalsMethod = Object.class.getMethod("equals", new Class[] { Object.class });
			toStringMethod = Object.class.getMethod("toString", null);
		} catch (NoSuchMethodException e) {
			throw new NoSuchMethodError(e.getMessage());
		}
	}

    
	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コントローラオブジェクト
	 */
	public AbstractController getController() {
		return controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラオブジェクト
	 */
	public void setController(AbstractController controller) {
		this.controller = controller;
	}

	/**
	 * <p>[概 要] </p>
	 * このプロキシがハンドリングするイベントリスナの型を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * listenerTypeフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return このプロキシがハンドリングするイベントリスナの型
	 */
	public Class<? extends EventListener> getListenerType() {
		return listenerType;
	}

	/**
	 * <p>[概 要] </p>
	 * このプロキシがハンドリングするイベントリスナの型を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * listenerTypeフィールドを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param listenerType このプロキシがハンドリングするイベントリスナの型
	 */
	public void setListenerType(Class<? extends EventListener> listenerType) {
		this.listenerType = listenerType;
	}

	/**
	 * <p>[概 要] </p>
	 * handlerFacadeに処理委譲するイベントタイプメソッド名のリストを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * eventTypesフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return handlerFacadeに処理委譲するイベントタイプメソッド名のリスト
	 */
	public List<String> getEventTypes() {
		return eventTypes;
	}

	/**
	 * <p>[概 要] </p>
	 * handlerFacadeに処理委譲するイベントタイプメソッドのリストを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * eventTypesフィールドに引数eventTypesを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventTypes handlerFacadeに処理委譲するイベントタイプメソッド名のリスト
	 */
	public void setEventTypes(List<String> eventTypes) {
		this.eventTypes = eventTypes;
	}

	
	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数情報をフィールドに保存します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller
	 * @param listenerType
	 * @param eventTypes
	 */
	public EventInvocationHandler(AbstractController controller, 
								   Class<? extends EventListener> listenerType,
								   List<String> eventTypes) 
	{
		this.controller = controller;
		this.listenerType = listenerType;
		this.eventTypes = eventTypes;
	}

	/**
	 * <p>[概 要] </p>
	 * イベント発生時のハンドリングを行います。
	 * 
	 * <p>[詳 細] </p>
	 * 呼び出されるメソッドが、eventTypesリストに含まれるメソッド名である場合、
	 * {@link AbstractController#handlerFacade(EventObject, Class, String, EventListener)}
	 * を呼び出します。
	 * 
	 * handlerFacadeでは、譲渡された引数情報を元に、実行するべきアクションを決定します。
	 * 
	 * <p>[備 考] </p>
	 *
	 * @param eventListenerProxy イベントを受け取ったリスナプロキシ
	 * @param method 呼び出されるイベントハンドリングメソッド
	 * @param args イベントハンドリングメソッドの引数
	 */
	public Object invoke(Object eventListenerProxy, Method method, Object[] args) throws Throwable {
		Object result = null;

		Class declaringClass = method.getDeclaringClass();

		if (declaringClass == Object.class) {
			if (method.equals(hashCodeMethod)) {
				return proxyHashCode(eventListenerProxy);
			} else if (method.equals(equalsMethod)) {
				return proxyEquals(eventListenerProxy, args[0]);
			} else if (method.equals(toStringMethod)) {
				return proxyToString(eventListenerProxy);
			} else {
				throw new InternalError("unexpected Object method dispatched: " + method);
			}
		} else {
			if(this.eventTypes.contains(method.getName())) {
				this.controller.handlerFacade((EventObject)args[0], 
												this.listenerType, 
												method.getName(), 
												(EventListener)eventListenerProxy);
			}
		}
		return result;
	}
	
	/**
	 * <p>[概 要] </p>
	 * プロキシクラスのhashCodeメソッド実装です。
	 * 
	 * <p>[詳 細] </p>
	 * プロキシクラスがhashCodeメソッド実行要請を受けた時の処理を実装します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param proxy hashCodeメソッド実行要請を受けるプロキシ
	 * @return proxyのハッシュ値
	 */
	protected Integer proxyHashCode(Object proxy) {
		return new Integer(System.identityHashCode(proxy));
	}

	/**
	 * <p>[概 要] </p>
	 * equalsメソッド実装です。
	 * 
	 * <p>[詳 細] </p>
	 * プロキシクラスがequalsメソッド実行要請を受けた時の処理を実装します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param proxy equalsメソッド実行要請を受けるプロキシ
	 * @param other proxyと比較するオブジェクト 
	 * @return true : 同じオブジェクト、false : 違うオブジェクト
	 */
	protected Boolean proxyEquals(Object proxy, Object other) {
		return (proxy == other ? Boolean.TRUE : Boolean.FALSE);
	}

	/**
	 * <p>[概 要] </p>
	 * toStringメソッド実装です。
	 * 
	 * <p>[詳 細] </p>
	 * プロキシクラスがtoStringメソッド実行要請を受けた時の処理を実装します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param proxy toStringメソッド実行要請を受けるプロキシ
	 * @return proxyの文字列表現
	 */
	protected String proxyToString(Object proxy) {
		return proxy.getClass().getName() + '@' + Integer.toHexString(proxy.hashCode());
	}
}
