/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.rpc {
	
	import flash.events.IEventDispatcher;
	
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.error.CoreLogicError;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	import mx.rpc.events.FaultEvent;
	import mx.rpc.events.ResultEvent;
	
	/**
	 * <p>[概 要]</p>
	 * HTTP経由でRPC呼び出しを行う機能モデルの基底抽象モデルクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * 各種RPC通信クラスはこのクラスを継承して通信オブジェクト実体の生成、
	 * 通信成功、通信失敗ハンドラの実装を行います。<br>
	 * 実装クラスのHTTPServiceCore、WebServiceCoreでは、
	 * このクラスで定義されている返却フォーマットを設定することで、
	 * レスポンスデータの型を指定することが出来ます。<br>
	 * <ul>
	 *   <li>OBJECT</li>
	 *   <li>E4X</li>
	 *   <li>XML</li>
	 * </ul>
	 * OBJECTがresultFormatとして指定されていると、ActionScriptとJavaの間でデータ型変換が行われます。<br>
	 * ActionScriptとJavaの型変換については下記関連項目サイトを参照して下さい。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @see http://www.adobe.com/livedocs/flex/201_jp/html/wwhelp/wwhimpl/common/html/wwhelp.htm?context=LiveDocs_Book_Parts&file=ent_services_config_097_08.html
	 * @see jp.co.fujitsu.reffi.client.flex.model.rpc.HTTPServiceCore
	 * @see jp.co.fujitsu.reffi.client.flex.model.rpc.WebServiceCore
	 * @author Project Reffi
	 */ 
	public class RPCCore extends BaseModel {
		
		/** レスポンス返却フォーマット「object」を示す定数です. */
		public static const OBJECT:String = "object";
		
		/** レスポンス返却フォーマット「e4x」を示す定数です. */
		public static const E4X:String = "e4x";
		
		/** レスポンス返却フォーマット「xml」を示す定数です. */
		public static const XML:String = "xml";
		
		
		// 継承先で定義される、各種通信オブジェクトのI/F
		private var _service : IEventDispatcher;
		
		/**
		 * <p>[概 要]</p>
		 * 継承先で定義される、各種通信オブジェクトのI/Fです.
		 * 
		 * <p>[詳 細]</p>
		 * HTTPServiceCoreはHTTPServiceオブジェクトを、
		 * RemoteObjectCoreはRemoteObjectオブジェクトをこのプロパティに設定します。
		 * 
		 * <p>[備 考]</p>
		 *
		 */ 
		public function get service():IEventDispatcher{
			return this._service;
		}
		public function set service(service:IEventDispatcher):void{
			this._service = service; 
		}
		
		/**
		 * <p>[概 要]</p>
		 * メイン処理メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 通信オブジェクトの実体を生成するcreateCocreteService()を
		 * 呼び出し、生成されたオブジェクトに対して受信成功ハンドラ
		 * 「resultHandler」、受信失敗ハンドラ「faultHandler」を追加します。
		 * 
		 * <p>[備 考]</p>
		 */ 
		override protected function mainProc() : void{
			super.mainProc();
			
			// 具象サービスクラスインスタンスを生成
			this.service = createConcreteService();
			
			// 受信ハンドラを設定
			this.service.addEventListener(ResultEvent.RESULT, resultHandler);
			// 受信失敗ハンドラを設定
			this.service.addEventListener(FaultEvent.FAULT, faultHandler);
		}

		/**
		 * <p>[概 要]</p>
		 * 具象イベントディスパッチャを作成します.
		 * 
		 * <p>[詳 細]</p>
		 * このクラスを継承した場合は、必ずこのメソッドをオーバーライドし、通信オブジェクト実体を返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return 通信オブジェクト
		 */ 
		protected function createConcreteService() : IEventDispatcher {
			throw new CoreLogicError("EFC0001");
			return null;
		} 
		
		/**
		 * <p>[概 要]</p>
		 * 受信成功ハンドラです.
		 * 
		 * <p>[詳 細]</p>
		 * 受信成功処理用メソッド「onSuccess」のテンプレートコールを行い、
		 * モデル成功イベントModelProcessEvent.SUCCESSと、
		 * モデル完了イベントModelProcessEvent.FINISHEDを発行します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event 通信成功イベント
		 */ 
		protected final function resultHandler(event : ResultEvent) : void {
			
			// オーバーライド用テンプレートメソッドをコール
			onSuccess(event, this.parameterMapping);

			// モデル終了イベントを生成			
			var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
			// モデル終了の原因となったResultEventをModelFinishEventに格納
			modelSuccessEvent.cause = event;
			// モデル終了イベントをディスパッチしてコントローラにキャッチさせる		
			dispatchModelSuccess(modelSuccessEvent);

			// モデル完了イベントを生成			
			var modelFinishedEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
			// モデル終了の原因となったResultEventをModelFinishEventに格納
			modelFinishedEvent.cause = event;
			// モデル終了イベントをディスパッチしてコントローラにキャッチさせる		
			dispatchModelFinished(modelFinishedEvent);
		}

		/**
		 * <p>[概 要]</p>
		 * 受信成功処理用オーバーライドメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 汎用的な受信成功処理を記述したい場合、
		 * モデルを継承してこのメソッドをオーバーライドします。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event 通信成功イベント
		 * @param parameterMapping パラメータマップ
		 */ 
		protected function onSuccess(event:ResultEvent, parameterMapping:ParameterMapping):void{
		}
		
		/**
		 * <p>[概 要]</p>
		 * 受信失敗ハンドラです.
		 * 
		 * <p>[詳 細]</p>
		 * 受信失敗処理用メソッド「onFailure」のテンプレートコールを行い、
		 * モデル失敗イベントModelProcessEvent.FAILUREを発行します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event 通信失敗イベント
		 */ 
		protected final function faultHandler(event : FaultEvent) : void {
			
			// オーバーライド用テンプレートメソッドをコール
			onFailure(event, this.parameterMapping);

			// モデル終了イベントを生成			
			var modelFinishedEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
			// モデル終了の原因となったFaultEventをModelFinishEventに格納
			modelFinishedEvent.cause = event;
			// モデル終了イベントをディスパッチしてコントローラにキャッチさせる。		
			dispatchModelFinished(modelFinishedEvent);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 受信失敗処理用オーバーライドメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 汎用的な受信失敗処理を記述したい場合、
		 * モデルを継承してこのメソッドをオーバーライドします。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param event 通信失敗イベント
		 * @param parameterMapping パラメータマップ
		 */ 
		protected function onFailure(event:FaultEvent, parameterMapping:ParameterMapping):void{
		}
	}
}
