/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.manager{

	import flash.events.AsyncErrorEvent;
	import flash.events.SecurityErrorEvent;
	import flash.net.LocalConnection;
	import flash.utils.Dictionary;
	
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.localConnection.LocalConnectionReceiverCore;

	/**
	 * <p>[概 要]</p>
	 * LocalConnectionを内包するLocalConnectionReceiverCoreを管理するマネージャクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * localConnectionsNameをキーにしたマッププロパティ「localConnections」に、
	 * LocalConnectionインスタンスと、<br>
	 * イベントハンドリングを行うLocalConnectionReceiverCoreのインスタンスを保持します。<p>
	 * 
	 * localConnectionsプロパティの構造は以下のようになります。
	 * <pre>
	 * localConnections:Dictionary
	 *     "LocalConnectionName1" = localConnectionMap:Dictionary
	 *                                  "connection" = LocalConnectionインスタンス
	 *                                  "core" = LocalConnectionReceiverCore
	 *     "LocalConnectionName2" = localConnectionMap:Dictionary
	 *                                  "connection" = LocalConnectionインスタンス
	 *                                  "core" = LocalConnectionReceiverCore
	 *     "LocalConnectionName3" = localConnectionMap:Dictionary
	 *         :
	 *         :
	 * </pre>
	 * 1 LocalConnection接続につき、1個のLocalConnectionインスタンスと、1個のLocalConnectionReceiverCoreが保持されます。<br>
	 * 既に接続中のLocalConnectionNameを指定した場合、登録済みのLocalConnectionReceiverCoreに対して
	 * 新規LocalConnectionReceiverCoreのresultHandlerメソッドとfaultHandlerメソッドがaddEventListenerされます。<br>
	 * 
	 * <p>[備 考]</p>
	 * LocalConnectionReceiverCoreをconnectedモードfalseで実行させることでLocalConnectionの解除が行えますが、<br>
	 * LocalConnectionManager#unconnectを使用すると、任意のタイミングでのLocalConnection接続解除が行えます。
	 * 
	 * <p>Copyright (c) 2008-2010 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 * @see jp.co.fujitsu.reffi.client.flex.model.localConnection.LocalConnectionReceiverCore
	 */
	public class LocalConnectionManager {
		
		private var _localConnections:Dictionary = new Dictionary();

		/**
		 * <p>[概 要]</p>
		 * LocalConnectionのマッピング.
		 * 
		 * <p>[詳 細]</p>
		 * 現在接続中のLocalConnectionNameと、実際に接続処理を行うLocalConnectionインスタンス、
		 * LocalConnectionの接続結果を受信する為のイベントハンドラを持つLocalConnectionReceiverCore
		 * をマッピング管理します。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get localConnections():Dictionary {
			return this._localConnections;
		}
		public function set localConnections(localConnections:Dictionary) :void {
			this._localConnections = localConnections;
		}

		/**
		 * <p>[概 要]</p>
		 * コンストラクタです.
		 * 
		 * <p>[詳 細]</p>
		 * 本クラスはシングルトンクラスのため外部からの呼び出しは行えません。
		 * 
		 * <p>[備 考]</p>
		 * LocalConnectionManagerのインスタンスを取得する場合は、getInstance()を使用してください。
		 * 
		 * <p>
		 * @param blocker シングルトン生成用ブロックインスタンス
		 * @see #getInstance()
		 */
		public function LocalConnectionManager(blocker:LocalConnectionBlocker) {
			if (blocker == null) {
				throw Error("This class is singleton.");
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * インスタンスを取得します.
		 * 
		 * <p>[詳 細]</p>
		 * シングルトンインスタンスを取得します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return LocalConnectionManagerインスタンス
		 */
		public static function getInstance():LocalConnectionManager {
			return LocalConnectionBlocker.instance;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 指定したLocalConnection名称で接続しているか確認します.
		 * 
		 * <p>[詳 細]</p>
		 * LocalConnection管理オブジェクト「localConnections」から、引数localConnectionNameが
		 * 登録されているかどうか調べます。<br>
		 * localConnectionNameを接続しているLocalConnectionReceiverCoreが一つでも有った場合はtrue
		 * を返却します。
		 * <p>[備 考]</p>
		 * 
		 * @param localConnectionName 接続されているかどうか調べるLocalConnection名称
		 * @return 接続されている場合はtrue、それ以外はfalse
		 */
		public function isConnect(localConnectionName:String):Boolean {
			return this._localConnections.hasOwnProperty(localConnectionName);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 現在接続中のLocalConnection名称の配列を返却します.
		 * 
		 * <p>[詳 細]</p>
		 * LocalConnection名称管理プロパティ「localConnections」内のキー名配列を返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return 現在接続中のLocalConnection名称配列
		 */
		public function getConnectingLocalConnectionNames():Array {
			var localConnectionNames:Array = new Array();
			for(var localConnectionName:String in this._localConnections) {
				localConnectionNames.push(localConnectionName);
			}
			return localConnectionNames; 
		}
		
		/**
		 * <p>[概 要]</p>
		 * LocalConnectionの接続を開始します.
		 * 
		 * <p>[詳 細]</p>
		 * 未接続状態であった場合は新規にlocalConnectionを生成し、LocalConnectionReceiverCoreをクライアントとし接続します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param localConnectionName 接続を開始するlocalConnection名称
		 * @param localConnectionCore このマネージャに管理されるlocalConnectionReceiveCoreインスタンス
		 */
		public function connect(localConnectionName:String, localConnectionCore:LocalConnectionReceiverCore):void{
			var localConnectionMap:Dictionary = null;
			var localConnection:LocalConnection = null;

			if(!this._localConnections.hasOwnProperty(localConnectionName)) {
				// 新規接続の場合
				localConnection = new LocalConnection();
				localConnection.client = localConnectionCore;
				executeAllowDomain(localConnection, localConnectionCore);
				localConnection.addEventListener(SecurityErrorEvent.SECURITY_ERROR, localConnectionCore.faultHandler);
				localConnection.addEventListener(AsyncErrorEvent.ASYNC_ERROR, localConnectionCore.faultHandler);
				localConnection.connect(localConnectionName);

				// 接続したLocalConnection情報を保存
				localConnectionMap = new Dictionary();
				localConnectionMap["connection"] = localConnection;
				localConnectionMap["core"] = localConnectionCore;
				this._localConnections[localConnectionName] = localConnectionMap;
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * 許可するドメインをLocalConnectionに指定します.
		 * 
		 * <p>[詳 細]</p>
		 * 許可するドメインをLocalConnectionに指定します。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param localConnection LocalConnection
		 * @param localConnectionCore LocalConnectionReceiverCore
		 */
		private function executeAllowDomain(localConnection:LocalConnection, localConnectionCore:LocalConnectionReceiverCore):void{
			for each(var domain:String in localConnectionCore.allowDomain){
				if (localConnectionCore.https){
					localConnection.allowInsecureDomain(domain);
				} else{
					localConnection.allowDomain(domain);
				}
			}
		}

		/**
		 * <p>[概 要]</p>
		 * LocalConnectionの接続を解除します.
		 * 
		 * <p>[詳 細]</p>
		 * 引数localConnectionNameで接続中のLocalConnectionを解除します。<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param localConnectionName 接続解除するLocalConnection名称
		 */
		public function unconnect(localConnectionName:String):void {
			// 現在接続していないLocalConnection名称の場合は処理しない
			if(!this._localConnections.hasOwnProperty(localConnectionName)) {
				return;
			}

			// 指定されたLocalConnectionを取得し接続解除
			var localConnectionMap:Dictionary = this._localConnections[localConnectionName];
			var localConnection:LocalConnection = localConnectionMap["connection"];
			var localConnectionCore:LocalConnectionReceiverCore = localConnectionMap["core"];

			localConnection.removeEventListener(SecurityErrorEvent.SECURITY_ERROR, localConnectionCore.faultHandler);
			localConnection.removeEventListener(AsyncErrorEvent.ASYNC_ERROR, localConnectionCore.faultHandler);
			localConnection.close();
			delete this._localConnections[localConnectionName];

			// モデル終了イベント送信
			var modelFinishedEvent:ModelProcessEvent = 	new ModelProcessEvent(ModelProcessEvent.FINISHED);
			localConnectionCore.dispatchModelFinished(modelFinishedEvent);
		}
	}
}

import jp.co.fujitsu.reffi.client.flex.manager.LocalConnectionManager;

/**
 * シングルトンブロック用クラス
 */
class LocalConnectionBlocker {
	public static var instance:LocalConnectionManager = new LocalConnectionManager(new LocalConnectionBlocker());
	
	public function LocalConnectionBlocker() {
		/** */
	}
}