/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.model;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import jp.co.fujitsu.reffi.server.invoker.DefaultInvoker;

/**
 * <p>[概 要]</p>
 * 基底モデルクラス。
 * 
 * <p>[詳 細]</p>
 * {@link DefaultInvoker}にて呼び出されるモデルの基底クラスとなります。
 * <br />
 * 開発者は本クラスを継承したクラスにてロジックを開発することができます。
 * <br />
 * 以下はサンプルの実装例です。
 * 
 * <pre class="samplecode">
 *public class DemoModel extends AbstractModel {
 *	public void mainProcess() throws Exception {
 *		// リクエストから「key」に設定されている値を取得
 *		String value = getRequestParameter("key", String.class);
 *		// ループ処理を実行し、結果配列へ格納
 *		List&lt;String&gt; results = new ArrayList&lt;String&gt;();
 *		for (int i = 0; i < 10; i++) {
 *			results.add(value + "-" + String.valueOf(i));
 *		}
 *		// クライアントへ結果を送信するため、配列を設定
 *		setResponse(results);
 *	}
 *}
 * </pre>
 * <p>
 * サンプルではクライアントから送信されたパラメータ「key」を文字列で受け取り、
 * 結果として生成した配列に、取得した「key」パラメータのデータとループの回数
 * を結合した文字列を登録、配列をレスポンスとして登録しています。
 * </p>
 * <p>
 * この場合、{@link AbstractModel#setResponse}にて登録した変数「results」がクライアントへ
 * 送信される結果となります。実際の送信処理はモデルの呼び出し元にて行われます。
 * </p>
 * <p>
 * また、モデルのメイン処理({@link AbstractModel#mainProcess})を処理する前に何かしらの処理を
 * 行いたい場合は{@link AbstractModel#preProcess()}を実装することで対処が
 * できます。
 * </p>
 * <p>
 * 同様に{@link AbstractModel#postProcess()}にて{@link AbstractModel#mainProcess()}
 * を実行したあとの処理が実装可能です。
 * </p>
 * <p>
 * {@link AbstractModel#finalProcess()}は、処理の成功/失敗に関わらず、全ての
 * 処理が終了した際に呼び出されます。
 * </p>
 * 
 * <p>[備 考]</p>
 *
 * @author Project Reffi
 */
public abstract class AbstractModel implements Serializable {
	/** シリアルバージョン番号 */
	private static final long serialVersionUID = 4900328223908816747L;
	/** リクエスト情報 */
	private Map<Object, Object> _request = new HashMap<Object, Object>();
	/** レスポンス情報 */
	private Object _response;
	
	/**
	 * <p>[概 要]</p>
	 * リクエスト情報取得。
	 * 
	 * <p>[詳 細]</p>
	 * リクエスト情報を取得します。
	 * <br />
	 * リクエスト情報はモデルの呼び出し元クラスで生成されたものが設定されています。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @return マップ化されたリクエスト情報
	 */
	public Map<Object, Object> getRequest() {
		return this._request;
	}

	/**
	 * <p>[概 要]</p>
	 * リクエスト情報設定。
	 * 
	 * <p>[詳 細]</p>
	 * リクエスト情報を設定します。
	 * <br />
	 * リクエストの設定はモデルの呼び出し元クラスで行うため、モデル内で本メソッドを
	 * 利用することはありません。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @param request マップ化されたリクエスト情報
	 */
	public void setRequest(Map<Object, Object> request) {
		this._request = request;
	}
	
	/**
	 * <p>[概 要]</p>
	 * リクエストパラメータ情報取得。
	 * 
	 * <p>[詳 細]</p>
	 * 指定したキーに該当するリクエストパラメータ情報を取得します。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @param key リクエストパラメータキー
	 * @return 該当するリクエストパラメータ情報、またはnull
	 */
	public Object getRequestParameter(Object key) {
		if (this._request.containsKey(key)) {
			return this._request.get(key);
		}
		return null;
	}

	/**
	 * 
	 * <p>[概 要]</p>
	 * リクエストパラメータ情報取得。
	 *
	 * <p>[詳 細]</p>
	 * 指定したキーに該当するリクエストパラメータ情報を取得します。
	 *
	 * <p>[備 考]</p>
	 *
	 * @param key リクエストパラメータキー
	 * @param clazz 取得するパラメータ値のクラス
	 * @return 該当するリクエストパラメータ情報、またはnull
	 */
	public <T> T getRequestParameter(Object key, Class<T> clazz) {
		try {
			Object ret = getRequestParameter(key);
			return clazz.cast(ret);
		} catch (ClassCastException e) {
			return null;
		}
	}
	
	/**
	 * <p>[概 要]</p>
	 * レスポンス情報取得。
	 * 
	 * <p>[詳 細]</p>
	 * レスポンス情報を取得します。
	 * <br />
	 * 本メソッドはモデルの呼び出し元でレスポンスの送信処理を行うために利用する
	 * もので、モデル内で本メソッドを利用することはありません。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @return レスポンス情報
	 */
	public Object getResponse() {
		return this._response;
	}
	
	/**
	 * <p>[概 要]</p>
	 * レスポンス情報設定。
	 * 
	 * <p>[詳 細]</p>
	 * クライアントへ送信するレスポンス情報を設定します。
	 * <br />
	 * モデルではクライアントへのレスポンス処理は行わず、モデルの呼び出し元クラスで
	 * レスポンス処理を行います。
	 * <br />
	 * レスポンスが必要ない場合は本メソッドにてレスポンスを設定する必要はありません。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @param response レスポンス情報
	 */
	public void setResponse(Object response) {
		this._response = response;
	}
	
	/**
	 * <p>[概 要]</p>
	 * モデル実行。
	 * 
	 * <p>[詳 細]</p>
	 * モデルの実行を制御します。
	 * 
	 * <p>[備 考]</p>
	 * 本メソッドは殆どのケースにおいて開発者が実装することはありません。
	 *
	 * @throws Exception モデル実行例外
	 */
	public void run() throws Exception {
		try {
			if (!preProcess()) {
				return;
			}
			mainProcess();
			postProcess();
		} catch (Exception e) {
			throw trap(e);
		} finally {
			finalProcess();
		}
	}

	/**
	 * <p>[概 要]</p>
	 * 事前処理。
	 * 
	 * <p>[詳 細]</p>
	 * モデル実行における事前処理を行います。
	 * <br />
	 * モデルのメイン処理が実行する前に処理すべき内容を記述できます。
	 * 結果としてtrueを返すとメイン処理以降を実行し、falseを返すと事前処理のみでモデルの実行は終了します。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @return 処理結果
	 * @throws Exception 事前処理時の例外
	 */
	protected boolean preProcess() throws Exception {
		return true;
	}
	
	/**
	 * <p>[概 要]</p>
	 * メイン処理。
	 * 
	 * <p>[詳 細]</p>
	 * モデル実行におけるメイン処理を行います。
	 * 
	 * <p>[備 考]</p>
	 *
	 * @throws Exception メイン処理時の例外
	 */
	protected void mainProcess() throws Exception {
		/** */
	}

	/**
	 * <p>[概 要]</p>
	 * 事後処理。
	 * 
	 * <p>[詳 細]</p>
	 * モデル実行における事後処理を行います。
	 * 
	 * <p>[備 考]</p>
	 * 事後処理は{@link #mainProcess()}実行後に行われるもので
	 * {@link #finalProcess()}とは異なります。
	 *
	 * @throws Exception 事後処理時の例外
	 */
	protected void postProcess() throws Exception {
		/** */
	}
	
	/**
	 * <p>[概 要]</p>
	 * 最終処理。
	 * 
	 * <p>[詳 細]</p>
	 * モデル実行における最終処理を行います。
	 * 
	 * <p>[備 考]</p>
	 * 最終処理はモデルの各処理結果には依存せず、最終段階(finally)で必ず呼ばれる処理となります。
	 *
	 */
	protected void finalProcess() {
		/** */
	}

	/**
	 * <p>[概 要]</p>
	 * 例外ハンドリング。
	 * 
	 * <p>[詳 細]</p>
	 * 例外のハンドリングを行います。
	 * 
	 * <p>[備 考]</p>
	 * 本メソッドでは受け取った例外をログへエラーとして出力し、そのまま返すのみとなっています。
	 * その他実装をしたい場合は各実装モデルにて本メソッドを継承して行ってください。
	 * 
	 * @param e 例外情報
	 * @return スローする例外情報
	 */
	protected Exception trap(Exception e) {
		return e;
	}
}
