/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.controller;

import java.net.SocketException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import jp.co.fujitsu.reffi.client.nexaweb.action.AbstractAction;
import jp.co.fujitsu.reffi.client.nexaweb.action.Action;
import jp.co.fujitsu.reffi.client.nexaweb.action.BaseAction;
import jp.co.fujitsu.reffi.client.nexaweb.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.nexaweb.listener.ModelProcessListener;
import jp.co.fujitsu.reffi.client.nexaweb.model.BaseModel;
import jp.co.fujitsu.reffi.client.nexaweb.model.DefaultModel;
import jp.co.fujitsu.reffi.client.nexaweb.model.Model;
import jp.co.fujitsu.reffi.client.nexaweb.util.DialogUtil;
import jp.co.fujitsu.reffi.client.nexaweb.util.ElementSnapshot;
import jp.co.fujitsu.reffi.common.exception.CoreExceptionIF;
import jp.co.fujitsu.reffi.common.exception.CoreLogicException;
import jp.co.fujitsu.reffi.common.nexaweb.dialog.CoreDialog;
import jp.co.fujitsu.reffi.common.nexaweb.dialog.CoreDialog.MessageDialogType;

import com.nexaweb.client.ClientEvent;
import com.nexaweb.client.ClientSession;
import com.nexaweb.client.mco.AbstractMco;
import com.nexaweb.client.mco.Container;
import com.nexaweb.client.netservice.NetServiceException;
import com.nexaweb.util.Log;
import com.nexaweb.util.LogConsumer;
import com.nexaweb.util.LogFactory;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.DocumentRegistry;
import com.nexaweb.xml.Element;
import com.nexaweb.xml.events.ChangeRejectedException;
import com.nexaweb.xml.events.StructureChangeEvent;
import com.nexaweb.xml.events.StructureChangeListener;
import com.nexaweb.xml.xpath.XPathFactory;

/**
 * <p>[概 要] </p>
 * アクションとモデルを制御する基底コントローラクラスです。
 * 
 * <p>[詳 細] </p>
 * コントローラは大きく分けて二つの役割を担います。
 * <p>
 * １．【コンポーネントイベントとアクションの紐付け】
 * <p>
 * BaseControllerを継承したコントローラクラスで{@link #bind(EventBinder)}をオーバーライドする、<br>
 * 若しくはreffi-client-config.xmlを定義することでコンポーネントイベントとアクションの紐付けを行います。<br>
 * ui DOMに該当のコンポーネントが挿入された時、紐付けられた情報を元にイベント登録を行います。
 * <p>
 * 
 * ２．【イベント発生時の処理フロー形成】
 * <p>
 * １で登録されたイベントが発生した時、{@link #handlerFacade()}が全てのイベントハンドリングの
 * 入り口となります。<br>
 * イベント処理を委譲されたBaseControllerは以下の処理フローを作ります。
 * 
 * <p>
 * 
 * <p>
 * 	<table width="100%" border="1" style="border-collapse:collapse;">
 * 		<tr>
 * 			<td>実行順序</td>
 * 			<td>BaseAction</td>
 * 			<td>BaseController</td>
 * 			<td>BaseModel</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">1</td>
 * 			<td>　</td>
 * 			<td>{@link #handlerFacade()}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">2</td>
 * 			<td>　</td>
 * 			<td>　{@link #invoke(Class, ParameterMapping)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">3</td>
 * 			<td>　</td>
 * 			<td>　　{@link #runAction(Class, ParameterMapping)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">4</td>
 * 			<td>{@link BaseAction#run(ParameterMapping) run(ParameterMapping)}</td>
 * 			<td>　</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">5</td>
 * 			<td>　</td>
 * 			<td>　　{@link #runModels(List, ParameterMapping, int, ModelProcessEvent)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">5´</td>
 * 			<td>　</td>
 * 			<td>　　{@link #runModelsAndNoWait(List, ParameterMapping)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">6</td>
 * 			<td>　</td>
 * 			<td>　</td>
 * 			<td>{@link BaseModel#run() run()}</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">7</td>
 * 			<td>　</td>
 * 			<td>　　モデル処理管理リスナ（成功）</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">8</td>
 * 			<td>{@link BaseAction#successForward(int, Model, Object) successForward(int, Model, Object)}</td>
 * 			<td>　</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">7´</td>
 * 			<td>　</td>
 * 			<td>　　モデル処理管理リスナ（失敗）</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">8´</td>
 * 			<td>{@link BaseAction#failureForward(int, Model, Exception) failureForward(int, Model, Exception)}</td>
 * 			<td>　</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">9</td>
 * 			<td>　</td>
 * 			<td>　　{@link #trap(Throwable)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 		<tr>
 * 			<td align="center">10</td>
 * 			<td>　</td>
 * 			<td>　　{@link #handlerFinalize(ParameterMapping)}</td>
 * 			<td>　</td>
 * 		</tr>
 * 	</table>
 * この過程で呼ばれるアクション（run）とモデル（run）の中では別途処理フローが作られ、
 * 機能実装者にフックポイントを提供します。
 * 
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b><br>
 * ・典型的な具象コントローラ（BaseController継承クラス）
 * <pre class="samplecode">
 *    package demo.client.controller;
 *
 *    import jp.co.fujitsu.reffi.client.nexaweb.controller.BaseController;
 *    import jp.co.fujitsu.reffi.client.nexaweb.controller.EventBinder;
 *    import jp.co.fujitsu.reffi.client.nexaweb.controller.ParameterMapping;
 *    import demo.client.chat.action.ChatWindowClosingAction;
 *    import demo.client.chat.action.EnterChannelAction;
 *    import demo.client.chat.action.LeaveChannelAction;
 *    import demo.client.chat.action.RemarkSendAction;
 *    import demo.client.menu.action.ConsoleClearButtonAction;
 *    import demo.client.menu.action.LineupWindowsAction;
 *    import demo.client.menu.action.OpenChatAction;
 *    import demo.client.menu.action.OpenDemoErrorWindowAction;
 *    import demo.client.menu.action.OpenDemoMapWindowAction;
 *    import demo.client.menu.action.OpenDemoScrollWindowAction;
 *    import demo.client.menu.action.OpenFormAction;
 *    import demo.client.menu.action.OpenRmiWindowAction;
 *    import demo.client.menu.action.OpenServerPushAction;
 *    import demo.client.windowGrouping.action.OpenWindowGroupParentAction;
 *
 *    public class DemoController extends BaseController {
 *    
 *        &#064;Override
 *        protected void initialize() {
 *            // 初期化処理記述
 *        }
 *
 *        &#064;Override
 *        protected void bind(EventBinder eventBinder) {
 *            // メニュー画面アクション
 *            eventBinder.addEventBinding("menu.openChat", "onCommand", OpenChatAction.class);
 *            eventBinder.addEventBinding("menu.openForm", "onCommand", OpenFormAction.class);
 *            eventBinder.addEventBinding("menu.openWindowGroupParent", "onCommand", OpenWindowGroupParentAction.class);
 *            eventBinder.addEventBinding("menu.lineupWindows", "onCommand", LineupWindowsAction.class);
 *            eventBinder.addEventBinding("menu.openServerPush", "onCommand", OpenServerPushAction.class);
 *            eventBinder.addEventBinding("menu.demoScrollWindowButton", "onCommand", OpenDemoScrollWindowAction.class);
 *            eventBinder.addEventBinding("menu.demoMapWindowButton", "onCommand", OpenDemoMapWindowAction.class);
 *            eventBinder.addEventBinding("menu.demoTrafficWindowButton", "onCommand", OpenDemoErrorWindowAction.class);
 *            eventBinder.addEventBinding("menu.demoRmiWindowButton", "onCommand", OpenRmiWindowAction.class);
 *            eventBinder.addEventBinding("menu.consoleClearButton", "onCommand", ConsoleClearButtonAction.class);
 *        
 *            // チャットデモ画面アクション
 *            eventBinder.addEventBinding("chat.remarkSend", "onCommand", RemarkSendAction.class);
 *            eventBinder.addEventBinding("chat.enterChannel", "onCommand", EnterChannelAction.class);
 *            eventBinder.addEventBinding("chat.leaveChannel", "onCommand", LeaveChannelAction.class);
 *            eventBinder.addEventBinding("chat.chatWindow", "onClose", ChatWindowClosingAction.class);
 *        }
 *    
 *        &#064;Override
 *        protected void handlerFinalize(ParameterMapping mapping) {
 *            // 毎イベント最終処理記述
 *        }
 *        
 *        &#064;Override
 *        public void unload() {
 *		      // ブラウザが閉じるイベント処理を記述
 *        }
 *    }
 * </pre>
 *
 * ・具象コントローラの登録方法
 * <pre class="samplecode">
 *     &lt;xal xmlns="http://openxal.org/ui/java"&gt;
 *        &lt;mco:mco xmlns:mco="http://openxal.org/core/mco" id="controller"
 *            src="demo.client.controller.DemoController"/&gt; 
 *
 *        &lt;rootPane&gt;
 *            &lt;freePane height="768px" width="1024px"&gt; &lt;/freePane&gt;
 *            &lt;window height="300px" title="Reffi Functional Demo メニュー" width="300px" x="10px" y="10px"&gt;
 *                &lt;freePane&gt;
 *                    &lt;button height="25px" name="openChat" text="チャットアプリデモ" width="250px" x="10px"
 *                        y="10px"/&gt;
 *                    &lt;button name="openTableWindow" height="25px" text="テーブルデモ" width="250px" x="10px" y="50px"/&gt;
 *                &lt;/freePane&gt;
 *            &lt;/window&gt;
 *        &lt;/rootPane&gt;
 *    &lt;/xal&gt; 
 * </pre>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class BaseController extends AbstractMco {
	/** ハンドラエントリポイント */
	public static final String HANDLER_ENTRY_POINT = "mco:controller.handlerFacade()";
	/** イベント紐付けオブジェクトです。 */
	private EventBinder eventBinder;

	/** アプリ起動～終了まで存在するデータ保存領域です。 */
	private Map<Object, Object> permanent;

	/** バリデーションエラーが発生したエレメントのエラー前クローン保存領域です。 */
	private ElementSnapshot errorElementSnapshot;

	volatile Vector<String> invokeThreadIds = new Vector<String>();

	/** クライアントログ発行オブジェクトです。 */
	private Log clientLogger;

	/** フレームワークの挙動情報を保持するオブジェクトです。 */
	private ClientConfig clientConfig;

	/**
	 * <p>[概 要] </p>
	 * イベント紐付けオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * eventBinderフィールドオブジェクトを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return イベント紐付けオブジェクト
	 */
	public EventBinder getEventBinder() {
		return this.eventBinder;
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付けオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * eventBinderフィールドオブジェクトを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventBinder イベント紐付けオブジェクト
	 */
	public void setEventBinder(EventBinder eventBinder) {
		this.eventBinder = eventBinder;
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * permanentフィールドオブジェトを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return アプリ起動～終了まで存在するデータ保存領域
	 */
	public Map<Object, Object> getPermanent() {
		return this.permanent;
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * permanentフィールドオブジェクトを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param permanent アプリ起動～終了まで存在するデータ保存領域
	 */
	public void setPermanent(Map<Object, Object> Permanent) {
		this.permanent = Permanent;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションエラーが発生したエレメントのエラー前クローン保存領域を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * errorElementSnapshotフィールドオブジェクトを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return バリデーションエラーが発生したエレメントのエラー前クローン保存領域
	 */
	public ElementSnapshot getErrorElementSnapshot() {
		return errorElementSnapshot;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションエラーが発生したエレメントのエラー前クローン保存領域を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * errorElementSnapshotフィールドオブジェクトを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param errorElementSnapshot バリデーションエラーが発生したエレメントのエラー前クローン保存領域
	 */
	public void setErrorElementSnapshot(ElementSnapshot errorElementSnapshot) {
		this.errorElementSnapshot = errorElementSnapshot;
	}

	/**
	 * <p>[概 要] </p>
	 * クライアントログ発行オブジェトを取得します。
	 *  
	 * <p>[詳 細] </p>
	 * clientLoggerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return クライアントログ発行オブジェト
	 */
	public Log getClientLogger() {
		return clientLogger;
	}

	/**
	 * <p>[概 要] </p>
	 * クライアントログ発行オブジェトを設定します。
	 *  
	 * <p>[詳 細] </p>
	 * clientLoggerフィールドを引数clientLoggerで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientLogger
	 */
	public void setClientLogger(Log clientLogger) {
		this.clientLogger = clientLogger;
	}

	/**
	 * <p>[概 要] </p>
	 *  フレームワークの挙動情報を保持するオブジェクトを取得します。
	 *  
	 * <p>[詳 細] </p>
	 * clientConfigフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return フレームワークの挙動情報を保持するオブジェクト
	 */
	public ClientConfig getClientConfig() {
		return clientConfig;
	}

	/**
	 * <p>[概 要] </p>
	 *  フレームワークの挙動情報を保持するオブジェクトを設定します。
	 *  
	 * <p>[詳 細] </p>
	 * clientConfigフィールドを引数clientConfigで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientConfig フレームワークの挙動情報を保持するオブジェクト
	 */
	public void setClientConfig(ClientConfig clientConfig) {
		this.clientConfig = clientConfig;
	}

	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラの初期化を行います。
	 * <p>
	 * 	<UL>
	 * 		<LI>イベント紐付け情報オブジェクトの生成</LI>
	 * 		<LI>アプリ起動～終了まで存在するデータ保存領域の生成</LI>
	 * 		<LI>reffi-client設定情報保持領域の生成</LI>
	 * 		<LI>エレメントのエラー前クローン保存領域</LI>
	 * 	</UL>
	 * <p>
	 * を行った後、イベント紐付け登録読込みの為、{@link #bind(EventBinder)}メソッドを
	 * テンプレートコールします。
	 * 
	 * <p>[備 考] </p>
	 */
	public BaseController() {
		super();

		// イベント紐付けオブジェクトインスタンスを生成
		setEventBinder(new EventBinder(this));
		// MVCレイヤを循環するパラメータオブジェクトを生成
		setPermanent(Collections.synchronizedMap(new HashMap<Object, Object>()));
		// フレームワークの挙動設定クラスを生成
		setClientConfig(new ClientConfig());

		// エラー状態エレメントのエラー前状態を保存するオブジェクトを生成
		setErrorElementSnapshot(new ElementSnapshot());

		// 機能実装者が定義を行ったイベント紐付け情報を汲取る
		bind(getEventBinder());
	}

	/**
	 * <p>[概 要] </p>
	 * ui DOMに対して、イベントハンドラ属性の登録を行うエレメント挿抜監視リスナを登録します。
	 * 
	 * <p>[詳 細] </p>
	 * AbstractMco#addedToContainerをオーバーライドします。<br>
	 * Nexaweb RuntimeがAbstractMco#addedToContainerをコールし終えた時点で、<br>
	 * xalのDocument化、ClientSessionの生成が終了します。<br>
	 * <p>
	 * 
	 * このタイミングで、出来上がったDocumentに対してStructureChangeListenerを追加、
	 * NexawebによるDOMへのエレメント挿抜を監視します。
	 * 
	 * <p>
	 * 【エレメント挿入イベント時】<br>
	 * 追加エレメントのname属性を調べ、イベント紐付け情報（EventBinder）から登録されている
	 * イベントタイプ（onCommand等）を取り出します。<br>
	 * 登録イベントが有った場合は追加エレメントに属性追加します。<br>
	 * この際の属性値、即ちイベントハンドラはBaseController#handlerFacadeで統一されます。<br>
	 * 例）&lt;button onCommand="mco:controller.handlerFacade()"/&gt;<br>
	 * 
	 * <p>
	 * 【エレメント削除イベント時】
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	@Override
	public void addedToContainer(Container container, String name) {
		// super.addedToContainer(container, name)が実行終了した時点でui Doucmentが生成される
		super.addedToContainer(container, name);

		// ui Documentを取得
		ClientSession clientSession = getSession();
		DocumentRegistry registry = clientSession.getDocumentRegistry();
		Document document = registry.getUiDocument();

		// ui Documentに構造変更管理リスナを追加
		document.addStructureChangeListener(new StructureChangeListener() {
			// 構造追加イベントハンドラ
			public void beforeChildAdded(StructureChangeEvent evt)
					throws ChangeRejectedException {
				// 追加されたのがElementの場合
				if (evt.getChange() instanceof Element) {
					Element source = (Element) evt.getChange();
					// 追加されたエレメントの子要素を全て取得
					Vector<?> addedElements = XPathFactory.createXPath(
							"descendant-or-self::*").evaluate(source);

					for (int ii = 0; ii < addedElements.size(); ii++) {
						Element addedElement = (Element) addedElements.get(ii);

						if ("window".equals(addedElement.getLocalName())) {
							if (getClientConfig().isWindowElementDefaultFocus()) {
								addedElement.setAttribute("focused", "true");
							}
						}
						// 追加されたエレメントがname属性を持たない場合は処理しない
						String name = addedElement.getAttribute("name");
						if (name == null) {
							continue;
						}

						// name属性値を基にEventBinderから登録イベントタイプ群を取得
						String[] eventTypes = getEventBinder().getEventTypes(
								name);
						if (eventTypes == null) {
							continue;
						}

						// 登録イベントタイプ数分、ハンドラを登録
						for (int jj = 0; jj < eventTypes.length; jj++) {
							String eventType = eventTypes[jj];
							// ハンドラのエントリポイントは全てBaseController#handlerFacade()
							addedElement.setAttribute(eventType, HANDLER_ENTRY_POINT);
						}
					}
				}
			}

			// 構造削除イベントハンドラ
			public void beforeChildRemoved(StructureChangeEvent evt)
					throws ChangeRejectedException {
				// 削除されたのがElementの場合　現状処理がないためコメント
				/*
				if(evt.getChange() instanceof Element){
					Element source = (Element)evt.getChange();
					Vector<?> addedElements = XPathFactory.createXPath("descendant-or-self::*").evaluate(source);
					for(int ii=0; ii<addedElements.size(); ii++){
						Element addedElement = (Element)addedElements.get(ii);
						String name = addedElement.getAttribute("name");
						if(name == null){
							continue;
						}
					}
				}
				*/
			}

			public void onChildAdded(StructureChangeEvent arg0) {
			}

			public void onChildRemoved(StructureChangeEvent arg0) {
			}
		});

		// ClientSession生成、McoContainerへの登録が済んでから初期化処理挿入タイミングを作る
		initialize(getClientConfig());

		// 実装者が初期化処理を終えた後、ClientConfigオブジェクトに依存する初期化処理を行う
		postInitialize(getClientConfig());
	}

	/**
	 * <p>[概 要] </p>
	 * 初期化処理が記述可能なメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * Document、ClientSessionが生成されるタイミングでテンプレートコールされます。
	 * デフォルトの処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * 業務固有の初期化処理が必要な場合は、具象コントローラ内でこのメソッドを
	 * オーバーライドして下さい。
	 * 又、Reffiの挙動設定をこのタイミングで設定することが出来ます。<p>
	 * 
	 * <b>使用例）</b><br>
	 * <pre class="samplecode">
	 *	protected void initialize(ClientConfig config){
	 *		
	 *		// バリデーションエラー時に背景色とツールチップの変更を行わないようにする
	 *		config.setComponentColorAndTipChangeOnValidationFault(false);
	 *
	 *		// クライアントログをローカルファイルに残す（Applet実行時、要デジタル署名）
	 *		config.setEnableLogConsumer(true);
	 *		config.setLogConsumerClass(FileLogConsumer.class);
	 *		config.setLogFileDir("operation_log");
	 *		config.setLogFileNamePrefix("abc-system-client-");
	 *
	 *		// Reffiに「ウィンドウ」として認識させるエレメントを追加する
	 *		config.addWindowLevelElementDefinition("customWindow");
	 *	}
	 * </pre>
	 */
	protected void initialize(ClientConfig config) {
	}

	/**
	 * <p>[概 要] </p>
	 * ClientConfigの値に依存するクライアント初期化設定を行います。
	 * 
	 * <p>[詳 細] </p>
	 * ClientConfig#isEnableLogConsumer()の値を判断し、<br>
	 * trueであればClientConfig#getLogConsumerClass()で取得可能なLogConsumer実装をインスタンス化、<br>
	 * 「clientlogger」の名前で取得したログ発行オブジェクトのコンシューマとして設定します。<br>
	 * ログ発行オブジェクトはログレベルDEBUG設定で、clientLoggerフィールドに保存されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	private void postInitialize(ClientConfig config) {
		// ログコンシューマインスタンス生成
		LogConsumer logConsumer = null;

		Class<? extends LogConsumer> clazz = config.getLogConsumerClass();
		try {
			logConsumer = (LogConsumer) clazz.newInstance();
		} catch (Exception e) {
			e.printStackTrace();
		}

		// ログコンシューマ初期化
		logConsumer.init(config);

		// ログイベント発行オブジェクト生成、フィールドに保存
		Log clientLogger = LogFactory.getLog("clientlogger");
		clientLogger.setLevel(Log.DEBUG);
		clientLogger.addLogConsumer(logConsumer);
		setClientLogger(clientLogger);
	}

	/**
	 * <p>[概 要] </p>
	 * 全ユーザ定義イベントをハンドルする入り口になるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 発生したイベントタイプ、イベントを起こしたコンポーネント名を元に、<br>
	 * イベント紐付けオブジェクト（EventBinder）から対応するアクションクラス型を取得します。<br>
	 * 取得したアクションクラス型を引数にして{@link #invoke(Class, ParameterMapping)}に処理委譲します。
	 * 
	 * <p>[備 考] </p>
	 * {@link #bind(EventBinder)}で定義したイベントは全てこのメソッドがハンドリングします。
	 * MCOを追加開発していくスタイルの場合、UIからのイベントハンドリングエントリポイントは
	 * 分散されますが、Reffiを使用したアプリケーションでは統一されます。
	 * 
	 */
	public void handlerFacade() {

		// MVC各レイヤを伝播するパラメターオブジェクトを生成、初期化
		final ParameterMapping parameterMapping = createParameterMapping();

		// イベントタイプとコンポーネント名を元にEventBinderから登録アクションクラスを取得
		final Class<? extends Action> actionClass = getEventBinder()
				.getActionClass(parameterMapping.getEventSourceElementName(), 
						parameterMapping.getEventType());

		// 二度押し制御を判定して処理フロー構築メソッド「invoke」開始
		if (getClientConfig().isDuplicateActionInvoke()) {
			invoke(actionClass, parameterMapping);
		} else {
			synchronized (this.invokeThreadIds) {
				if (!this.invokeThreadIds.contains(parameterMapping.getEventSourceElementId())) {
					this.invokeThreadIds.add(parameterMapping.getEventSourceElementId());
					getSession().getUiUpdateQueue().invokeLater(new Runnable() {
						public void run() {
							invoke(actionClass, parameterMapping);
						}
					});
				}
			}
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * 多重起動抑制管理から対象となるエレメントIDを削除します。
	 * 
	 * <p>[詳 細] </p>
	 * 多重起動抑制を設定している場合のみ、内部にて処理が実行されます。
	 * <br />
	 * これは{@link #invoke(Class, ParameterMapping)}内にてActionの途中終了、
	 * Action実行時の例外発生、及びモデルの終了イベント時に共通で呼び出される必要があります。
	 * 
	 * <p>[備 考] </p>
	 * {@link #invoke(Class, ParameterMapping)}のfinallyでは呼び出しは行いません。
	 * <br />
	 * これはモデルがパラレルに実行されている際、モデルの終了を待たずにActionは終了してしまい、
	 * モデル実行中にActionが多重起動されてしまう恐れがあるためです。
	 * 
	 * @param elementId 多重起動抑制対象となるエレメントID
	 * 
	 */
	protected final void removeInvokeThreadId(final String elementId) {
		// 多重起動抑制が設定されている場合のみ処理を実施
		if (!getClientConfig().isDuplicateActionInvoke()) {
			getSession().getUiUpdateQueue().invokeLater(new Runnable() {
				public void run() {
					// 多重起動抑制一覧からエレメント名を削除
					invokeThreadIds.remove(elementId);
				}
			});
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * MVC各レイヤを巡回するParameteraMappingオブジェクトを生成、初期化します。
	 * 
	 * <p>[詳 細] </p>
	 * イベント発生の際、Controller、Action、Modelを流れるデータマップを作成します。<br>
	 * このメソッドによって、以下の情報がParameterMappingオブジェクトに設定されます。<p>
	 * <ul>
	 * 		<li>Nexaweb ClientSession</li>
	 * 		<li>画面レンダリングされているui Document</li>
	 * 		<li>イベントを発生させたエレメントインスタンス</li>
	 * 		<li>発生したイベントタイプ</li>
	 * 		<li>イベントを発生させたエレメントのname属性値</li>
	 * 		<li>ClientEventパラメータ</li>
	 * 		<li>ClientEventクエリ文字列</li>
	 * </ul>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return MVC各レイヤを巡回するパラメータオブジェクト
	 */
	@SuppressWarnings("unchecked")
	protected ParameterMapping createParameterMapping() {
		// 返却用パラメータマッピングオブジェクトを生成
		ParameterMapping ret = new ParameterMapping();

		// Nexawebクライアントセッションオブジェクト取得
		ClientSession clientSession = this.getSession();
		// 画面にレンダリングされているUI DOMオブジェクトを取得
		Document uiDocument = clientSession.getDocumentRegistry()
				.getUiDocument();
		
		ClientEvent clientEvent = clientSession.getEventHandler().getClientEvent();
		// イベントソースエレメント、イベント発生タイプ、エレメントのname属性を取得
		Element eventSourceElement = (Element) clientEvent.getSource();
		String eventType = clientEvent.getParameter("event");
		String componentName = eventSourceElement.getAttribute("name");
		String componentId = eventSourceElement.getAttribute("id");
		
		// ClientEvent内パラメータを複製
		// ＊Nexaweb 4.5.20から、任意起動スレッドからClientEventが取れない為、
		// このタイミングでparameterMappingに保持
		Map<String, String> clientEventParameters = new HashMap<String, String>();
		Enumeration<String> parameterNames = clientEvent.getParameterNames();
		while (parameterNames.hasMoreElements()) {
			String name = parameterNames.nextElement();
			String value = clientEvent.getParameter(name);
			clientEventParameters.put(name, value);
		}
		// ClientEvent内パラメータと同様に複製
		String queryString = clientEvent.getQueryString();

		// 引数＋取得した情報をマッピング
		ret.setEventSourceElement(eventSourceElement);
		ret.setClientSession(clientSession);
		ret.setUiDocument(uiDocument);
		ret.setEventType(eventType);
		ret.setEventSourceElementName(componentName);
		ret.setEventSourceElementId(componentId);
		ret.setClientEventParameters(clientEventParameters);
		ret.setQueryString(queryString);
		ret.setRunModelsAndNoWait(false);
		// ClientEventの設定
		// 但し、これはClientConfig#setDuplicateActionInvoke(boolean)でfalse
		// を設定している場合のみ、getClientEventでClientEvent取得が行える。
		ret.setClientEvent(clientEvent);

		return ret;
	}

	/**
	 * <p>[概 要]</p>
	 * コントローラの主幹メソッドです。
	 * 
	 * <p>[詳 細]</p>
	 * コントローラ処理フローの幹を形成します。
	 * このメソッドのtryスコープで以下が行われます。
	 * 	<ol>
	 * 		<li>{@link #createParameterMapping()} MVCレイヤを巡回するParameterMappingオブジェクトの生成</li>
	 * 		<li>{@link #runAction(Class, ParameterMapping)} アクションの実行</li>
	 * 		<li>{@link #runModels(List, ParameterMapping, int, ModelProcessEvent)} or <br>
	 *          {@link #runModelsAndNoWait(List, ParameterMapping)} アクションで予約されたモデル群の実行
	 *      </li>
	 * 	</ol>
	 * 
	 * 上記の処理中に例外が発生した場合、trapメソッドがテンプレートコールされます。
	 * 最終的にfinallyスコープに入るとhanderFinalizeメソッドがテンプレートコールされます。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param event 画面コンポーネントから発生したイベントオブジェクト
	 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
	 */
	public final Object invoke(Class<? extends Action> actionClass,
			ParameterMapping parameterMapping) {

		// 具象イベントハンドラ実行後の戻り値（返しても多分使えない）
		Object ret = null;
		// Actionが途中終了した際の多重起動抑制エレメントID
		String elementId = parameterMapping.getEventSourceElementId();

		try {
			// アクセスログ採取
			if (getClientLogger() != null) {
				getClientLogger().log(Log.INFO, actionClass.getName());
			}

			// Action実行
			parameterMapping = runAction(actionClass, parameterMapping);
			if (parameterMapping == null) {
				// Actionが途中で終了するため、処理終了前に多重起動抑制から削除
				removeInvokeThreadId(elementId);
				return null;
			}

			// Actionで予約されたモデルクラス群を取得
			List<Class<? extends Model>> modelClasses = parameterMapping.getModelClasses();
			if (modelClasses.size() == 0) {
				modelClasses.add(DefaultModel.class);
			}
			parameterMapping.setModelReservedNum(modelClasses.size());
			parameterMapping.setModelFinishedNum(0);
			// Model群実行
			if (parameterMapping.isRunModelsAndNoWait()) {
				// 待たずに次Model実行
				runModelsAndNoWait(modelClasses, parameterMapping);
			} else {
				// Modelの戻り値を待って次Model実行
				runModels(modelClasses, parameterMapping, 0, null);
			}
		} catch (Throwable e) {
			// 例外が発生したため、多重起動抑制から削除
			removeInvokeThreadId(elementId);
			// 共通例外処理
			trap(e);
		} finally {
			// 共通終了処理
			handlerFinalize(parameterMapping);
		}
		return ret;
	}

	/**
	 * <p>[概 要] </p>
	 * イベントに対応するアクションを実行します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数で指定されたアクションクラス型をインスタンス化、
	 * 実行（{@link BaseAction#run(ParameterMapping)}）します。<br>
	 * execute実行後、結果戻り値であるParameterMappingを返却します。
	 * 
	 * <p>[備 考] </p>
	 * この結果がnullだった場合はコントローラの以降の処理は中止されます。
	 * {@link BaseAction#run(ParameterMapping)}がnullを返却するのは、
	 * prepareがfalseを返却、もしくはvalidate結果がエラーだった場合があります。
	 * 
	 * @param actionClass 実行するアクションクラスの型
	 * @param parameterMapping MVCを巡回するパラメータマッピング
	 * @return アクション実行後のParameterMappingオブジェクト
	 * @throws Exception
	 */
	public ParameterMapping runAction(Class<? extends Action> actionClass,
			ParameterMapping parameterMapping) throws Exception {
		// アクションを実行
		AbstractAction action = (AbstractAction) actionClass.newInstance();
		action.setController(this);
		parameterMapping = action.run(parameterMapping);

		// Action#runの結果がnull（prepareがfalseを返却、又はvalidationFault時）
		// であれば以降の処理は中止
		if (parameterMapping == null) {
			return null;
		}
		parameterMapping.setActionInstance(action);

		return parameterMapping;
	}

	/**
	 * <p>[概 要] </p>
	 * アクションで予約されたモデル郡をインスタンス化して連続実行します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link BaseAction#reserveModels(List)}で予約されたBaseModel実装モデル群を実行します。<br>
	 * 登録モデルが複数有る場合、前回モデルの実行終了を待ってから次回モデルが実行されます。<p>
	 * 
	 * モデル実行直前に{@link BaseAction#nextModel(int, ModelProcessEvent, Model)}がコールバックされます。<br>
	 * 実行モデルへのパラメータ設定を上記メソッドで行うことが出来ます。<p>
	 * 
	 * モデル実行後、成功時は{@link BaseAction#successForward(int, Model, Object)}が、
	 * 失敗時は{@link BaseAction#failureForward(int, Model, Exception)}がコールバックされます。<br>
	 * モデルの実行結果は上記メソッドで取得することが出来ます。
	 * 
	 * <p>[備 考] </p>
	 * {@link BaseAction#isRunModelsAndNoWait()}
	 * がfalseの場合、同期モードで実行されます。<br>
	 * 同メソッドはデフォルトでfalseを返却します。
	 * 
	 * @param modelClasses {@link BaseAction#reserveModels(List)} で予約されたモデルクラス群
	 * @param parameterMapping MVC各レイヤを伝播するパラメータ
	 * @param executeIndex モデル実行順序インデックス
	 * @param modelProcessEvent 直前に実行したモデルの処理結果イベントオブジェクト
	 * @throws Exception モデル内で発生し得る例外
	 */
	public void runModels(final List<Class<? extends Model>> modelClasses,
			ParameterMapping parameterMapping, int executeIndex,
			ModelProcessEvent modelProcessEvent) throws Exception {

		// コールバック用に事前実行されたActionインスタンスを取得
		BaseAction action = (BaseAction) parameterMapping.getActionInstance();

		if (modelClasses.size() > 0) {
			Class<? extends Model> modelClass = modelClasses.remove(0);
			// モデルクラスからインスタンス生成
			final BaseModel model = (BaseModel) modelClass.newInstance();
			// パラメータマッピングをモデルに設定
			model.setParameterMapping(parameterMapping);
			// コントローラインスタンスをモデルに設定
			model.setController(this);
			// モデル実行順序をモデルに設定
			model.setExecuteIndex(executeIndex);
			// モデル処理監視リスナをモデルに追加
			model.addModelProcessListener(new ModelProcessListener() {
				// モデル処理成功
				public void modelSuccess(ModelProcessEvent evt) {
					BaseModel model = (BaseModel) evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction) parameterMapping.getActionInstance();
					int executeIndex = model.getExecuteIndex();
					try {
						// BaseAction#successForwardをコールバック。
						// モデル結果をActionレイヤで取得可能にする。
						action.successForward(executeIndex, model, evt
								.getResult());

						if (model.getSuccessCount() == 1) {
							runModels(modelClasses, parameterMapping,
									++executeIndex, evt);
						}
					} catch (Exception e) {
						// 例外が発生したため、多重起動抑制から削除
						removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						trap(e);
					}
				}

				// モデル処理失敗
				public void modelFailure(ModelProcessEvent evt) {
					BaseModel model = (BaseModel) evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction)parameterMapping.getActionInstance();
					try {
						// BaseAction#failureForwardをコールバック。
						// モデル失敗例外をActionレイヤで取得可能にする。
						Exception e = action.failureForward(model
								.getExecuteIndex(), model, evt.getException());
						// BaseAction#failureForwardがnullを返却した場合、
						// コントローラによる例外処理は行わない。
						if (e != null) {
							trap(e);
						}
					} catch (Exception e) {
						trap(e);
					}
					// エラーのため、多重起動抑制から削除
					removeInvokeThreadId(parameterMapping.getEventSourceElementId());
				}
				
				// モデル終了処理
				public void modelFinished(ModelProcessEvent evt) {
					BaseModel model = (BaseModel)evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction)parameterMapping.getActionInstance();
					try {
						int finishedNum = parameterMapping.getModelFinishedNum();
						parameterMapping.setModelFinishedNum(finishedNum + 1);
						if (parameterMapping.getModelReservedNum() == parameterMapping.getModelFinishedNum()) {
							// アクションの全モデル終了通知メソッドをテンプレートコール
							action.complete();
							// モデルの完了カウントをクリア
							parameterMapping.setModelFinishedNum(0);
							// 多重起動対象から削除
							removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						}
					} catch (Exception e) {
						// 例外が発生したため、多重起動抑制から削除
						removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						trap(e);
					}
				}
			});
			// モデル実行直前にActionのメソッドをコールバック。モデルインスタンス設定フックタイミングを作る。
			boolean isProceed = action.nextModel(executeIndex,
					modelProcessEvent, model);
			if (!isProceed) {
				// モデルの前処理でfalseが返されてしまうとイベントが発行されないため、
				// 処理終了前に終了イベントを発行
				model.fireModelFinished(new ModelProcessEvent(model));
				return;
			}
			if (model.isSkip()) {
				// モデルがSkipされてしまうとイベントが発行されないため、
				// 次のモデル実行前に終了イベントを発行
				model.fireModelFinished(new ModelProcessEvent(model));
				// 次のモデルへ
				runModels(modelClasses, parameterMapping, ++executeIndex, null);
			} else {
				// nextModelでsetSkip(true)されていなければモデル実行
				model.run();
			}
		} else {
			action.nextModel(executeIndex, modelProcessEvent, null);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * アクションで予約されたモデル郡をインスタンス化して連続実行します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link BaseAction#reserveModels(List)}で予約されたBaseModel実装モデル群を実行します。<br>
	 * 登録モデルが複数有る場合、前回モデルの実行終了を待たずに次回モデルが実行されます。<br>
	 * 
	 * モデル実行直前に{@link BaseAction#nextModel(int, ModelProcessEvent, Model)}がコールバックされます。<br>
	 * 実行モデルへのパラメータ設定を上記メソッドで行うことが出来ます。<br>
	 * 非同期モードでモデル実行した場合は前回モデルの結果を待たずに次回モデルを実行する為、
	 * 第二引数prev:ModelProcessEventが常時nullになります。<br>
	 * 前回モデルの結果を判断して、次回モデルのパラメータ設定をすることは出来ません。<p>
	 * 
	 * モデル実行後、成功時は{@link BaseAction#successForward(int, Model, Object)}が、
	 * 失敗時は{@link BaseAction#failureForward(int, Model, Exception)}がコールバックされます。<br>
	 * モデルの実行結果は上記メソッドで取得することが出来ます。
	 * 
	 * <p>[備 考] </p>
	 * {@link BaseAction#isRunModelsAndNoWait()}
	 * がtrueの場合、非同期モードで実行されます。<br>
	 * 非同期モードで実行する場合は、{@link BaseAction#isRunModelsAndNoWait()}を
	 * オーバーライドしてtrueを返却して下さい。<br>
	 * 
	 * <pre class="samplecode">
	 *	  &#064;Override
	 *    protected boolean isRunModelsAndNoWait() {
	 *       return true;
	 *    }
	 * </pre>
	 * 
	 * @param modelClasses {@link BaseAction#reserveModels(List)} で予約されたモデルクラス群
	 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
	 * @throws Exception モデル内で発生し得る例外
	 */
	public void runModelsAndNoWait(List<Class<? extends Model>> modelClasses,
			ParameterMapping parameterMapping) throws Exception {

		// コールバック用に事前実行されたActionインスタンスを取得
		BaseAction action = (BaseAction) parameterMapping.getActionInstance();

		// コントローラが実行するモデルの実行順位
		int executeIndex = 0;
		// 全モデルクラス群を実行

		for (; executeIndex < modelClasses.size(); executeIndex++) {
			Class<? extends Model> modelClass = modelClasses.get(executeIndex);
			// モデルクラスからインスタンス生成
			final BaseModel model = (BaseModel) modelClass.newInstance();
			// パラメータマッピングをモデルに設定
			model.setParameterMapping(parameterMapping);
			// コントローラインスタンスをモデルに設定
			model.setController(this);
			// モデル実行順序をモデルに設定
			model.setExecuteIndex(executeIndex);
			// モデル処理監視リスナをモデルに追加
			model.addModelProcessListener(new ModelProcessListener() {
				// モデル処理成功
				public void modelSuccess(ModelProcessEvent evt) {
					BaseModel model = (BaseModel) evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction)parameterMapping.getActionInstance();
					try {
						// BaseAction#successForwardをコールバック。
						// モデル結果をActionレイヤで取得可能にする。
						action.successForward(model.getExecuteIndex(), model,
								evt.getResult());
					} catch (Exception e) {
						// 例外が発生したため、多重起動抑制から削除
						removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						trap(e);
					}
				}

				// モデル処理失敗
				public void modelFailure(ModelProcessEvent evt) {
					BaseModel model = (BaseModel) evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction)parameterMapping.getActionInstance();
					try {
						// BaseAction#failureForwardをコールバック。
						// モデル失敗例外をActionレイヤで取得可能にする。
						Exception e = action.failureForward(model
								.getExecuteIndex(), model, evt.getException());
						// BaseAction#failureForwardがnullを返却した場合、
						// コントローラによる例外処理は行わない。
						if (e != null) {
							trap(e);
						}
					} catch (Exception e) {
						trap(e);
					}
					// 多重起動対象から削除
					removeInvokeThreadId(parameterMapping.getEventSourceElementId());
				}

				// モデル終了処理
				public void modelFinished(ModelProcessEvent evt) {
					BaseModel model = (BaseModel)evt.getSource();
					ParameterMapping parameterMapping = model.getParameterMapping();
					BaseAction action = (BaseAction)parameterMapping.getActionInstance();
					try {
						int finishedNum = parameterMapping.getModelFinishedNum();
						parameterMapping.setModelFinishedNum(finishedNum + 1);
						if (parameterMapping.getModelReservedNum() == parameterMapping.getModelFinishedNum()) {
							// アクションの全モデル終了通知メソッドをテンプレートコール
							action.complete();
							// モデルの完了カウントをクリア
							parameterMapping.setModelFinishedNum(0);
							// 多重起動対象から削除
							removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						}
					} catch (Exception e) {
						// 多重起動対象から削除
						removeInvokeThreadId(parameterMapping.getEventSourceElementId());
						trap(e);
					}
				}
			});
			// モデル実行直前にActionのメソッドをコールバック。モデルインスタンス設定フックタイミングを作る。
			boolean isProceed = action.nextModel(executeIndex, null, model);
			if (!isProceed) {
				// モデルの前処理でfalseが返されてしまうとイベントが発行されないため、
				// 処理終了前に終了イベントを発行
				model.fireModelFinished(new ModelProcessEvent(model));
				return;
			}
			if (model.isSkip()) {
				// モデルがSkipされてしまうとイベントが発行されないため、
				// 次のモデル実行前に終了イベントを発行
				model.fireModelFinished(new ModelProcessEvent(model));
			} else {
				// nextModelでsetSkip(true)されていなければモデル実行
				model.run();
			}
		}
		action.nextModel(executeIndex, null, null);
	}

	/**
	 * <p>[概 要] </p>
	 * モデルを単体実行します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param model 実行するモデルインスタンス
	 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
	 * @return 実行されたモデルの汎用処理結果
	 * @throws Exception
	 */
	public Object runModel(Model model, ParameterMapping parameterMapping)
			throws Exception {
		((BaseModel) model).setParameterMapping(parameterMapping);
		((BaseModel) model).setController(this);
		model.run();
		Object result = model.getResult();

		return result;
	}

	/**
	 * <p>[概 要] </p>
	 * モデルを単体実行します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param model 実行するモデルインスタンス
	 * @param listener モデル処理結果リスナ
	 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
	 * @throws Exception
	 */
	public void runModelWithProcessListener(Model model,
			ModelProcessListener listener, ParameterMapping parameterMapping)
			throws Exception {
		((BaseModel) model).setParameterMapping(parameterMapping);
		((BaseModel) model).setController(this);
		model.addModelProcessListener(listener);
		model.run();
	}

	/**
	 * <p>[概 要] </p>
	 * MVC各レイヤで発生した例外が最終的にハンドリングされるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e MVC各レイヤで発生したスロー可能オブジェクト
	 */
	protected void trap(Throwable e) {
		// エラーログ採取
		if (getClientLogger() != null) {
			getClientLogger().log(Log.EXCEPTION, e);
		}

		// サーバ接続エラーの場合
		if (recursiveCheckConnectError(e)) {
			e = new CoreLogicException("EFC1007", e);
		}

		if (e instanceof CoreExceptionIF) {
			//想定例外時はエラーコードとメッセージでダイアログを出力
			if (((CoreExceptionIF) e).isNotifyToUser()) {
				String id = ((CoreExceptionIF) e).getId();
				String message = ((CoreExceptionIF) e).getMessage();

				errorDialog(id, message);
			}
		} else {
			//予期せぬエラー時はコンフィグ設定されたタイトルとメッセージでダイアログを出力
			ClientConfig config = getClientConfig();
			errorDialog(config.getUnexpectedErrorDialogTitle(), config
					.getUnexpectedErrorDialogMessage());
		}
	}

	/**
	 * <p>[概 要] </p>
	 * エラーダイアログ表示を行います。
	 * 
	 * <p>[詳 細]</p>
	 *  指定されたタイトルとメッセージでダイアログ表示を行います。
	 *  
	 * <p>[備 考] </p>
	 * 
	 * @param title エラーダイアログ タイトル
	 * @param message エラーダイアログ 表示メッセージ
	 */
	protected void errorDialog(String title, String message) {
		Document document = getSession().getDocumentRegistry().getUiDocument();

		CoreDialog cd = new CoreDialog.Builder(title, message).dialogType(
				MessageDialogType.ERROR.getType()).build();

		DialogUtil.messageAlert(document, cd);
	}

	/**
	 * <p>[概 要] </p>
	 * 発生した例外が原因がサーバ接続エラーかを判定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param e 発生例外
	 * @return 判定結果 true:サーバ接続エラー false:その他のエラー
	 */
	private boolean recursiveCheckConnectError(Throwable e) {

		boolean result = false;
		if (e instanceof SocketException) {
			result = true;
		} else if (e instanceof NetServiceException
				&& ((NetServiceException) e).getCausalThrowable() != null) {
			result = recursiveCheckConnectError(((NetServiceException) e)
					.getCausalThrowable());
		} else if (e.getCause() != null) {
			result = recursiveCheckConnectError(e.getCause());
		}
		return result;
	}

	/**
	 * <p>[概 要]</p>
	 * xalで定義されたエレメントと、アクションを紐付けるメソッドです。
	 * 
	 * <p>[詳 細]</p>
	 * 
	 * <p>[備 考]</p>
	 * eventBinder.addEventBinding("name属性値", "イベントタイプ", BaseAction継承クラス);<BR>
	 * のように紐付け処理を列挙して下さい。<BR>
	 * 
	 * ex.) eventBinder.addEventBinding("openTableWindow", "onCommand", OpenTableWindowAction.class);
	 * 
	 * @param eventBinder イベント紐付けオブジェクト
	 */
	protected void bind(EventBinder eventBinder) {
		// イベントバインド用XMLの読込み
		eventBinder.loadConfig("reffi-client-config.xml");
	}

	/**
	 * <p>[概 要] </p>
	 * 各ユーザ定義イベントハンドリングの最後にテンプレートコールされるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * デフォルト処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * このメソッドを具象コントローラでオーバーライドすると、全イベントアクションの
	 * 共通最終処理を実装出来ます。
	 * 
	 * @param mapping MVC各レイヤを伝播するパラメータオブジェクト
	 */
	protected void handlerFinalize(ParameterMapping mapping) {
	}
}
