/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.validator {

	import mx.validators.StringValidator;
	import mx.validators.ValidationResult;
	import jp.co.fujitsu.reffi.client.flex.util.CharTypeUtil;

	/**
	 * <p>[概 要]</p>
	 * 入力文字の文字種をチェックするクラスです.
	 *
	 * <p>[詳 細]</p>
	 * 呼び出し元で指定された入力チェック条件に従い、入力文字のチェック処理を実施します。
	 *
	 * <p>[備 考]</p>
	 *
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */
	public class CharacterTypeValidator extends StringValidator {

		private var results:Array;
		
		private static const ERROR_MESSAGE:String="入力文字が無効です。";

		// 半角数字(0-9)の入力可否
		private var _type_ascii_number:Boolean=true;
		
		// 半角英字(a-z,A-Z)の入力可否
		private var _type_ascii_alphabet:Boolean=true;
		
		// 半角記号(!"#$%&'()*+,-./:;<=>?@[\]^_`{|}~)の入力可否
		private var _type_ascii_other:Boolean=true;
		
		// 半角スペース(0x20)の入力可否を表します。
		private var _type_ascii_space:Boolean=true;
		
		// 全角文字（JIS非漢字、JIS第一水準漢字、JIS第二水準漢字）の入力可否
		private var _type_wide:Boolean=true;


		/**
		 * <p>[概 要]</p>
		 * 数字の入力可否を設定します.
		 *
		 * <p>[詳 細]</p>
		 * 半角数字(0-9)の入力可否を表します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param type_ascii_number trueであれば入力可、falseであれば入力不可
		 * @default true(入力可)
		 */
		public function get type_ascii_number():Boolean {
			return this._type_ascii_number;
		}
		public function set type_ascii_number(type_ascii_number:Boolean):void {
			this._type_ascii_number=type_ascii_number;
		}

		/**
		 * <p>[概 要]</p>
		 * 半角英字の入力可否を設定します.
		 *
		 * <p>[詳 細]</p>
		 * 半角英字(a-z,A-Z)の入力可否を表します。
		 * 
		 * <p>[備 考]</p>
		 * @param type_ascii_alphabet trueであれば入力可、falseであれば入力不可
		 * @default true(入力可)
		 */
		public function get type_ascii_alphabet():Boolean {
			return this._type_ascii_alphabet;
		}
		public function set type_ascii_alphabet(type_ascii_alphabet:Boolean):void {
			this._type_ascii_alphabet=type_ascii_alphabet;
		}

		/**
		 * <p>[概 要]</p>
		 * 半角記号の入力可否を設定します.
		 *
		 * <p>[詳 細]</p>
		 * 半角記号(!"#$%&'()*+,-./:;<=>?@[\]^_`{|}~)の入力可否を表します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param type_ascii_other trueであれば入力可、falseであれば入力不可
		 * @default true(入力可)
		 */
		public function get type_ascii_other():Boolean {
			return this._type_ascii_other;
		}
		public function set type_ascii_other(type_ascii_other:Boolean):void {
			this._type_ascii_other=type_ascii_other;
		}

		/**
		 * <p>[概 要]</p>
		 * 半角スペースの入力可否を設定します.
		 *
		 * <p>[詳 細]</p>
		 * 半角スペース(0x20)の入力可否を表します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param type_ascii_space trueであれば入力可、falseであれば入力不可
		 * @default true(入力可)
		 */
		public function get type_ascii_space():Boolean {
			return this._type_ascii_space;
		}
		public function set type_ascii_space(type_ascii_space:Boolean):void {
			this._type_ascii_space=type_ascii_space;
		}

		/**
		 * <p>[概 要]</p>
		 * 全角文字の入力可否を設定します.
		 *
		 * <p>[詳 細]</p>
		 * 全角文字（JIS非漢字、JIS第一水準漢字、JIS第二水準漢字）の入力可否を表します。
		 *
		 * <p>[備 考]</p>
		 * 
		 * @param type_wide trueであれば入力可、falseであれば入力不可
		 * @default true(入力可)
		 */
		public function get type_wide():Boolean {
			return this._type_wide;
		}
		public function set type_wide(type_wide:Boolean):void {
			this._type_wide=type_wide;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字のチェック結果を返却します.
		 *
		 * <p>[詳 細]</p>
		 * 入力文字をチェックし、指定文字以外の入力文字が存在した場合は、<BR>
		 * 指定されたエラーメッセージを結果配列に設定します。
		 *
		 * <p>[備 考]</p>
		 * 
		 * @param value 入力文字列
		 * @return チェック結果配列
		 */
		override protected function doValidation(value:Object):Array
		{

			// 結果 Array の初期化
			results=[];
			results=super.doValidation(value);

			//チェックエラーの場合、エラー結果を返却
			if (!(checkCharacter(value.toString())))
			{
				results.push(new ValidationResult(true, null, "NaN", ERROR_MESSAGE));
				return results;
			}
			else
			{
				if (results.length == 0)
				{
					results.push(new ValidationResult(false, null, "", ""));
				}
			}

			return results;
		}

		/**
		 * <p>[概 要]</p>
		 * 入力文字を１文字ずつチェックし、チェック結果を返却します.
		 *
		 * <p>[詳 細]</p>
		 * 入力文字を１文字ずつチェックし、少なくとも１文字が指定文字以外の入力文字であれば、<BR>
		 * チェックエラーを返却します。
		 *
		 * <p>[備 考]</p>
		 * @param inputValue 入力文字
		 * @return チェックOKであればtrue,チェックNGであればfalse
		 *
		 */
		private function checkCharacter(inputValue:String):Boolean
		{
			var ret:Boolean=true;
			for (var i:int=0; i < inputValue.length; i++)
			{
				var inputStr:String=inputValue.substr(i, 1);
				if (!checkCharacterArea(inputStr))
				{
					ret=false;
					break;
				}
			}
			return ret;
		}


		/**
		 * <p>[概 要]</p>
		 * 入力文字１文字について、指定した入力条件を満たしているか検証します.
		 *
		 * <p>[詳 細]</p>
		 * 入力文字１文字に対し、本クラス呼び出し元で指定した入力条件を満たしているか、<BR>
		 * チェックを行います。
		 *
		 * <p>[備 考]</p>
		 *
		 * @param inputStrValue 入力文字
		 * @return チェックOKであればtrue,チェックNGであればfalse
		 *
		 */
		private function checkCharacterArea(inputStrValue:String):Boolean
		{

			//半角数字をチェック
			if ((this._type_ascii_number) && (CharTypeUtil.isDigit(inputStrValue) == CharTypeUtil.OK))
			{
				return true;
			}
			//半角英字をチェック
			if ((this._type_ascii_alphabet) && (CharTypeUtil.isAlphabet(inputStrValue) == CharTypeUtil.OK))
			{
				return true;
			}
			//半角記号をチェック
			if ((this._type_ascii_other) && (CharTypeUtil.isOther(inputStrValue) == CharTypeUtil.OK))
			{
				return true;
			}
			//半角スペースをチェック
			if ((this._type_ascii_space) && (CharTypeUtil.isSpace(inputStrValue) == CharTypeUtil.OK))
			{
				return true;
			}
			//全角文字をチェック
			if ((this._type_wide) && (CharTypeUtil.isWide(inputStrValue) == CharTypeUtil.OK))
			{
				return true;
			}

			return false;
		}
	}
}

