/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.flex.producer;

import java.io.Serializable;

import jp.co.fujitsu.reffi.server.messaging.AbstractMessageSender;
import flex.messaging.MessageBroker;
import flex.messaging.MessageClient;
import flex.messaging.messages.AsyncMessage;
import flex.messaging.util.UUIDUtils;

/**
 * <p>[概 要]</p>
 * Flex用基底プロデューサクラス。
 * 
 * <p>[詳 細]</p>
 * Flexでのメッセージ送信基底クラスです。
 * <br />
 * 開発者は本クラスを継承してクライアントへメッセージを送信することが可能となります。
 * 
 * <p>
 * 以下が簡単な実装サンプルとなります。
 * 
 * <pre class="samplecode">
 *public class DemoMessageProducer extends ProxyMessageProducer {
 *	private static final String DESTINATION = "demoDestination";
 *
 *	protected void processSend() {
 *		// クライアントへ送信するデータの生成
 *		ArrayList&lt;String&gt; results = new ArrayList&lt;String&gt;();
 *		for (int i = 0; i < 10; i++) {
 *			results.add("results-" + String.valueOf(i));
 *		}
 *		send(DESTINATION, results);
 *	}
 *}
 * </pre>
 * 
 * <p>
 * 本サンプルでは、あらかじめmessaging-config.xmlに登録されている宛先「demoDestination」
 * に対してメッセージ受信を開始しているクライアントへ生成した文字列の配列を送信しています。
 * <br />
 * {@link ProxyMessageProducer#send(String, Serializable)}を呼び出すことで生成した
 * メッセージの送信を行います。
 * <br />
 * 送信するメッセージは{@link java.io.Serializable}を実装したクラスである必要があります。
 * </p>
 * <p>
 * {@link jp.co.fujitsu.reffi.server.messaging.SenderManager}を利用することで
 * 定期的なメッセージ送信をすることが可能となります。
 * <br />
 * 以下は{@link jp.co.fujitsu.reffi.server.model.AbstractModel}を継承したモデル
 * を実装し、その中でメッセージ送信を1分毎に定期的に実行するサンプルとなります。
 * 
 * <pre class="samplecode">
 *import java.util.ArrayList;
 *import jp.co.fujitsu.reffi.server.flex.producer.ProxyMessageProducer;
 *import jp.co.fujitsu.reffi.server.messaging.SenderManager;
 *import jp.co.fujitsu.reffi.server.model.AbstractModel;
 *
 *public class DemoModel extends AbstractModel {
 *	private static final String PRODUCER_NAME = "DemoProducer";
 *	private static final String DESTINATION = "demoDestination";
 *	private static final int INTERVAL = 60;
 *
 *	protected void mainProcess() throws Exception {
 *		String isCancel = getRequestParameter("isCancel", String.class);
 *		if ("true".equals(isCancel)) {
 *			SenderManager.instance.cancelMessageSender(PRODUCER_NAME);
 *		} else {
 *			SenderManager.instance.addMessageSender(PRODUCER_NAME, new DemoMessageProducer(), INTERVAL * 1000);
 *		}
 *	}
 *
 *	class DemoMessageProducer extends ProxyMessageProducer {
 *		protected void processSend() {
 *			ArrayList&lt;String&gt; results = new ArrayList&lt;String&gt;();
 *			for (int i = 0; i < 10; i++) {
 *				results.add("results-" + String.valueOf(i));
 *			}
 *			send(DESTINATION, results);
 *		}
 *	}
 *}
 * </pre>
 * 
 * サンプルでは、クライアントからのパラメータ「isCancel」に「true」の文字列が設定されている場合、
 * メッセージ送信を停止します。
 * <br />
 * 「true」でない場合はメッセージ送信を登録しますが、同一名で登録するため2つ以上は実行されません。
 * <br />
 * なお、あらかじめmessaging-config.xmlには「demoDestination」が設定されていることが必要です。
 * </p>
 * 
 * <p>[備 考]</p>
 * 本クラスは現在検証中のため、試験的な提供となります。
 * 
 * @author Project Reffi
 */
abstract public class ProxyMessageProducer extends AbstractMessageSender {
	/** デフォルトサービスID */
	private static final String DEFAULT_SERVICE_ID = "message-service";
	/** メッセージブローカー */
	private MessageBroker _broker;
	/** クライアントID　*/
	private String _clientID;
	/** 送信対象クライアント */
	private MessageClient _messageClient;
	/** サービスID */
	private String _serviceID = DEFAULT_SERVICE_ID;
	
	/**
	 * <p>[概 要]</p>
	 * メッセージサービスID取得
	 *
	 * <p>[詳 細]</p>
	 * メッセージサービスのIDを取得します。
	 *
	 * <p>[備 考]</p>
	 * デフォルトでは{@link DEFAULT_SERVICE_ID}が設定されています。
	 *
	 * @return 設定されているメッセージサービスのID
	 */
	public String getServiceID() {
		return this._serviceID;
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージサービスID設定
	 *
	 * <p>[詳 細]</p>
	 * 任意のメッセージサービスIDを設定します。
	 *
	 * <p>[備 考]</p>
	 * デフォルトでは{@link DEFAULT_SERVICE_ID}が設定されているため、
	 * 必要がない場合は設定しないでください。
	 *
	 * @param serviceID 任意のメッセージサービスID
	 */
	public void setServiceID(String serviceID) {
		this._serviceID = serviceID;
	}

	/**
	 * <p>[概 要]</p>
	 * 送信先クライアント情報取得
	 *
	 * <p>[詳 細]</p>
	 * 送信する特定のクライアント情報を取得します。
	 *
	 * <p>[備 考]</p>
	 * クライアント情報が設定されていない場合は接続クライアント全てに送信されます。
	 *
	 * @return 送信先クライアント情報
	 */
	public MessageClient getMessageClient() {
		return this._messageClient;
	}

	/**
	 * <p>[概 要]</p>
	 * 送信先クライアント情報設定
	 *
	 * <p>[詳 細]</p>
	 * 送信する特定のクライアント情報を設定します。
	 *
	 * <p>[備 考]</p>
	 * クライアント情報が設定されていない場合は接続クライアント全てに送信されます。
	 *
	 * @param messageClient 送信先クライアント情報
	 */
	public void setMessageClient(MessageClient messageClient) {
		this._messageClient = messageClient;
	}
	

	/**
	 * <p>[概 要]</p>
	 * メッセージ発行
	 * <p>[詳 細]</p>
	 * メッセージの発行をします。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 宛先
	 * @param object 発行するメッセージ
	 */
	protected final void send(String destination, Serializable object) {
		if (getMessageBroker() == null) {
			return;
		}
		try {
			AsyncMessage msg = new AsyncMessage();
			msg.setDestination(destination);
			msg.setClientId(getClientID());
			msg.setMessageId(UUIDUtils.createUUID());
			msg.setTimestamp(System.currentTimeMillis());
			msg.setBody(object);
			if (getMessageClient() != null) {
				getMessageBroker().routeMessageToMessageClient(msg, getMessageClient());
			} else {
				getMessageBroker().routeMessageToService(msg, null);
			}
		} catch (Exception e) {
			trap(e);
		}
	}

	private MessageBroker getMessageBroker() {
		if (this._broker == null) {
			this._broker = MessageBroker.getMessageBroker(null);
		}
		return this._broker;
	}

	private String getClientID() {
		if (this._clientID == null) {
			this._clientID = UUIDUtils.createUUID();
		}
		return this._clientID;
	}
}
